# -*- coding: utf-8 -*-
import logging
import tarfile

from sandbox import common

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.sup import BASS_YASTROKA_FIXLIST2, BASS_YASTROKA_FIXLIST_TOOL
from sandbox.sdk2.helpers import subprocess as sp

logger = logging.getLogger(__name__)


class BassYaStrokaFixList(sdk2.Task, ReleaseToNannyTask2):

    class Requirements(sdk2.Requirements):
        client_tags = Tag.LINUX_PRECISE
        disk_space = 100 * 1024
        ram = 8 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        tool_binary = sdk2.parameters.Resource(
            'tool binary',
            resource_type=BASS_YASTROKA_FIXLIST_TOOL,
            required=True)
        ytpath = sdk2.parameters.String(
            'Yt path template to fixlist table',
            default='hahn://home/desktop/searchband/soft/2018-02-22/software_dictionary',
            required=True)
        yttoken = sdk2.parameters.String(
            'Yt token name',
            description='vault secret name',
            default='yastroka.robot.yt-token',
            required=True
        )
        release_to_prod = sdk2.parameters.Bool(
            'Release to production',
            default=True,
            required=False
        )
        release_to_priemka = sdk2.parameters.Bool(
            'Release to priemka',
            default=True,
            required=False
        )
        release_to_testing = sdk2.parameters.Bool(
            'Release to testing',
            default=True,
            required=False
        )
        kill_timeout = 3600 * 2

    def on_execute(self):
        tool_resource = sdk2.ResourceData(self.Parameters.tool_binary)

        token = sdk2.Vault.data(self.owner, self.Parameters.yttoken)
        env = {'YT_TOKEN': token}

        cmd = [str(tool_resource.path),
               '--ytpath', self.Parameters.ytpath,
               '--out-dir', '.']

        with sdk2.helpers.ProcessLog(self, logger='make') as pl:
            ret_code = sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret_code == -2:
                logger.info('table not exists, do nothing')
                return
            if ret_code != 0:
                raise common.errors.TaskFailure('make failed')
            with tarfile.open('windows_fixlist.tar.gz', 'w:gz', dereference=True) as tar:
                tar.add('windows_fixlist.data')
                tar.add('windows_fixlist.offsets')

        logger.info('create resource')
        res = sdk2.ResourceData(BASS_YASTROKA_FIXLIST2(self, '', 'windows_fixlist.tar.gz', ttl='inf'))
        res.ready()

        release_to = []
        if self.Parameters.release_to_testing:
            release_to.append('testing')
        if self.Parameters.release_to_priemka:
            release_to.append('prestable')
        if self.Parameters.release_to_prod:
            release_to.append('stable')

        logger.info('release to: {}'.format(release_to))

        for release_type in release_to:
            release_params = {
                'release_comments': 'automatic release',
                'release_subject': '',
                'release_status': release_type,
                'email_notifications': {
                    'cc': [],
                    'to': []
                },
                'releaser': self.owner
            }
            ReleaseToNannyTask2.on_release(self, release_params)
            sdk2.Task.on_release(self, release_params)
