# -*- coding: utf-8 -*-
import json

import sandbox.sdk2 as sdk2

import sandbox.common.types.resource as ctr

from sandbox.projects.common import binary_task
import sandbox.projects.sandbox.resources as sb_resources

from sandbox.projects.surfwax.common.podman import PodmanWrapper, get_selenoid_image, get_video_recorder_image, load_docker_registry_creds
from sandbox.projects.surfwax.resource_types import SurfwaxLayersCache


class SurfwaxBuildLayersCache(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Builds squashfs image with podman storage, to use it in additionalimagestores
    """

    class Requirements(sdk2.Requirements):
        # Task is privileged to run podman from root and use overlay storage driver
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value="2239708134",
            resource_type=sb_resources.LXC_CONTAINER,
            platform="linux_ubuntu_18.04_bionic",
            required=True,
        )
        binary_task_params = binary_task.binary_release_parameters(stable=True)

        cache_type = sdk2.parameters.RadioGroup('Layers cache type', choices=(
            # TODO extract selenoid to separate type
            ('Browsers and selenoid', 'browsers'),
            ('Surfwax agent', 'agent'),
        ), default='browsers')

        with cache_type.value['browsers']:
            browsers = sdk2.parameters.String('Browsers json', default='', multiline=True)
        with cache_type.value['agent']:
            surfwax_agent_version = sdk2.parameters.String('SurfWax agent version')

    @property
    def binary_executor_query(self):
        return {
            "owner": "SURFWAX_STAGING_BROWSERS",
            "attrs": {"project": "surfwax", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY],
        }

    # Default data root is in /var, and privileged task have overlayfs there
    # This does not allow podman to use overlay2 storage driver
    # We are moving storage to task dir, to ensure task isolation, and to use overlay2 storage driver
    # TODO maybe move this setting from CLI to config file
    def podman_data_dir(self):
        return str(self.path("podman-data"))

    def get_images(self):
        images = []

        cache_type = self.Parameters.cache_type
        if cache_type == "browsers":
            browsers = json.loads(self.Parameters.browsers)

            # TODO split selenoid images to separate cache type
            images.append(get_selenoid_image())
            images.append(get_video_recorder_image())

            for browser in browsers:
                for version in browsers[browser]["versions"]:
                    if "image" in browsers[browser]["versions"][version]:
                        image = browsers[browser]["versions"][version]["image"]
                        images.append(image)
        elif cache_type == "agent":
            images.append("registry.yandex.net/search-interfaces/surfwax-agent:{}".format(self.Parameters.surfwax_agent_version))
        else:
            raise Exception("Unknown cache_type value: {}".format(cache_type))

        return images

    def on_execute(self):
        super(SurfwaxBuildLayersCache, self).on_execute()

        podman_wrapper = PodmanWrapper(self.podman_data_dir())

        (login, password) = load_docker_registry_creds()
        podman_wrapper.login(login, password)

        images = self.get_images()

        results = map(
            lambda i: podman_wrapper.pull_async(i),
            images,
        )
        for result in results:
            result.wait()

        squashfs_image_path = str(self.path("podman_layers_cache.squashfs"))

        with sdk2.helpers.ProcessLog(task=self, logger="mksquashfs") as log:
            sdk2.helpers.subprocess.check_call(
                [
                    "mksquashfs",
                    self.podman_data_dir(),
                    squashfs_image_path,
                ],
                stdout=log.stdout,
                stderr=log.stderr,
            )

        resource = SurfwaxLayersCache(
            self,
            "podman layers cache for Surfwax",
            squashfs_image_path,
            cache_type=self.Parameters.cache_type,
        )
        data = sdk2.ResourceData(resource)
        data.ready()
