# -*- coding: utf-8 -*-

import traceback
import logging

from multiprocessing import cpu_count
from multiprocessing.pool import ThreadPool


class ThreadError(Exception):
    def __init__(self, cause):
        super(ThreadError, self).__init__('{}\n\nThread {}'.format(cause, traceback.format_exc()))
        self.cause = cause


def with_thread_traceback_serialized(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            raise ThreadError(e)

    return wrapper


def parallel(func, tasks, threads_count=None):
    """
    :param func: функция, которая будет выполнена в каждом треде для каждой таски
    :type func: func
    :param tasks: таски, которые будут переданы в функцию :func:
    :type tasks: list
    :param threads_count: число тредов, которым будут выдаваться задания. Если `None`, то равно количеству ядер
    :type threads_count: int
    :rtype: list
    """
    if len(tasks) == 0:
        return []

    if threads_count is None:
        threads_count = cpu_count()

    threads_count = min(threads_count, len(tasks))

    logging.debug('create {threads_count} threads for {tasks_count} tasks'.format(
        threads_count=threads_count,
        tasks_count=len(tasks),
    ))
    decorated_func = with_thread_traceback_serialized(func)

    threads_pool = ThreadPool(processes=threads_count)
    try:
        return threads_pool.map(decorated_func, tasks)
    finally:
        threads_pool.close()
        threads_pool.join()
