# -*- coding: utf-8 -*-

import logging
import os
import time

import sandbox.sdk2 as sdk2


def get_selenoid_image():
    # https://github.com/gemini-testing/selenoid/commits/patches
    # See details in https://st.yandex-team.ru/FEI-24459
    selenoid_image = "registry.yandex.net/search-interfaces/selenoid:1.10.7-s3-metadata-surfwax-patches"
    return selenoid_image


def get_video_recorder_image():
    video_recorder_image = "registry.yandex.net/selenium/video-recorder:latest-release"
    return video_recorder_image


def load_docker_registry_creds():
    docker_registry_secret = sdk2.yav.Secret("sec-01f6ek7b78fyq7v5zqgbsdmtb3")
    docker_registry_oauth_token = docker_registry_secret.data()["oauth"]
    return "robot-surfwax", docker_registry_oauth_token


class ExecAsyncResult:
    def __init__(self, process, log, start):
        self.process = process
        self.log = log
        self.start = start

    def wait(self):
        logging.info("Waiting for %s", self.process.args)
        retcode = self.process.wait()
        self.log.close()
        finish = time.time()
        logging.debug("Command {} took {} seconds".format(self.process.args, finish - self.start))
        if retcode != 0:
            raise sdk2.helpers.subprocess.CalledProcessError(retcode, self.process.args)


def exec_async(*args, **kwargs):
    start = time.time()
    log_name = kwargs.pop("log", None)

    log = sdk2.helpers.ProcessLog(logger=log_name)
    process = sdk2.helpers.subprocess.Popen(
        *args,
        stdout=log.stdout,
        stderr=log.stderr,
        **kwargs
    )
    return ExecAsyncResult(process, log, start)


class PodmanWrapper:
    def __init__(self, data_dir):
        self.data_dir = data_dir

    def pull_async(self, image):
        return self._exec_podman_async([
            "pull",
            image,
        ])

    def create_network(self, name, dns=True, ipv6=False, subnet=None):
        args = [
            "network",
            "create",
        ]

        if not dns:
            args.append("--disable-dns")

        if ipv6:
            args.append("--ipv6")

        if subnet is not None:
            args.append("--subnet")
            args.append(subnet)

        args.append(name)

        self._exec_podman(args)

    def _exec_podman(self, command_tail):
        command = [
            "podman",
            "--root",
            self.data_dir,
        ] + command_tail

        with sdk2.helpers.ProcessLog(logger="podman") as log:
            sdk2.helpers.subprocess.check_call(
                command,
                stdout=log.stdout,
                stderr=log.stderr,
            )

    def _exec_podman_async(self, command_tail):
        command = [
            "podman",
            "--root",
            self.data_dir,
        ] + command_tail

        return exec_async(
            command,
            log="podman"
        )

    def login(self, login, password):
        podman_login_env = os.environ.copy()
        podman_login_env["PODMAN_OAUTH_TOKEN"] = password

        logging.info("Running podman login")
        # Using bash to put token in CLI argument without leaking to logs
        # --password-stdin refuses to work without TTY
        # TODO TTY issue was hit with docker, recheck this with podman
        podman_login_command = [
            "bash",
            "-c",
            "podman --root {} login --username {} --password $PODMAN_OAUTH_TOKEN registry.yandex.net".format(self.data_dir, login),
        ]
        with sdk2.helpers.ProcessLog(logger="podman_login") as log:
            sdk2.helpers.subprocess.check_call(
                podman_login_command,
                env=podman_login_env,
                stdout=log.stdout,
                stderr=log.stderr,
            )
