# -*- coding: utf-8 -*-

import requests

PRODUCTION_ENVIRONMENT = 'production'
STAGING_ENVIRONMENT = 'staging'

PRODUCTION_SURFWAX_URL = "https://sw.yandex-team.ru"
STAGING_SURFWAX_URL = "https://sw-staging.yandex-team.ru"

DEFAULT_API_VERSION = 0

MAX_RETRY_COUNT = 3


class Api:
    def __init__(self, environ='production', api_version=DEFAULT_API_VERSION):
        self.__base_url = PRODUCTION_SURFWAX_URL if environ == 'production' else STAGING_SURFWAX_URL
        self.__api_version = api_version
        self.__session = requests.Session()

        self.__session.mount(self.__base_url, requests.adapters.HTTPAdapter(max_retries=MAX_RETRY_COUNT))

    def __build_url(self, route_url):
        return '{base}/api/v{api_version}/{route}'.format(
            base=self.__base_url,
            api_version=self.__api_version,
            route=route_url
        )

    def __send_request(self, route_url, method='get'):
        try:
            url = self.__build_url(route_url)
            return getattr(self.__session, method)(url).json()
        except requests.exceptions.RequestException as error:
            raise ApiException("Error when executing the query \"{method}\" to url \"{url}\": {error}".format(
                method=method,
                url=url,
                error=error
            ))

    def get_quotas(self):
        return self.__send_request('quotas')

    def get_quota_caches(self, quota_name=''):
        return self.__send_request('quotas/{}/caches'.format(quota_name))

    def __del__(self):
        self.__session.close()


class ApiException(Exception):
    pass
