# -*- coding: utf-8 -*-

import shutil
import os
import json
import logging
import re
import signal
import subprocess

import sandbox.sdk2 as sdk2

MAC_OS_BROWSERS = ['safari']


def get_data_from_all_pages(get_method, limit=100, **kwargs):
    should_read = True
    offset = 0
    items = []

    while should_read:
        response = get_method(offset=offset, limit=limit, **kwargs)
        total = response.get('total', 0)
        items += response.get('items', [])
        offset += limit

        should_read = total > offset

    return items


def shutdown_process(process, graceful_timeout):
    if process.returncode is None:
        # SIGINT is used because bash does not always forward SIGTERM
        # Both Selenoid and sugwax-agent should shut down on SIGINT
        # https://a.yandex-team.ru/arc_vcs/frontend/projects/infratest/packages/surfwax-agent/src/index.ts?rev=eceb1581994e6afefd16862581b944507f9aa0be#L190
        # https://github.com/aerokube/selenoid/blob/e89b7ab6c42b602bc897779b92512d35c59fe807/main.go#L362
        logging.info("Going to stop process {} with SIGINT".format(process.pid))
        process.send_signal(signal.SIGINT)
    try:
        logging.info("Waiting process {} for termination".format(process.pid))
        process.communicate(timeout=graceful_timeout)
        logging.info("Process {} has been gracefully shut down".format(process.pid))
    except subprocess.TimeoutExpired:
        logging.info("Process {} has not been stopped in {} seconds, going to kill it.".format(process.pid, graceful_timeout))
        process.kill()


def has_mac_os_bro(browsers):
    browsers = json.loads(browsers)
    res = False

    for browser in browsers:
        for browser_matchers in MAC_OS_BROWSERS:
            if re.match(browser, browser_matchers):
                res = True

    return res


def install_res(task, resource_id, res_dir):
    resource = sdk2.Resource.find(id=resource_id).first()
    data = sdk2.ResourceData(resource)
    res_path = data.path
    res_dir_path = str(task.path(res_dir))
    os.mkdir(res_dir_path)
    shutil.copy(str(res_path), res_dir_path)

    return res_dir_path
