# -*- coding: utf-8 -*-
import json
import logging
import math
import random

from functools import partial

import sandbox.sdk2 as sdk2

from sandbox.common.errors import ResourceNotFound, TaskFailure, TaskNotEnqueued
from sandbox.common.itertools import chain
from sandbox.common.types import task as ctt, resource as ctr
from sandbox.common.utils import singleton_property

from sandbox.projects.common import binary_task
from sandbox.projects.common.utils import flatten

from sandbox.projects.surfwax.common.surfwax_api import Api, ApiException, PRODUCTION_ENVIRONMENT, STAGING_ENVIRONMENT
from sandbox.projects.surfwax.common.flow import parallel
from sandbox.projects.surfwax.common.utils import get_data_from_all_pages
from sandbox.projects.surfwax.resource_types import SurfwaxLayersCache
from sandbox.projects.surfwax.sync_resources_on_agent import SurfwaxSyncResourcesOnAgent

CUSTOM_SOURCE_FOR_RESOURCES = "custom"
DATABASE_SOURCE_FOR_RESOURCES = "db"

SUITABLE_AGENT_TAGS = "GENERIC&LINUX&~MULTISLOT&CORES32"
AGENTS_PER_PAGE_LIMIT = 1000

FAILURE_STATUSES = tuple(chain(ctt.Status.FAILURE, ctt.Status.Group.BREAK))


class SurfwaxSyncResources(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Synchronizes browsers cache resources on agents
    """

    class Parameters(sdk2.Task.Parameters):
        priority = ctt.Priority(ctt.Priority.Class.BACKGROUND, ctt.Priority.Subclass.LOW)
        binary_task_params = binary_task.binary_release_parameters(stable=True)

        resources_source = sdk2.parameters.RadioGroup("Source for getting resources", choices=(
            ("Custom", CUSTOM_SOURCE_FOR_RESOURCES),
            ("From DB", DATABASE_SOURCE_FOR_RESOURCES),
        ), default=DATABASE_SOURCE_FOR_RESOURCES)

        with resources_source.value[CUSTOM_SOURCE_FOR_RESOURCES]:
            resources = sdk2.parameters.Resource(
                "Resources",
                resource_type=SurfwaxLayersCache,
                multiple=True,
                required=True,
            )

        with resources_source.value[DATABASE_SOURCE_FOR_RESOURCES]:
            source_environ = sdk2.parameters.RadioGroup("Source environment", choices=(
                ("production", PRODUCTION_ENVIRONMENT),
                ("staging", STAGING_ENVIRONMENT),
            ), default=PRODUCTION_ENVIRONMENT)
            quotas = sdk2.parameters.List(
                "Quota names",
                description="If the list is empty, all quotas will be used",
                default=[]
            )

        coverage = sdk2.parameters.Float("Agents coverage", default=1, required=True)

    class Context(sdk2.Context):
        subtasks = []

    @singleton_property
    def __api(self):
        return Api(self.Parameters.source_environ)

    @property
    def binary_executor_query(self):
        return {
            "owner": "SURFWAX_STAGING_BROWSERS",
            "attrs": {"project": "surfwax", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY],
        }

    def on_save(self):
        super(SurfwaxSyncResources, self).on_save()

        self.__validate_parameters()

    def __validate_parameters(self):
        if self.Parameters.coverage <= 0 or self.Parameters.coverage > 1:
            raise TaskFailure("Parameter \"{parameter_name}\" must be float value in the range from 0 to 1.".format(
                parameter_name=SurfwaxSyncResources.Parameters.coverage.name
            ))

    def on_execute(self):
        super(SurfwaxSyncResources, self).on_execute()

        with self.memoize_stage.run_sync():
            resources = self.__get_resources()
            sandbox_agents = self.__get_sandbox_agents()

            self.Context.subtasks = map(int, filter(bool, parallel(partial(self.__run_subtask, resources), sandbox_agents)))

            raise sdk2.WaitTask(
                tasks=self.Context.subtasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        with self.memoize_stage.handle_results():
            self.__handle_subtasks()

    def __get_resources(self):
        if self.Parameters.resources_source == CUSTOM_SOURCE_FOR_RESOURCES:
            logging.debug("Getting resource list from parameter")

            return self.Parameters.resources

        logging.debug("Getting resource list from db")

        return self.__get_resources_from_db()

    def __get_resources_from_db(self):
        quotas = self.__get_quotas()

        try:
            caches = flatten(parallel(self.__api.get_quota_caches, quotas))
        except ApiException as error:
            raise TaskFailure('Failed to get a list of resources from the database: {error}'.format(
                error=error
            ))

        return filter(bool, map(lambda cache: self.__find_resource_by_id(cache["cacheId"]), caches))

    def __find_resource_by_id(self, resource_id):
        try:
            resource = sdk2.Resource[resource_id]

            return resource if resource.state == ctr.State.READY else None
        except ResourceNotFound:
            return None

    def __get_quotas(self):
        if self.Parameters.quotas:
            return self.Parameters.quotas
        else:
            return self.__get_quotas_from_db()

    def __get_quotas_from_db(self):
        try:
            return map(lambda quota: quota["name"], self.__api.get_quotas())
        except ApiException as error:
            raise TaskFailure('Failed to get quotas from the database: {error}'.format(
                error=error
            ))

    def __get_sandbox_agents(self):
        agents = self.__get_suitable_sandbox_agents()

        logging.debug("All suitable agents for caching: {agents}".format(
            agents=json.dumps(agents)
        ))

        return self.__sample_sandbox_agents_by_coverage(agents)

    def __get_suitable_sandbox_agents(self):
        agents_filtered_by_tags = get_data_from_all_pages(self.server.client.read, limit=AGENTS_PER_PAGE_LIMIT, tags=SUITABLE_AGENT_TAGS, alive=True)

        return map(lambda agent: agent["id"], agents_filtered_by_tags)

    def __sample_sandbox_agents_by_coverage(self, agents):
        needed_agent_count = math.ceil(len(agents) * self.Parameters.coverage)

        logging.debug("Using for caching {coverage}% ({used}/{all}) of all suitable agents".format(
            coverage=self.Parameters.coverage * 100,
            used=needed_agent_count,
            all=len(agents)
        ))

        return random.sample(agents, int(needed_agent_count))

    def __run_subtask(self, resources, sandbox_agent_id):
        subtask = SurfwaxSyncResourcesOnAgent(
            self,
            description="Task for caching resources {resource_ids} on the {agent_name} agent".format(
                resource_ids=", ".join(map(lambda resource: str(resource.id), resources)),
                agent_name=sandbox_agent_id
            ),
            priority=self.Parameters.priority,
            resources=resources,

            __requirements__={
                "host": sandbox_agent_id
            }
        )

        try:
            return subtask.enqueue()
        except TaskNotEnqueued as error:
            logging.error("Failed to enqueue subtask {task_id} for {agent_name} agent. Reason: {error}".format(
                task_id=subtask.id,
                agent_name=sandbox_agent_id,
                error=error
            ))

    def __handle_subtasks(self):
        subtasks = map(lambda task_id: sdk2.Task[task_id], self.Context.subtasks)
        failed_subtasks = filter(lambda task: task.status in FAILURE_STATUSES, subtasks)

        if failed_subtasks:
            raise TaskFailure("{failed_count} of {total_count} subtasks are failed.".format(
                failed_count=len(failed_subtasks),
                total_count=len(subtasks)
            ))
