import logging
import math
import os
import subprocess
import time
import psutil
import sandbox.projects.common.network as network
import sandbox.sdk2 as sdk2

from sandbox.common.errors import TaskFailure
try:
    from multiprocessing import cpu_count
except ImportError:
    from os import cpu_count

from sandbox.common.utils import singleton_property


SAFE_EXECUTION_TIMEOUT_MARGIN_SEC = 6 * 60

# 1 minute less than gap between agent and task lifetime to allow for all other bookkeeping
SURFWAX_SESSION_SHUTDOWN_TIMEOUT_SEC = SURFWAX_GRACEFUL_SHUTDOWN_TIMEOUT_SEC = SAFE_EXECUTION_TIMEOUT_MARGIN_SEC - 1 * 60

SELENOID_GRACEFUL_SHUTDOWN_TIMEOUT_SEC = 15

PRODUCTION_TAG = "PRODUCTION"

NODE_VERSION = "12.13.0"

INFINITY = float("inf")


class CommonTaskManager():
    def __init__(self, task):
        self._task = task

    def get_task_processes(self):
        raise "Not implemented"

    def _run_selenoid(self):
        selenoid_run_cmd = self._prepare_selenoid()

        selenoid_run_cmd.extend([
            "-limit", str(self._sessions_per_agent),
            "--listen", ":{}".format(self._task.Parameters.selenoid_port),
            "-conf", self.selenoid_config_path,
            "-log-output-dir", self.selenoid_log_dir,
            "-session-attempt-timeout", str(self._task.Parameters.session_attempt_timeout),
            "-service-startup-timeout", str(self._task.Parameters.service_startup_timeout),
            # We are managing queues in Surfwax proxy, so it's simpler for selenoid to just respond with 429
            "-disable-queue"
        ])

        if self.selenoid_video_dir:
            selenoid_run_cmd.extend(["-video-output-dir", self.selenoid_video_dir])

        s3 = self.__get_s3()
        if s3:
            logging.info("Using S3 to store sessions logs")
            selenoid_run_cmd.extend([
                "-s3-endpoint", s3["endpoint"],
                "-s3-region", s3["region"],
                "-s3-bucket-name", s3["bucket"],
                "-s3-access-key", s3["access_key"],
                "-s3-secret-key", s3["secret_key"],
                "-s3-exclude-files", "*.json",
            ])

        log_file = open(str(self._task.log_path("podman_selenoid_log")), "w")
        proc = self._exec_proc(selenoid_run_cmd, log_file)
        log_file.close()
        return (proc, SELENOID_GRACEFUL_SHUTDOWN_TIMEOUT_SEC)

    @singleton_property
    def _sessions_per_agent(self):
        sessions_per_agent = self._task.Parameters.selenoid_sessions_per_agent_limit

        if self._task.Parameters.selenoid_cpu_cores_per_session > 0:
            sessions_per_agent = min(sessions_per_agent or INFINITY, cpu_count() / self._task.Parameters.selenoid_cpu_cores_per_session)

        if self._task.Parameters.ramdrive_size_gb > 0 and self._task.Parameters.selenoid_ramdrive_gb_per_session > 0:
            sessions_per_agent = min(sessions_per_agent or INFINITY, self._task.Parameters.ramdrive_size_gb / self._task.Parameters.selenoid_ramdrive_gb_per_session)

        if self._task.Parameters.selenoid_ram_gb_per_session > 0:
            free_ram_gb = (psutil.virtual_memory().total >> 30) - self._task.Parameters.ramdrive_size_gb
            sessions_per_agent = min(sessions_per_agent or INFINITY, free_ram_gb / self._task.Parameters.selenoid_ram_gb_per_session)

        return int(math.floor(sessions_per_agent))

    def _run_surfwax_agent(self):
        coordinator_args = []
        if not self._task.Parameters.surfwax_quota \
           or not self._task.Parameters.surfwax_quota_agent_secret \
           or not self._task.Parameters.surfwax_coordinator_address_type:
            raise TaskFailure("SurfWax agent is not configured")

        if self._task.Parameters.surfwax_coordinator_address_type == "endpoint":
            if not self._task.Parameters.surfwax_coordinator_endpoint:
                raise TaskFailure("SurfWax agent is not configured")
            coordinator_args = ["--coordinator-endpoint", self._task.Parameters.surfwax_coordinator_endpoint]
        elif self._task.Parameters.surfwax_coordinator_address_type == "deploy":
            if not self._task.Parameters.surfwax_coordinator_deploy_stage \
               or not self._task.Parameters.surfwax_coordinator_deploy_unit \
               or not self._task.Parameters.surfwax_coordinator_deploy_dc:
                raise TaskFailure("SurfWax agent is not configured")
            coordinator_args = [
                "--coordinator-deploy-stage",
                self._task.Parameters.surfwax_coordinator_deploy_stage,
                "--coordinator-deploy-unit",
                self._task.Parameters.surfwax_coordinator_deploy_unit,
                "--coordinator-deploy-dc",
            ] + self._task.Parameters.surfwax_coordinator_deploy_dc
        else:
            raise TaskFailure("Unknown surfwax agent address type: {}".format(self._task.Parameters.surfwax_coordinator_address_type))

        command_to_run = self._get_surfwax_agent_cmd()

        addr = network.get_my_ipv6()
        # TODO: Get rid of static ports.
        # TODO: Do not stop selenoid on task timeout?
        command_to_run += [
            "--quota", self._task.Parameters.surfwax_quota,
            "--browser-group", self._task.Parameters.surfwax_browser_group,
            "--secret", self._task.Parameters.surfwax_quota_agent_secret.data()[self._task.Parameters.surfwax_quota_agent_secret.default_key],
            "--listen-port", "14445",
            # This line requires host network
            "--listen-host", addr,
            "--selenoid-endpoint", "[{}]:{}".format(addr, self._task.Parameters.selenoid_port),
            "--lifetime", str(self.__safe_execution_timeout()),
            "--session-shutdown-timeout", str(SURFWAX_SESSION_SHUTDOWN_TIMEOUT_SEC),
            "--selenoid-proxy-timeout", self._task.Parameters.selenoid_proxy_timeout,
        ]

        command_to_run += coordinator_args

        env = os.environ.copy()
        env["DEBUG"] = "surfwax-agent,surfwax-agent:*,si.ci.requests"
        env["NODE_VERSION"] = NODE_VERSION
        log_file = open(str(self._task.log_path("agent_log")), "w")
        proc = self._exec_proc(command_to_run, log_file, env)
        log_file.close()
        return (proc, SURFWAX_GRACEFUL_SHUTDOWN_TIMEOUT_SEC + 5)

    def _install_surfwax_agent(self):
        raise "Not implemented"

    def _agent_npm_string(self):
        return "@yandex-int/surfwax-agent@{}".format(self._task.Parameters.surfwax_agent_version)

    def __safe_execution_timeout(self):
        running_time = int(time.time() - self._task.Context.execution_started_at)

        return self._task.Parameters.kill_timeout - running_time - SAFE_EXECUTION_TIMEOUT_MARGIN_SEC

    def __get_s3(self):
        try:
            secret = sdk2.yav.Secret("sec-01f4bptwjwnqhv41v0r7nwq5x4")

            return {
                "endpoint": "https://s3.mds.yandex.net",
                "region": "us-west-1",
                "bucket": "surfwax-sessions",
                "access_key": secret.data()["access-key-id"],
                "secret_key": secret.data()["access-secret-key"],
            }
        except Exception as e:
            logging.info("Cannot get S3 credentials: {}".format(e.message))
            return None

    def _exec_proc(self, cmd, log_file, env=None):
        return subprocess.Popen(cmd, stdout=log_file, stderr=log_file, env=env)
