import grp
import json
import logging
import os
import pwd
import subprocess
import sandbox.sdk2 as sdk2

from sandbox.common.errors import TaskFailure
from operator import itemgetter
from sandbox.projects.surfwax.task_manager.common import CommonTaskManager
from sandbox.projects.surfwax.common.podman import PodmanWrapper, exec_async, get_selenoid_image, get_video_recorder_image, load_docker_registry_creds
from sandbox.projects.sandbox_ci.decorators.retriable import retriable
from sandbox.projects.surfwax.common.utils import shutdown_process

PODMAN_SERVICE_SHUTDOWN_TIMEOUT_SEC = 5

PODMAN_POLL_TIMEOUT_SEC = 5
PODMAN_POLL_INTERVAL_SEC = 0.1
PODMAN_POLL_RETRY_COUNT = PODMAN_POLL_TIMEOUT_SEC / PODMAN_POLL_INTERVAL_SEC

SOURCE_NVM = "source /home/sandbox/.nvm/nvm.sh"
NVM_EXEC = "/home/sandbox/.nvm/nvm-exec"

NODE_VERSION = "12.13.0"

INFINITY = float("inf")


class LinuxTaskManager(CommonTaskManager):
    def get_task_processes(self):
        selenoid_process = self._run_selenoid()
        surfwax_agent_process = self._run_surfwax_agent()

        return [
            surfwax_agent_process,
            selenoid_process,
            (self.podman_service_process, PODMAN_SERVICE_SHUTDOWN_TIMEOUT_SEC)
        ]

    def _prepare_selenoid(self):
        self.__mount_cache()
        self.__prepare_agent_dirs()

        podman_wrapper = self.__create_podman_network()

        selenoid_image = get_selenoid_image()
        video_recorder_image = get_video_recorder_image()

        self.__pull_images(podman_wrapper, selenoid_image, video_recorder_image)

        self.podman_service_process = self.__run_podman_service()

        try:
            self.__poll_podman_socket()
        except:
            shutdown_process(self.podman_service_process, PODMAN_SERVICE_SHUTDOWN_TIMEOUT_SEC)
            raise

        conf_path, logs_path, video_path = itemgetter('conf_path', 'logs_path', 'video_path')(self.__init_selenoid_dirs())
        browsers_json_path = self.__init_selenoid_conf(conf_path)

        # Selenoid uses /.dockerenv to check if it is running inside container
        # Podman uses /run/.containerenv for the same thing
        # So we emulate it just to trick selenoid
        # See https://github.com/aerokube/selenoid/blob/e6523b6110ee79cc6dbb86bd07a383dbff73dad3/main.go#L126-L130
        dockerenv_path = str(self._task.path("selenoid_dockerenv"))
        with open(dockerenv_path, "w") as f:
            f.write("")

        return [
            "podman",
            "--root",
            self.__podman_data_dir(),
            "run",
            "--rm",
            "--name", "selenoid",
            "-e", "TZ=Europe/Moscow",
            "-e", "OVERRIDE_VIDEO_OUTPUT_DIR={}".format(video_path),
            "-e", "GGR_HOST={}".format(self._task.Parameters.ggr_host),
            "-p", "{port}:{port}".format(port=self._task.Parameters.selenoid_port),
            "--log-opt", "max-size=500m",
            "--log-opt", "max-file=7",
            "-v", "{}:/etc/selenoid/browsers.json:ro".format(browsers_json_path),
            "-v", "{}:/opt/selenoid/video/".format(video_path),
            "-v", "{}:/opt/selenoid/logs/".format(logs_path),
            "-v", "{}:/var/run/docker.sock".format(self.__podman_socket_path()),
            "-v", "{}:/.dockerenv".format(dockerenv_path),
            "--network", "podman6",
            selenoid_image,
            "-video-recorder-image", video_recorder_image,
            "--container-network", "podman6"
        ]

    def __mount_cache(self):
        # This import is messing with non-binary tasks
        import toml

        with open("/etc/containers/storage.conf") as f:
            storage_conf = toml.load(f)
        logging.info("Loaded storage conf {}".format(storage_conf))

        cache_resources = self._task.Parameters.surfwax_layers_cache_resources + self._task.Parameters.surfwax_browsers_cache_resources
        for cache_resource in cache_resources:
            cache_data = sdk2.ResourceData(cache_resource)
            cache_mount_point = cache_data.mount()
            logging.info("Layers cache resource {} downloaded to {} and mounted to {}".format(cache_resource.id, cache_data.path, cache_mount_point))
            storage_conf['storage']['options']['additionalimagestores'].append(str(cache_mount_point))

        logging.info("storage conf after patching {}".format(storage_conf))
        with open("/etc/containers/storage.conf", "w") as f:
            toml.dump(storage_conf, f)

    def __prepare_agent_dirs(self):
        # Agent will be running from sandbox user, so we need to prepare install dir with proper owner
        os.makedirs(self.__agent_install_path())
        os.chown(
            self.__agent_install_path(),
            pwd.getpwnam("sandbox").pw_uid,
            grp.getgrnam("sandbox").gr_gid,
        )

    def __create_podman_network(self):
        podman_wrapper = PodmanWrapper(self.__podman_data_dir())

        # Podman does not have IPv6 network by default, so we create one
        podman_wrapper.create_network(
            "podman6",
            dns=False,
            ipv6=True,
            subnet="fd00::/8",
        )

        (login, password) = load_docker_registry_creds()
        podman_wrapper.login(login, password)

        return podman_wrapper

    def __pull_images(self, podman_wrapper, selenoid_image, video_recorder_image):
        browsers = json.loads(self._task.Parameters.browsers)

        pull_images = [
            selenoid_image,
            video_recorder_image,
            self.__agent_image(),
        ]
        for browser in browsers:
            for version in browsers[browser]["versions"]:
                if "image" in browsers[browser]["versions"][version]:
                    image = browsers[browser]["versions"][version]["image"]
                    pull_images.append(image)
        pull_results = map(
            lambda image: podman_wrapper.pull_async(image),
            pull_images
        )

        prepare_results = [] + pull_results
        if self._task.Parameters.surfwax_agent_run_method == "npm":
            prepare_results.append(self._install_surfwax_agent())

        for result in prepare_results:
            result.wait()

    # Default data root is in /var, and privileged task have overlayfs there
    # This does not allow podman to use overlay2 storage driver
    # We are moving storage to task dir, to ensure task isolation, and to use overlay2 storage driver
    # TODO maybe move this setting from CLI to config file
    def __podman_data_dir(self):
        if self._task.Parameters.ramdrive_size_gb > 0:
            return str(self._task.ramdrive.path / "podman-data")
        else:
            return str(self._task.path("podman-data"))

    def __agent_install_path(self):
        return str(self._task.path("surfwax_agent"))

    def __agent_exe_path(self):
        return os.path.join(
            self.__agent_install_path(),
            "node_modules",
            ".bin",
            "surfwax-agent",
        )

    @retriable(retries_count=PODMAN_POLL_RETRY_COUNT, delay=PODMAN_POLL_INTERVAL_SEC)
    def __check_podman_socket(self):
        with sdk2.helpers.ProcessLog(task=self._task, logger="podman_socket_poll") as log:
            sdk2.helpers.subprocess.check_call(
                [
                    "podman-remote",
                    "--url",
                    self.__podman_socket_url(),
                    "info",
                ],
                stdout=log.stdout,
                stderr=log.stderr,
            )

    def __poll_podman_socket(self):
        try:
            self.__check_podman_socket()
        except:
            logging.info("Podman socket {} not ready after {} seconds".format(self.__podman_socket_url(), PODMAN_POLL_TIMEOUT_SEC))
            raise

    def _install_surfwax_agent(self):
        env = os.environ.copy()
        env["NPM_CONFIG_REGISTRY"] = "http://npm.yandex-team.ru"

        return exec_async(
            [
                "sudo",
                "-i",
                "-u",
                "sandbox",
                "bash",
                "-c",
                "(curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.37.2/install.sh | bash) && {} && NVM_FLAVOR=node nvm install {} && nvm use {} && \
                    cd {} && NPM_CONFIG_REGISTRY=\"http://npm.yandex-team.ru\" npm i {}".format(
                    SOURCE_NVM,
                    NODE_VERSION,
                    NODE_VERSION,
                    self.__agent_install_path(),
                    self._run_surfwax_agent(),
                )
            ],
            env=env,
            log="surfwax_agent_install",
        )

    def __run_podman_service(self):
        command = [
            "podman",
            "--root",
            self.__podman_data_dir(),
            "system",
            "service",
            "-t",
            "0",
            "--log-level=info",
            self.__podman_socket_url()
        ]

        log_file = open(str(self._task.log_path("podman_service_log")), "w")
        proc = subprocess.Popen(command, stdout=log_file, stderr=log_file)
        log_file.close()
        return proc

    def __podman_socket_path(self):
        return str(self._task.path("podman.sock"))

    def __podman_socket_url(self):
        return "unix://" + self.__podman_socket_path()

    def __init_selenoid_conf(self, conf_path):
        browsers_json_path = os.path.join(conf_path, "browsers.json")
        with open(browsers_json_path, "w") as f:
            f.write(self._task.Parameters.browsers)

        return browsers_json_path

    def __init_selenoid_dirs(self):
        conf_path = str(self._task.path("selenoid_config"))
        logs_path = str(self._task.log_path("logs"))
        video_path = str(self._task.log_path("video"))
        for d in [conf_path, logs_path, video_path]:
            os.makedirs(d)

        return {'conf_path': conf_path, 'logs_path': logs_path, 'video_path': video_path}

    def _get_surfwax_agent_cmd(self):
        if self._task.Parameters.surfwax_agent_run_method == "npm":
            # setuidgid from daemontools is used to drop privileges to sandbox user for surfwax-agent
            cmd = [
                "setuidgid",
                "sandbox",
                NVM_EXEC,
                self.__agent_exe_path(),
            ]
        elif self._task.Parameters.surfwax_agent_run_method == "oci":
            cmd = [
                "podman",
                "--root",
                self.__podman_data_dir(),
                "run",
                "--rm",
                "--name", "surfwax-agent",
                "--log-opt", "max-size=500m",
                "--log-opt", "max-file=7",
                # Using host network so agent can announce valid address to coordinator
                "--network", "host",
                "--env", "DEBUG=surfwax-agent,surfwax-agent:*,si.ci.requests",
                self.__agent_image(),
            ]
        else:
            raise TaskFailure("Unknown agent_install_method: {}".format(self._task.Parameters.surfwax_agent_run_method))

        return cmd

    def __agent_image(self):
        return "registry.yandex.net/search-interfaces/surfwax-agent:{}".format(self._task.Parameters.surfwax_agent_version)

    @property
    def selenoid_config_path(self):
        return "/etc/selenoid/browsers.json"

    @property
    def selenoid_log_dir(self):
        return "/opt/selenoid/logs"

    @property
    def selenoid_video_dir(self):
        return "/opt/selenoid/video"
