import logging
import subprocess
import json
import os
import tarfile
import time
from sandbox.common.errors import TaskFailure
from sandbox.projects.surfwax.task_manager.common import CommonTaskManager
from sandbox.projects.surfwax.common.utils import install_res


SOURCE_BASH = "source ~/.bash_profile"
BREW_RES_ID = 2665104141
SELENOID_RES_ID = 3323430550
NVM_INSTALL_RES_ID = 2766405356
NODE_VERSION = "12.13.0"
BROWSER_RUNNER_SHUTDOWN_TIMEOUT_MS = 10


class SafariTaskManager(CommonTaskManager):
    def get_task_processes(self):
        selenoid_process = self._run_selenoid()
        surfwax_agent_process = self._run_surfwax_agent()

        return [
            surfwax_agent_process,
            selenoid_process,
            (self._browser_runner, BROWSER_RUNNER_SHUTDOWN_TIMEOUT_MS)
        ]

    def __prepare_env(self):
        os.environ["NPM_CONFIG_REGISTRY"] = "http://npm.yandex-team.ru"
        # have to pass this env to work with binary `xcrun simctl <cmd>`
        os.environ["DEVELOPER_DIR"] = "/Applications/Xcode13.app/Contents/Developer"
        # disable homebrew autoupdate to quick installation
        os.environ["HOMEBREW_NO_AUTO_UPDATE"] = "1"

        self.__install_brew()
        install_nvm_c = self.__install_nvm()
        install_nvm_c.wait()

        self.__install_selenoid()
        self.__prepare_conf()

        prepare_env_processes = [
            self.__delete_simulators(),
            self.__install_appium(),
            self.__install_browser_runner(),
            self._install_surfwax_agent()
        ]

        for result in prepare_env_processes:
            result.wait()

        for code in prepare_env_processes:
            if code.returncode != 0:
                raise TaskFailure("Error while preparing environment. See logs")

        self._browser_runner = self.__start_browser_runner()

        # Need to wait browser runner initialization
        time.sleep(5)

    def __install_brew(self):
        hb_path = install_res(self._task, BREW_RES_ID, "homebrew")

        tar = tarfile.open(os.path.join(hb_path, "homebrew.tar.gz"))
        tar.extractall(hb_path)

        return os.path.join(hb_path, "homebrew", "bin", "brew")

    def __delete_simulators(self):
        log_file = open(str(self._task.log_path("delete_simulators")), "w")

        return self._exec_proc(["xcrun", "simctl", "delete", "all"], log_file)

    def __install_nvm(self):
        nvm_dir = install_res(self._task, NVM_INSTALL_RES_ID, "nvm")
        nvm_path = str(os.path.join(nvm_dir, "install.sh"))

        commands = [
            "touch ~/.bash_profile",
            "export NVM_DIR=\"$HOME/.nvm\"",
            "echo \"export NVM_DIR=\"$HOME/.nvm\"\" >> ~/.bash_profile",
            "echo \"[ -s \"$NVM_DIR/nvm.sh\" ] && \\. \"$NVM_DIR/nvm.sh\"\" >> ~/.bash_profile",
            "chmod +x {}".format(nvm_path),
            nvm_path,
            SOURCE_BASH,
            "nvm install {}".format(NODE_VERSION),
        ]

        return self.__exec_multiple_commands(commands, "install_nvm")

    def __install_selenoid(self):
        return install_res(self._task, SELENOID_RES_ID, self.__get_selenoid_dir())

    def __install_appium(self):
        commands = [
            SOURCE_BASH,
            "npm i {}".format(self.appium_string),
        ]

        return self.__exec_multiple_commands(commands, "install_appium")

    def __install_browser_runner(self):
        commands = [
            SOURCE_BASH,
            "npm i {}".format(self.browser_runner_npm_string)
        ]

        return self.__exec_multiple_commands(commands, "install_browser_runner")

    def _install_surfwax_agent(self):
        commands = [
            SOURCE_BASH,
            "npm i {}".format(self._agent_npm_string())
        ]

        return self.__exec_multiple_commands(commands, "install_surfwax_agent")

    def __prepare_conf(self):
        conf_path = self.selenoid_config_path
        os.mkdir(str(self._task.log_path("selenoid_logs")))
        log_path = str(self._task.log_path("selenoid_logs/log"))
        open(log_path, "a").close()
        browsers = json.loads(self._task.Parameters.browsers)

        for browser in browsers:
            for v in browsers[browser]["versions"]:
                curr_bro = browsers[browser]["versions"][v]

                self.__add_attr_to_conf(curr_bro, "image", [
                    "node", "{}/build/scripts/runner-proxy.js".format(self.browser_runner_dir),
                    "--logPath", log_path, "--browserName", browser, "--browserVersion", v
                ])
                self.__add_attr_to_conf(curr_bro, "platform", "ios")
                self.__add_attr_to_conf(curr_bro, "simulator", [
                    "com.apple.CoreSimulator.SimDeviceType.iPhone-11",
                    "com.apple.CoreSimulator.SimRuntime.iOS-13-3"
                ])
                self.__add_attr_to_conf(curr_bro, "runnerBinPath", "./node_modules/.bin/appium")

        f = open(conf_path, "w")
        f.write(json.dumps(browsers))
        f.close()

    def _prepare_selenoid(self):
        self.__prepare_env()

        selenoid_bin_path = self.__get_selenoid_bin_path()
        commands = [
            SOURCE_BASH,
            "chmod +x {}".format(selenoid_bin_path),
            "{} -disable-docker -save-all-logs".format(selenoid_bin_path)
        ]

        return [self.__join_commands(commands)]

    def __add_attr_to_conf(self, obj, key, value):
        if not hasattr(obj, key) or obj[key] is None:
            obj[key] = value

        return obj

    def __start_browser_runner(self):
        command = [
            SOURCE_BASH,
            "node {}/build/index.js --port 4433 --config {} --limit {}".format(self.browser_runner_dir, self.selenoid_config_path, self.simulators_limit)
        ]

        return self.__exec_multiple_commands(command, "start_browser_runner")

    def __get_selenoid_dir(self):
        return str(self._task.path("selenoid"))

    def __get_selenoid_bin_path(self):
        selenoid_path = self.__get_selenoid_dir()

        return str(os.path.join(selenoid_path, "selenoid_darwin_amd64"))

    def _get_surfwax_agent_cmd(self):
        return ["source ~/.bash_profile && {}".format(self.__get_surfwax_agent_bin())]

    def __get_surfwax_agent_bin(self):
        return str(self._task.path(os.path.join("node_modules", ".bin", "surfwax-agent")))

    def __join_commands(self, commands):
        return " && ".join(commands)

    def __exec_multiple_commands(self, commands, log_file):
        env = os.environ.copy()
        env["DEBUG"] = "surfwax-browser-runner*"

        log_file = open(str(self._task.log_path(log_file)), "w")
        proc = self._exec_proc([self.__join_commands(commands)], log_file, env)
        log_file.close()

        return proc

    def _exec_proc(self, cmd, log_file, env=None):
        logging.debug('Gonna execute command {}'.format(cmd))
        cmd = " ".join(cmd)

        return subprocess.Popen(cmd, stdout=log_file, stderr=log_file, env=env, shell=True)

    @property
    def appium_string(self):
        return "appium@{}".format(self._task.Parameters.appium_version)

    @property
    def browser_runner_npm_string(self):
        return "@yandex-int/surfwax-browser-runner@{}".format(self._task.Parameters.surfwax_browser_runner_version)

    @property
    def browser_runner_dir(self):
        return "./node_modules/@yandex-int/surfwax-browser-runner"

    @property
    def selenoid_config_path(self):
        return "{}/browsers.json".format(self.__get_selenoid_dir())

    @property
    def simulators_limit(self):
        return self._task.Parameters.selenoid_sessions_per_agent_limit

    @property
    def selenoid_log_dir(self):
        return str(self._task.path())

    @property
    def selenoid_video_dir(self):
        return None
