#  -*- coding: utf-8 -*-

import requests

from functools import reduce
from sandbox import sdk2
from sandbox.sdk2 import yav


PAGE_HEAD = """{{toc}}
=====Общая информация=====
Некоторые панели хранятся в Аркадии: https://a.yandex-team.ru/arc/trunk/arcadia/search/tools/iconostasis_panels
Генератор с ручным приводом. Отредактированная панель явным образом загружается в голован из консоли.
Большая часть светофорных панелей при этом всё ешё генерируется с помощью quentao (aka junk сереглонда) - подробности на странице
((https://wiki.yandex-team.ru/AndrejjAbroskin/golovan/panel Генератор алертов и панелей))
Информация про нативные методы работы с панелями в головане в официальной документации: https://wiki.yandex-team.ru/golovan/templatium/writing/
Приёмка алерта на иконостас проводится согласно ((https://wiki.yandex-team.ru/jandekspoisk/sepe/dezhurnajasmena/pravila-raboty/priemka-alerta/ описанию))


"""


def get_martydb_panels():
    """ curl -sH -X POST "https://workplace.z.yandex-team.ru/api/workplace.services.CatalogService/listVerticals"
    --data '{}' | jq -r '.objects | map(.martyPanels) | map (.objects) | add | map(select(.needCheck == true)) | map(.url)[]'
    """

    """ curl -s -XPOST https://warden.z.yandex-team.ru/api/warden.Warden/getComponentList --data '{}' |
    jq '.components[] | select (.monitoring.panels | length > 0) | [.name, (.monitoring.panels[] | [ .name, .url, .iconostasAllocated]) ]'
    """

    verticals_resp = requests.post(
        'https://warden.z.yandex-team.ru/api/warden.Warden/getComponentList', json={})
    verticals_data = verticals_resp.json()
    panels = []
    for vertical in verticals_data['components']:
        if 'monitoring' in vertical and 'panels' in vertical['monitoring']:
            objects = vertical['monitoring']['panels']
            if not objects:
                continue
            for panel in objects:
                if (
                    'iconostasAllocated' in panel and panel['iconostasAllocated']
                    and 'url' in panel and panel['url']
                ):
                    panels.append(
                        {
                            'vertical': vertical['name'],
                            'name': panel['name'] if 'description' in panel else '-',
                            'url': panel['url'],
                            'description': panel['description'] if 'description' in panel else '-'
                        }
                    )
    return panels


def format_table(panels):
    result = ['#|']
    header = ['name', 'url', 'description']
    result.append('|| ' + ' | '.join(header) + ' ||')

    def reduce_panels(results, panel):
        v = panel['vertical']
        if v in results:
            results[v].append(panel)
        else:
            results[v] = [panel, ]
        return results

    grouped = reduce(reduce_panels, panels, {})

    for vertical in sorted(grouped.keys()):
        result.append('|| ' + '=====**' + vertical.capitalize() + '** ||')

        for panel in grouped[vertical]:
            result.append(
                '|| ' +
                ' | '.join([panel.get(key, '-') for key in header]) +
                ' ||'
            )

    result.append('|#')
    return '\n'.join(result)


def upload_to_wiki(token, page, body):
    headers = {'Authorization': 'OAuth {token}'.format(token=token)}
    request_data = {
        'title': 'Iconostasis panels from MartyDB',
        'body': body
    }
    result = requests.post(page, headers=headers, json=request_data)
    return result.text


class Sync_Marty_Panels(sdk2.Task):
    class Parameters(sdk2.Parameters):
        secret = sdk2.parameters.String("ID секрета")
        token_name = sdk2.parameters.String("Ключ секрета")
        page = sdk2.parameters.String("Адрес страницы")

    def on_execute(self):
        panels = get_martydb_panels()
        table = format_table(panels)

        body = PAGE_HEAD + table

        secret = yav.Secret(self.Parameters.secret)
        token = secret.data()[self.Parameters.token_name]

        default_page = 'https://wiki.yandex-team.ru/jandekspoisk/sepe/dezhurnajasmena/iconostasispanels_from_MartyDB/'
        page = self.Parameters.page or default_page

        upload_to_wiki(
            token,
            page.replace('https://wiki.yandex-team.ru/', 'https://wiki-api.yandex-team.ru/_api/frontend/'),
            body
        )
        self.set_info('Page was uploaded: {page}'.format(page=page))
