import os.path
import platform

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task

from sandbox.projects.tank.load_resources import resources as tank_resources

import sandbox.common.types.client as ctc


_TANK_MODULE = 'yandextank'

_TANK_VERSION_18 = "1.8"
_TANK_VERSION_19 = "1.9"


class PythonPackages(parameters.SandboxStringParameter):
    """Package name input parameter."""
    name = "python_packages"
    description = "Python package names (if necessary pass version here: pack==v.vv.vvv)"


class DebianPackages(parameters.SandboxStringParameter):
    name = "debian_packages"
    description = "Space-separated .deb packages list to be installed"
    default_value = ""


class DebianBinaries(parameters.SandboxStringParameter):
    name = "debian_binaries"
    description = "Space-separated binaries to copy to tarball"
    default_value = ""


class TankVersion(parameters.SandboxStringParameter):
    name = "tank_version"
    description = "Tank version"
    choices = [("1.8", _TANK_VERSION_18), ("1.9", _TANK_VERSION_19)]
    default_value = _TANK_VERSION_18


class BuildTankVirtualenv(task.SandboxTask):
    """
    Build virtual environment for yandex tank
    """

    input_parameters = (
        PythonPackages,
        DebianPackages,
        DebianBinaries,
        TankVersion
    )

    client_tags = ctc.Tag.LINUX_PRECISE
    type = "BUILD_TANK_VIRTUALENV"

    @property
    def privileged(self):
        return bool(self.ctx.get(DebianPackages.name, "").strip())

    def on_execute(self):
        debian_packages = self.ctx[DebianPackages.name]
        if debian_packages:
            # enable additional repositories
            with open("/etc/apt/sources.list.d/yandex-load.list", "w") as source_file:
                source_file.write("deb http://load.dist.yandex.ru/load stable/all/\n")
                source_file.write("deb http://load.dist.yandex.ru/load stable/amd64/\n")

            process.run_process(["apt-get", "update"], log_prefix="apt-get")
            process.run_process(
                ["apt-get", "install", "-y"] + debian_packages.split(),
                log_prefix="apt-get"
            )

        archive_name = 'yandextank-virtualenv.tar.gz'
        tank_version = self.ctx[TankVersion.name]

        with environments.VirtualEnvironment() as venv:
            # install python packages
            python_packages = self.ctx[PythonPackages.name]
            if tank_version == _TANK_VERSION_18 and _TANK_MODULE not in python_packages:
                raise errors.SandboxTaskFailureError("Module {} should be in list of python modules".format(_TANK_MODULE))
            for python_package in python_packages.split():
                venv.pip(python_package)

            # install binaries
            binary_dir = os.path.join(venv.root_dir, "bin")
            paths.make_folder(binary_dir)
            for binary_path in self.ctx[DebianBinaries.name].split():
                paths.copy_path(binary_path, os.path.join(binary_dir, os.path.basename(binary_path)))

            # special flag for new version
            if tank_version == _TANK_VERSION_19:
                with open(os.path.join(venv.root_dir, "next"), "w") as next_file:
                    next_file.write("1.9")

            venv.pack_bundle(archive_name)

        if tank_version == _TANK_VERSION_18:
            resource_type = tank_resources.YANDEX_TANK_VIRTUALENV_18
        elif tank_version == _TANK_VERSION_19:
            resource_type = tank_resources.YANDEX_TANK_VIRTUALENV_19
        else:
            raise errors.SandboxTaskFailureError("Unsupported tank version")

        self.create_resource(
            description=self.descr,
            resource_path=archive_name,
            resource_type=resource_type,
            attributes={'platform': platform.platform()},
        )


__Task__ = BuildTankVirtualenv
