# -*- coding: utf-8 -*-

import time
import json
import random
import logging

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary
import sandbox.common.types.notification as ctn
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.projects.tank.Firestarter import tasks, status
from sandbox.projects.tank.SandboxTank import SandboxTank


TEST_CHECK_INTERVAL = 5  # seconds
CLIENT_TAGS = {
    'iva': ctc.Tag.IVA,
    'man': ctc.Tag.MAN,
    'myt': ctc.Tag.MYT,
    'sas': ctc.Tag.SAS,
    'vla': ctc.Tag.VLA,
}
GIFS_404 = [
    {
        'alt': '404_1',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197411302',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197411302/view'
    },
    {
        'alt': '404_2',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197430477',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197430477/view'
    },
    {
        'alt': '404_3',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197433250',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197433250/view'
    },
    {
        'alt': '404_4',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197434042',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197434042/view'
    },
    {
        'alt': '404_5',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197435014',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197435014/view'
    },
    {
        'alt': '404_6',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197437797',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197437797/view'
    },
    {
        'alt': '404_7',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2196903277',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2196903277/view'
    },
    {
        'alt': '404_8',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197439985',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197439985/view'
    },
    {
        'alt': '404_9',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197440842',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197440842/view'
    },
    {
        'alt': '404_10',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197443126',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197443126/view'
    },
    {
        'alt': '404_11',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197447296',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197447296/view'
    },
    {
        'alt': '404_12',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197448201',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197448201/view'
    },
    {
        'alt': '404_13',
        'proxy_link': 'https://proxy.sandbox.yandex-team.ru/2197449764',
        'resource_link': 'https://sandbox.yandex-team.ru/resource/2197449764/view'
    }
]


class Firestarter(sdk2.Task):
    """ Start loadtest with YandexTank, upload results to Lunapark """

    result = {}

    class Context(sdk2.Task.Context):
        initial_config = {}
        fs_task = ''
        lunapark_id = 0
        last_stage = 'initiated'
        error = ''
        task_name = ''
        queue = ''
        custom = False
        crossdc = False

    class Parameters(sdk2.Task.Parameters):
        # TODO: timeout should depend on test duration
        kill_timeout = 3600

        dry_run = sdk2.parameters.Bool('Dry run', description='Pass all check stages without shooting', default=False)
        wait_finish = sdk2.parameters.Bool('Wait finish', description='Wait for the end of the test on the tank', default=False)

        tank_config = sdk2.parameters.String(
            'Tank yaml config',
            required=True,
            multiline=True
        )
        monitoring_config = sdk2.parameters.String('Monitoring config', multiline=True)

        with sdk2.parameters.Output:
            result = sdk2.parameters.String('Firestarter result status', default='')
            lunapark_id = sdk2.parameters.Integer('Lunapark test id', default=0)

        with sdk2.parameters.Group('Binary source'):
            use_last_binary = sdk2.parameters.Bool('Use last released binary', default=True)

    def on_save(self):

        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={
                    'task_name': 'Firestarter',
                    'release': 'stable'
                }).first()

        self.Parameters.notifications = [
            sdk2.Notification(
                [
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.NO_RES,
                ],
                ['host=firestarter_sandbox&service=execution_status'],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.CRIT
            ),
            sdk2.Notification(
                [
                    ctt.Status.TIMEOUT,
                    ctt.Status.EXPIRED,
                ],
                ['host=firestarter_sandbox&service=execution_status'],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.WARN
            ),
            sdk2.Notification(
                [
                    ctt.Status.SUCCESS
                ],
                ['host=firestarter_sandbox&service=execution_status'],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.OK
            )
        ]

    def on_enqueue(self):
        """
        Since there is no FirestarterTask initiated on this stage,
        we construct result message manually
        """
        if not self.Parameters.tank_config:
            self.result = {
                'status': status.InternalStatus.FAILED_TO_START,
                'lunapark_id': 0,
                'config': {},
                'errors': True,
                'message': {'firestarter': ['Empty tank config']}
            }
            raise common.errors.TaskFailure('Empty tank config')

    def collect_results(self):
        results = {
            'lp_link': '',
            'dl_link': '',
            'dl_chart': ''
        }
        if self.Parameters.result:
            if self.Parameters.lunapark_id:
                results['lp_link'] = 'https://lunapark.yandex-team.ru/{}'.format(self.Context.lunapark_id)
                results['dl_link'] = 'https://datalens.yandex-team.ru/'\
                    'clod4tpb3lek6-load-report?tab=Er&job_id={}&task_id={}'.format(
                        self.Context.lunapark_id, self.Context.task_name
                    )
                results['dl_chart'] = '<iframe frameborder="0" width="100%" height="400px" ' \
                                      'src="https://charts.yandex-team.ru/preview/editor/x68py0jj1tber?' \
                                      'job_id={job}&task_id={task}&case_id=overall&_embedded=1">'.format(
                    job=self.Parameters.lunapark_id, task=self.Context.task_name
                )
            else:
                random_404 = random.choice(GIFS_404)
                results['dl_chart'] = '<img src="{src}" alt="{alt}">'.format(
                    src=random_404['proxy_link'],
                    alt=random_404['alt']
                )

        return results

    @sdk2.header(title='Результаты')
    def test_results(self):
        results = self.collect_results()
        result_page = ''
        if results.get('dl_link'):
            result_page += 'Datalens report: <a href="{dl_link}">{dl_link}</a><br />'
        if results.get('lp_link'):
            result_page += 'Lunapark link: <a href="{lp_link}">{lp_link}</a><br />'
        if results.get('dl_chart'):
            result_page += '{dl_chart}'
        return result_page.format(**results)

    def on_execute(self):
        try:
            fs_task = tasks.FirestarterTask(
                sandbox_task_id=self.id,
                config=self.Parameters.tank_config,
                monitoring_config=self.Parameters.monitoring_config,
                create_sb_tank=self.create_sb_tank,
                wait_finish=self.Parameters.wait_finish
            )
        except status.FirestarterError as fe:
            self.Context.fs_task = json.dumps({
                'id_': self.id,
                'lunapark_id': None,
                'config': None,
                'errors': True,
                'error_message': fe.error,
                'status': fe.status,
                'tank_version': None
            })

            self.Context.initial_config = self.Parameters.tank_config
            self.Context.last_stage = fe.section
            self.Context.error = fe.error

        else:
            self.Context.initial_config = fs_task.config_manager.config
            self.Context.fs_task = repr(fs_task)
            self.process_task(fs_task)


    def on_finish(self, *args):
        self.save_status()

    def on_break(self, *args):
        self.save_status()

    @staticmethod
    def link_format(href, text, description=''):
        return '{}<a href="{}">{}</a>'.format(description, href, text)

    def save_status(self):
        assigned_tags = []
        self.Parameters.result = self.Context.fs_task

        if self.Context.error:
            self.set_info(
                'Task failed on stage {}. Error message:\n{}'.format(self.Context.last_stage, self.Context.error),
                do_escape=True
            )
            assigned_tags.extend(['failed', 'stage:{}'.format(self.Context.last_stage)])
        else:
            assigned_tags.append('test_started')

        try:
            tank_version = json.loads(self.Context.fs_task)['tank_version']
            assigned_tags.append('tank_version:{version}'.format(version=tank_version))
        except (KeyError, TypeError, IndexError, json.decoder.JSONDecodeError):
            logging.error('Incorrect FS task presentation format. Unknown tank version', exc_info=True)

        if self.Context.task_name:
            assigned_tags.append('task:{task}'.format(task=self.Context.task_name))
            assigned_tags.append('queue:{queue}'.format(queue=self.Context.task_name.split('-')[0]))

        if self.Context.custom:
            assigned_tags.append('custom')
        if self.Context.crossdc:
            assigned_tags.append('crossdc')


        self.Parameters.lunapark_id = self.Context.lunapark_id
        if self.Context.lunapark_id:
            lunapark_link = 'https://lunapark.yandex-team.ru/{}'.format(self.Context.lunapark_id)
            datalens_link = \
                'https://datalens.yandex-team.ru/clod4tpb3lek6-load-report?tab=Er&job_id={}&task_id={}'.format(
                    self.Context.lunapark_id, self.Context.task_name
                )
            self.set_info(
                self.link_format(lunapark_link, lunapark_link, description='Lunapark link: '),
                do_escape=False
            )
            self.set_info(
                self.link_format(datalens_link, datalens_link, description='Datalens link: '),
                do_escape=False
            )

        initial_tags = self.server.task[self.id].read()['tags']
        self.server.task[self.id] = {'tags': initial_tags + assigned_tags}

    def process_task(self, fs_task):
        logger = logging.getLogger('PROCESS TASK')
        try:
            fs_task.parse_config()
            self.Context.fs_task = repr(fs_task)
            self.Context.custom = fs_task.config_manager.custom

            self.Context.task_name = fs_task.parsing_result['task']
            # NB: old tags are erased!
            self.server.task[self.id] = {'tags': ['operator:{}'.format(fs_task.parsing_result['operator'])]}

            fs_task.validate_config()
            self.Context.fs_task = repr(fs_task)

            fs_task.define_hosts()
            self.Context.fs_task = repr(fs_task)
            self.Context.crossdc = fs_task.crossdc

            if not self.Parameters.dry_run:
                if fs_task.status == status.InternalStatus.TANK_FOUND:
                    fs_task.init_shooting()
                    self.Context.fs_task = repr(fs_task)

                while fs_task.status not in (
                    status.InternalStatus.FAILED_TO_START,
                    status.InternalStatus.FAILED,
                    status.InternalStatus.FINISHED
                ):
                    fs_task.check_test_status()
                    self.Context.fs_task = repr(fs_task)
                    if fs_task.test_id and not self.Context.lunapark_id:
                        self.Context.lunapark_id = fs_task.test_id
                        if not fs_task.wait_finish:
                            break
                    time.sleep(TEST_CHECK_INTERVAL)

        except status.FirestarterError as fe:
            logger.debug('Firestarter Error %s',fe.error, exc_info=True)
            logger.error('Firestarter Error. See the debug log')
            fs_task._save(target_status=fe.status, error=fe.error, section=fe.section)

        except Exception:
            logger.debug('Internal Error', exc_info=True)
            logger.error('Internal Error. See the debug log')
            fs_task._save(target_status=status.InternalStatus.FAILED, error='Firestarter Internal Error.', section=fs_task.stage)

        self.Context.fs_task = repr(fs_task)
        self.Context.last_stage = fs_task.stage
        self.Context.error = fs_task.errors.get(fs_task.stage, '')

    def create_sb_tank(self, dc):
        """
        Run the sandbox task SandboxTank
        :param dc: Datacenter
        :type dc: str
        :return: task_id, error
        :rtype: int str
        """
        logger = logging.getLogger('SANDBOX TANK')
        if str(dc).lower() in CLIENT_TAGS:
            try:
                tank_task = SandboxTank(
                    self,
                    __requirements__={'client_tags': CLIENT_TAGS[str(dc).lower()]},
                    description='Start from Firestarter'
                ).enqueue()
                return tank_task.id, ''
            except Exception:
                error = 'Failed to create sandbox tank!'
                return logger.warning(error, exc_info=True), error
        else:
            error = 'Unknown DataCenter {dc}!'.format(dc=dc)
            return logger.warning(error, exc_info=True), error
