import re
import random
import logging

from .status import InternalStatus, FirestarterError
from .config_parser import (
    IPv4,
    IPv6,
    FQDN,
)
from .external_calls import (
    call_tank_finder,
    call_tank_finder_deploy,
    call_tank_finder_nanny,
    define_host_dc,
    get_sandbox_output,
    get_tank_version,
)


MIN_TANK_VERSION = '1.17.4'
logger = logging.getLogger('DEFINE HOSTS')

class DefineHosts:

    def __init__(self, parsing_result):
        from load.projects.tankapi_cmd.src import client
        self.tankapi_client = client
        self.parsing_result = parsing_result

    # GET TARGET
    def get_target(self):
        """
        Checking the target parsed value and defining the target as a host or a special value. Cases are considered:
        - custom load generators as BFG, jMeter, Pandora and for the config-file option for Pandora
        - target as a FQDN, IPv4 or IPv6
        - target as a nanny service
        - target as a deploy stage
        - target as an output data for the sandbox task
        """
        target_found = ''
        target = self.parsing_result['target']['value']
        port = self.parsing_result['target']['port']
        dc = self._preliminary_tank_dc_() or self.parsing_result['dc']

        if target == 'in_config':
            self.target_in_config = True
            target_found, port_found = target, ''

        elif (
            re.match(IPv4, target)
            or re.match(IPv6, target)
            or re.match(FQDN, target)
        ):
            target_found, port_found = target, port

        elif target.startswith('nanny:'):
            target_found, port_found = self._get_nanny_target_(target, dc)

        elif target.startswith('deploy:'):
            target_found, port_found = self._get_deploy_target_(target, dc), port

        elif target.startswith('sandbox:'):
           target_found, port_found = self._get_sandbox_target_(target), port

        if not target_found:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='No one host is found for target {}'.format(target),
                section='get_target',
            )

        return target_found, port_found

    def _get_nanny_target_(self, target, dc):
        """
        Get target for the case when target is set as the nanny service
        :param target:
        :type target: str
        :param dc:
        :type dc: str
        :return: target
        :rtype: str
        """
        hosts = call_tank_finder_nanny(target, dc)
        if hosts and isinstance(hosts, list):
            random.shuffle(hosts)
            return hosts[0].split(':')
        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='RTC service {} has no instances'.format(target),
                section='get_target',
            )

    def _get_deploy_target_(self, target, dc):
        """
        Get target for the case when target is set as the deploy project
        :param target:
        :type target: str
        :param dc:
        :type dc: str
        :return: target
        :rtype: str
        """
        hosts = call_tank_finder_deploy(target, dc)
        if hosts and isinstance(hosts, list):
            random.shuffle(hosts)
            return hosts[0]
        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Deploy stage {} has no instances'.format(target),
                section='get_target',
            )

    def _get_sandbox_target_(self, target):
        """
        Get target for the case when target is set as the sandbox output parameter
        :param target:
        :type target: str
        :return: target
        :rtype: str
        """
        try:
            task_type, task_tags, output_value = target.split(':', maxsplit=1)[1].split('.', maxsplit=2)
        except (AttributeError, ValueError):
            logger.debug('Wrong sandbox parameters %s', target, exc_info=True)
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='The sandbox task is not sufficiently specified in the value {}'.format(target),
                section='get_target',
            )
        host = get_sandbox_output(task_type, output_value, task_tags)
        if host:
            if not (host.endswith('ru') or host.endswith('net')):
                host += '.ru'
            return host
        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Sandbox task {} has no host in output parameters'.format(task_type),
                section='get_target',
            )

    # GET TANKS
    def get_tanks(self, target):
        """
        Checking the tank parsed value and defining the tank as a list of the LunaparkTank objects. Cases are considered:
        - tank as a FQDN, IPv4 or IPv6
        - tank as a list of the FQDN, IPv4 or IPv6
        - tank as an alias for a group in the form deploy stage, nanny service or common value
        :param target:
        :type target: str
        :return: List of tanks
        :rtype: list
        """
        tank = self.parsing_result['tank']['value']
        port = self.parsing_result['tank']['port']
        tanks_found = []

        if isinstance(tank, list):
            tanks_found = self._get_tanks_if_tank_is_list_(tank, target, port)

        elif tank == 'common' or tank.startswith('nanny:') or tank.startswith('deploy:'):
            tanks_found = self._get_tanks_if_tank_is_alias_(tank, target, port)

        elif re.match(IPv4, tank) or re.match(IPv6, tank) or re.match(FQDN, tank):
            tanks_found = self._get_tanks_if_tank_is_single_(tank, target, port)

        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Tank value {tank} is an invalid'.format(tank=tank),
                section='define_hosts',
            )

        return self._filter_tank_by_version_(tanks_found)

    def _get_tanks_if_tank_is_single_(self, tank, target, port=8083):
        """
        Get tanks for the case when the tank is set as a strictly defined host
        :param tank:
        :type tank: str
        :param target:
        :type target: str
        :param port:
        :type port: str or int
        :return: List of tanks
        :rtype: list
        """
        if re.match(IPv6, tank):
            tank = '[{tank}]'.format(tank=tank)

        tank_dc = define_host_dc(tank)
        if not tank_dc:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Impossible to determine the datacenter for the tank {}'.format(tank),
                section='get_tanks',
            )

        elif tank_dc != define_host_dc(target) and target != 'in_config' and self.parsing_result['dc'] != 'cross':
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Tank and target are not in the same DC',
                section='get_tanks',
            )
        return [self.tankapi_client.LunaparkTank(host=tank, port=port)]

    def _get_tanks_if_tank_is_list_(self, tanks, target, port=8083):
        """
        Get tanks for the case when the tank is set as a strictly defined host
        :param tank:
        :type tank: list
        :param target:
        :type target: str
        :param port:
        :type port: str or int
        :return: List of tanks
        :rtype: list
        """
        tanks_found = []
        for tank in tanks:
            try:
                tanks_found.extend(self._get_tanks_if_tank_is_single_(tank, target, port))
            except FirestarterError:
                logger.debug('Tank %s is not suitable for this shooting', tank, exc_info=True)
        if tanks_found:
            return tanks_found
        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='No one tank from list is not suitable for this shooting',
                section='get_tanks',
            )

    def _get_tanks_if_tank_is_alias_(self, tank, target, port=8083):
        """
        Get tanks for the case when tank is set as the common, a nanny service or a deploy stage
        :param tank:
        :type tank: str
        :param port:
        :type port: str or int
        :return: List of tanks
        :rtype: list
        """
        if tank in ('common', 'nanny:production_yandex_tank'):
            if target == 'in_config':
                raise FirestarterError(
                    status=InternalStatus.FAILED_TO_START,
                    error='Shooting with custom generators from public tanks is not allowed',
                    section='get_tanks',
                )
            elif not define_host_dc(target):
                raise FirestarterError(
                    status=InternalStatus.FAILED_TO_START,
                    error='Crossdc shooting from public tanks is not allowed',
                    section='get_tanks',
                )
            else:
                tank_dc = ''
        else: 
            tank_dc = self.parsing_result['dc'] or define_host_dc(target)

        tanks_found = call_tank_finder(target, tank, tank_dc)
        logger.debug('For tank as alias, tanks_found %s', tanks_found)
        if not tanks_found:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='No tank was found for the target {target}'.format(target=target),
                section='get_tanks',
            )
        else: 
            return [self.tankapi_client.LunaparkTank(host=tank.split(':')[0], port=(tank.split(':') + [''])[1] or port) for tank in tanks_found]

    def _filter_tank_by_version_(self, tanks):
        """
        Check that the version of the found tanks is higher than the allowed one
        :param tank:
        :type tank: list
        :return: List of tanks with a suitable tank version
        :rtype: list
        """
        from packaging import version
        logger = logging.getLogger('TANK VERSION')
        true_tanks = []

        if isinstance(tanks, self.tankapi_client.LunaparkTank):
            tanks = [tanks]
        if not tanks:
            logger.debug('No one tank was received to check the version')
        elif isinstance(tanks, list):
            random.shuffle(tanks)
            true_tanks = [
                tank
                for tank in tanks
                if isinstance(tank, self.tankapi_client.LunaparkTank)
                and version.parse(str(get_tank_version('{}:{}'.format(tank.host, tank.port)))) >= version.parse(MIN_TANK_VERSION)
            ]
        else:
            logger.debug('Incorrect tanks format was received: %s', tanks)

        return true_tanks

    def _preliminary_tank_dc_(self):
        """
        Return tank datacenter if tank is specifid as host
        :return: dc
        :rtype: str
        """
        tank = self.parsing_result['tank']['value']
        if (
            re.match(IPv4, tank)
            or re.match(IPv6, tank)
            or re.match(FQDN, tank)
        ):
            return define_host_dc(tank)
