import time
import logging
import sandbox.sdk2 as sdk2
from sandbox.common.types import task as ctt
from .status import InternalStatus, FirestarterError


GET_IP_TIMEOUT = 15
MAX_ATTEMPT = 60
logger = logging.getLogger('SANDBOX TANK')

def get_tank_ip(task_id):
    """
    Monitoring the task SandboxTanx until the context parameter 'tank_ip' is set and returning this parameter.
    :param task_id: ID of the task with tank which ip we want to receive.
    :type task_id: int
    :return: IP address of the container with the tank
    :rtype: str
    """
    tank_ip = None
    attempt = 0

    while not tank_ip and attempt < MAX_ATTEMPT:
        sb_tank = None
        try:
            sb_tank = find_tank(task_id)
            tank_ip = sb_tank.Context.tank_ip
            logger.debug('updated: %s\tinfo: %s\tip: %s', sb_tank.updated, sb_tank.info, tank_ip)
        except AttributeError:
            logger.debug('Failed to get tank IP address. Attempt %s', attempt, exc_info=True)
        time.sleep(GET_IP_TIMEOUT)
        attempt += 1

    if attempt == MAX_ATTEMPT:
        raise FirestarterError(
            status=InternalStatus.SANDBOX_TANK_FAILED,
            error='Cannot to get the IP address os the sandbox tank',
            section='sandbox_tank',
        )

    logger.info('Tank %s is started in: %s', tank_ip, sb_tank.Context.tank_dc)
    return '[{}]'.format(tank_ip)


def find_tank(task_id):
    """
    Search for a sandbox task and determine its status.
    :param task_id: ID of the task you are looking for.
    :type task_id: int
    :return: sandbox task object or None
    :rtype: sdk2.Task or None
    """
    try:
        sb_tank = sdk2.Task.find(id=task_id).first()
        logger.info('Task status %s', sb_tank.status)
        if sb_tank.status in ctt.Status.Group.BREAK or sb_tank.status == ctt.TaskStatus.FAILURE:
            logger.error('Task %s is failed.', task_id)
            raise FirestarterError(
                status=InternalStatus.SANDBOX_TANK_FAILED,
                error='SandboxTank task failed',
                section='sandbox_tank',
            )
        else:
            return sb_tank

    except AttributeError:
        logger.debug('Task %s is not found.', task_id, exc_info=True)
