import os
import re
import uuid
import json
import time
import logging
import tempfile

from .status import InternalStatus, FirestarterError
from .define_hosts import DefineHosts
from .sandbox_tank import get_tank_ip
from .external_calls import (
    call_validator,
    get_tank_version,
    define_host_dc,
)
from .config_parser import (
    ConfigManager,
    IPv4,
    IPv6,
    FQDN,
    TASK_ID_REGEX,
)


WAIT_EXPONENTIAL_MAX = 10000
STOP_MAX_DELAY = 30000


class FirestarterTask:
    """
    Basic task for testflow
    """

    def __init__(self, sandbox_task_id, config, monitoring_config, create_sb_tank, wait_finish=False):
        """
        :parameter config: Tank config
        :type config: Dict
        :parameter monitoring_config: Monitoring config
        :type monitoring_config: str
        """
        self.task_id = uuid.uuid4().hex
        self.status = InternalStatus.NEW

        self.errors = {}
        self.parsing_result = {}
        self.tanks = []
        self.stage = ''

        self.wait_finish = wait_finish
        self.create_sb_tank = create_sb_tank
        self.tank_version = None
        self.crossdc = False

        self.config_manager = ConfigManager(sandbox_task_id, config)
        self.monitoring_config = monitoring_config
        self.test = TankTest()
        self.test_id = 0

        logging.info('FirestarterTask %s created', self.task_id)

    def __repr__(self):
        return json.dumps(
            {
                'id_': self.task_id,
                'lunapark_id': self.test_id,
                'config': self.config_manager.config,
                'errors': bool(self.errors),
                'error_message': self.errors,
                'status': self.status,
                'tank_version': self.tank_version,
            }
        )

    def _save(self, target_status, config=None, error=None, section='firestarter'):
        logger = logging.getLogger('TASK SAVE')
        logger.info('Start stage %s', section)
        self.status = target_status
        self.stage = section
        if config:
            self.config_manager.config = config
        if error:
            self.errors.setdefault(section, error)
            logger.warning('Task id %s got an error on the %s stage: %s', self.task_id, section, error)

    #########################################
    #            PARSING CONFIG             #
    #########################################

    def parse_config(self):
        self.config_manager.set_launched_from()
        self._save(target_status=self.status, config=self.config_manager.config, section='parse_config')
        logger = logging.getLogger('PARSE CONFIG')

        def tank_not_fits_for_custom(tank_value, dc=''):
            """
            Checking a tank value for using in a custom load scenarios
            """
            return (
                not isinstance(tank_value, str)
                or tank_value == 'common'
                or tank_value.startswith('nanny:')
                or tank_value == 'sandbox' and not dc
            )

        def set_parsed_value(value, purpose=''):
            """
            Checking that returned value of the tank or target is correct
            """
            if not value:
                raise FirestarterError(
                    status=InternalStatus.FAILED_TO_START, error='No one {purpose} defined in config'.format(purpose=purpose), section='parse_config'
                )

            elif purpose == 'tank' and isinstance(value, list):
                return [tank for tank in value if (re.match(IPv4, tank) or re.match(IPv6, tank) or re.match(FQDN, tank))]

            elif not isinstance(value, str):
                raise FirestarterError(
                    status=InternalStatus.FAILED_TO_START,
                    error='Wrong {purpose} format'.format(purpose=purpose),
                    section='parse_config',
                )
            return value

        # Get TASK value from config
        task = self.config_manager.get_task()
        if not task:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START, error='No task defined in uploader', section='parse_config'
            )
        elif not re.match(TASK_ID_REGEX, task):
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START, error='Wrong task format', section='parse_config'
            )

        # Get operator value from config or set default
        operator = self.config_manager.get_operator()
        if not operator:
            operator = 'lunapark'
            self.config_manager.set_operator(operator)

        # Get tank params from config
        value, tank_port = self.config_manager.get_tank()
        tank_value = set_parsed_value(value, 'tank')
        if tank_value == 'sandbox':
            self.wait_finish = True

        # Get the datacenter value from the metaconf section is exists
        dc = self.config_manager.get_datacenter() or ''

        # Checking the configuration for the use of custom load scenarios
        if self.config_manager.check_for_custom_gun():
            self.config_manager.custom = True
            if tank_not_fits_for_custom(tank_value, dc):
                raise FirestarterError(
                    status=InternalStatus.FAILED_TO_START,
                    error='Detected custom load scenario on the pandora or bfg.\nPlease specify the single or sandbox tank in config.',
                    section='parse_config',
                )
            else:
                logger.info('BFG, custom pandora or pandora.config_file option is used!')
                value, target_port = 'in_config', ''
        else:
            # Get target params from config
            value, target_port = self.config_manager.get_target()

        target_value = set_parsed_value(value, 'target')

        # Join getted params in the common structure
        self.parsing_result = {
            'target': {
                'value': target_value,
                'port': target_port or 80,
            },
            'tank': {
                'value': tank_value,
                'port': tank_port or 8083,
            },
            'operator': operator,
            'task': task,
            'dc': dc,
        }

        logger.info('Defined shoot points: %s', self.parsing_result)

    #########################################
    #           VALIDATE CONFIG             #
    #########################################

    def validate_config(self):
        """Validates config in validator"""
        self._save(target_status=self.status, config=self.config_manager.config, section='validate_config')
        logger = logging.getLogger('VALIDATE CONFIG')
        validated_config, errors = call_validator(self.config_manager.config)
        if errors:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Config invalid. {}'.format(errors),
                section='validate_config',
            )
        else:
            self._save(target_status=InternalStatus.VALIDATED, config=validated_config, section='validate_config')
            logger.info('Firestarter task: %s', self.__repr__())

    #########################################
    #             DEFINE HOSTS              #
    #########################################

    def define_hosts(self):
        """External call to tank finder: define apppropriate tank and target"""
        self._save(target_status=self.status, config=self.config_manager.config, section='define_hosts')
        df = DefineHosts(self.parsing_result)

        target, port = df.get_target()
        if target and target != 'in_config':
            self.config_manager.set_target(target, port)

        if self.parsing_result['tank']['value'] == 'sandbox':
            self.tanks = self._get_tanks_if_tank_is_sandbox_(target) 
        else:
            self.tanks = df.get_tanks(target)

        if self.tanks:
            self._save(
                target_status=InternalStatus.TANK_FOUND, config=self.config_manager.config, section='define_hosts'
            )
        else:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='No tank found',
                section='define_hosts'
            )

        if not self.config_manager.custom and define_host_dc(self.tanks[0].host) != define_host_dc(target):
            self.crossdc = True

    def _get_tanks_if_tank_is_sandbox_(self, target):
        from load.projects.tankapi_cmd.src import client
        logger = logging.getLogger('SANDBOX TANK')
        tank_dc = self.parsing_result['dc'] or define_host_dc(target)
        if not tank_dc:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='No datacenter was found to run the test',
                section='sandbox_tank',
            )
        else:
            # Run sandbox tank
            logger.info('Run SandboxTank in the datacenter: %s', tank_dc)
            sandbox_tank_id, error = self.create_sb_tank(tank_dc)

            if sandbox_tank_id:
                logger.info('SandboxTank task id: %d', sandbox_tank_id)
                return [client.LunaparkTank(host=get_tank_ip(sandbox_tank_id), port='8083')]
            else:
                logger.error('Error when creating a tank in the sandbox. %s', error)
                raise FirestarterError(
                    status=InternalStatus.SANDBOX_TANK_FAILED,
                    error='Error when creating a tank in the sandbox',
                    section='sandbox_tank',
                )

    #########################################
    #            INIT SHOOTING              #
    #########################################

    def init_shooting(self):
        from load.projects.tankapi_cmd.src import client
        logger = logging.getLogger('INIT SHOOTING')
        logger.info('Found tanks %s', ', '.join(['{tank}:{port}'.format(tank=tank.host, port=tank.port ) for tank in self.tanks])) 

        class TestInitiator(client.TestPerformer):
            def __init__(self):
                try:
                    super().__init__()
                except TypeError:
                    client.TestPerformer.__init__(self)
                self.retry_interval = 5
                self.tank_attempts = 3

            def choose_available_tank(self, candidates, **kwargs):
                for _ in range(self.tank_attempts):
                    # noinspection PyAttributeOutsideInit
                    self.api_clients = self.check_available(candidates)
                    if not self.api_clients:
                        logger.debug('Retry tanks check in %s sec...', self.retry_interval)
                        time.sleep(self.retry_interval)
                    else:
                        break

            @staticmethod
            def check_available(candidates):
                available = []
                for candidate in candidates:
                    logger.debug('Check tank %s...', candidate.host)
                    try:
                        res = candidate.get_status()
                    except Exception:
                        logger.debug('Failed to get tank status', exc_info=True)
                        res = {'success': False}
                    if res['success'] and not res['is_testing']:
                        logger.debug('%s is available for test', candidate.host)
                        available.append(candidate)
                        break
                    logger.debug('Status of %s:\n%s', candidate.host, res)
                logger.info('Available tanks: %s', available)
                return available

        self._save(target_status=self.status, config=self.config_manager.config, section='init_shooting')
        performer = TestInitiator()

        performer.api_clients = performer.check_available(self.tanks)
        if not performer.api_clients:
            logger.info('No available tanks found, failed to start')
            tanks = ', '.join([tank.host for tank in self.tanks])
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START,
                error='Tanks {} are unavailable or busy. Try shooting later, check the correctness and availability of tanks. '.format(
                    tanks
                ),
                section='init_shooting',
            )

        self.config_manager.set_tank(performer.api_clients[0].host, performer.api_clients[0].port)
        self.tank_version = str(
            get_tank_version('{}:{}'.format(performer.api_clients[0].host, performer.api_clients[0].port))
        )

        if self.monitoring_config:
            with tempfile.NamedTemporaryFile(delete=False) as monitoring_file:
                monitoring_file.write(self.monitoring_config.encode('utf-8'))
                files = {monitoring_file.name: monitoring_file}
            self.config_manager.set_monitoring_path(monitoring_file.name.split('/')[-1])
        else:
            files = None

        with tempfile.NamedTemporaryFile(delete=False) as config_file:
            config_file.write(json.dumps(self.config_manager.config).encode('utf-8'))

        performer.start_tests(
            config=[config_file.name], ammo=None, files=files, jobno_file=None, cli_options=None, cfg_patches=None
        )

        for api_client in performer.api_clients:
            if api_client.test_id:
                self.status = InternalStatus.INITIATED
                self.test.started = True
                self.test.tank = api_client
                self.test.tank.wait_exponential_max = WAIT_EXPONENTIAL_MAX
                self.test.tank.stop_max_delay = STOP_MAX_DELAY
                logger.info('Test started on %s', self.test.tank)
                break
        if not self.test.started:
            raise FirestarterError(
                status=InternalStatus.FAILED_TO_START, error='Failed to start test on the tank', section='init_shooting'
            )

        os.unlink(config_file.name)
        if self.monitoring_config:
            os.unlink(monitoring_file.name)

        self._save(self.status)

    #########################################
    #          CHECK TEST STATUS            #
    #########################################

    def check_test_status(self):
        logger = logging.getLogger('CHECK TEST STATUS')
        logger.info('Check status for task %s', self.task_id)
        lunapark_id, test_status, errors = self.test.check_status()

        if test_status == 'FINISHED' and not lunapark_id:
            logger.info(
                '[CHECK TEST STATUS] No lunapark id found for task %s. Test %s on tank %s',
                self.task_id,
                self.test.tank.test_id,
                self.test.tank.host,
            )
            errors = (
                errors
                or 'Some problems with test on tank {}. Results were not uploaded.\n'
                'Local test id {}'.format(self.test.tank.host, self.test.tank.test_id)
            )

        if errors:
            logger.error('Shooting failed. %s', errors)
            raise FirestarterError(
                status=InternalStatus.FAILED,
                error='Some error happend during shooting.\nSee the logs for detailed information.',
                section='tankapi',
            )
        elif test_status:
            self.status = test_status.lower()
            self._save(target_status=self.status, section='check_status')
            self.test_id = lunapark_id

        logger.info('Task %s status %s', self.task_id, self.status)


class TankTest:
    """
    Can be used for checking test status in case of sandbox tank.
    Now it's enough to get lunapark id on test start
    """

    def __init__(self):
        from load.projects.tankapi_cmd.src import client

        self.started = False
        self.tank = client.LunaparkTank()

    def check_status(self):
        logger = logging.getLogger('TANK TEST')
        current_status = {}
        try:
            current_status = self.tank.get_test_status(self.tank.test_id)
        except (IOError, OSError):
            logger.warning('Impossible to connect to the tank %s', self.tank.host)
            logger.debug('Exception when trying to get test status: ', exc_info=True)
            current_status.update({'status_code': 'FAILED'})
        logger.info('Current status %s', current_status)
        lunapark_id = current_status.get('lunapark_id') or 0
        test_status = current_status.get('status_code')
        # TODO: https://st.yandex-team.ru/YANDEXTANK-328
        errors = current_status.get('error') or current_status.get('tank_msg')
        return lunapark_id, test_status, errors

    def get_tank_connection_string(self):
        return '{host}:{port}'.format(host=self.tank.host, port=self.tank.port)
