import os
import tempfile
import logging
import requests

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.sdk2.helpers import ProcessLog
from sandbox.common.types import misc as ctm
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.AddrsDeploy.tools.misc import retry
from sandbox.projects.tank.load_resources.resources import STRESS_SESSION_IDS

TTL = 30


class GenerateStressSessionIds(sdk2.Task):
    """
        Generates session ids for load tests
        Broken! Must be fixed! https://st.yandex-team.ru/LOAD-1136

    """

    tvm2_secret = ''
    ticket = ''
    temp_dir = None

    class Requirements(sdk2.Requirements):
        disk_space = 512
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        _container = sdk2.parameters.Container(
            "tvm2 environment container",
            default_value=513532842,
            required=True
        )
        blackbox_host = sdk2.parameters.String('Blackbox host that will be used for sessions generation',
                                               default='blackbox-stress.yandex.net')
        tvm_client_id = sdk2.parameters.String(
            'Client id for tvm2',
            default='2000158',
            required=True
        )
        tvm_secret_token = sdk2.parameters.String(
            'Secret for tvm2 in vault',
            default='robot-lunapark-tvm2-stressids-secret',
            required=True
        )
        vault_owner = sdk2.parameters.String(
            'Vault item owner',
            required=False
        )
        uids = sdk2.parameters.String(
            'List of uids, separated by newline',
            required=True,
            multiline=True
        )
        result_label = sdk2.parameters.String('Label for resource with results')

    @retry(tries=3, delay=5)
    def _request_to_blackbox(self, ticket, user_id):
        create_session_url = '&'.join(
            [
                'http://{blackbox_host}/blackbox?method=createsession',
                'is_stress=1',
                'uid={uid}',
                'host=pass-stress.sezam.yandex.ru',
                'keyspace=yandex.ru',
                'ttl=5',
                'userip=1.1.1.1',
                'format=json',
            ]
        )
        headers = {
            'X-Ya-Service-Ticket': ticket,
            'Host': 'blackbox.yandex.net',
        }
        return requests.get(
            create_session_url.format(blackbox_host=self.Parameters.blackbox_host, uid=user_id),
            headers=headers, timeout=10)

    def get_uid(self, ticket, user_id):
        try:
            result = self._request_to_blackbox(ticket, user_id)
        except Exception as exc:
            logging.info('Request for uid %s failed with %s', user_id, exc)
            return None

        if result.ok:
            return result.content
        else:
            logging.info('Request for uid %s failed with %s %s %s',
                         user_id, result.status_code, result.reason, result.text)
            return None

    def modify_script(self):
        initial_script = os.path.join(os.path.dirname(__file__), 'get_tvm_ticket.py')
        self.temp_dir = tempfile.mkdtemp()
        python_script = os.path.join(self.temp_dir, 'get_ticket.py')

        with open(initial_script, 'r') as initial:
            script = initial.read()
        script = script.replace('client_id = ""', 'client_id = "{}"'.format(self.Parameters.tvm_client_id))
        script = script.replace('secret = ""', 'secret = "{}"'.format(self.tvm2_secret))
        with open(python_script, 'w') as output:
            output.write(script)

    def get_ticket(self):

        cmd = '. /usr/local/vennv/bin/activate && python ./{}'.format('get_ticket.py')

        with ProcessLog(self, 'script') as pl:
            try:
                self.ticket = sp.check_output(cmd, stderr=pl.stdout, shell=True, cwd=self.temp_dir)
            except Exception:
                raise TaskError('No service ticket, check script.out in logs')
            logging.info('Obtained ticket %s', self.ticket)

    def save_resource(self, data):
        cookies_file = STRESS_SESSION_IDS(
            self,
            'Stress sessions ids', 'stress_session_ids.list',
            ttl=TTL, label=self.Parameters.result_label)
        with open(str(cookies_file.path), 'w') as output:
            output.write(data)
        resource_data = sdk2.ResourceData(cookies_file)
        resource_data.ready()

    def on_execute(self):
        # get vault data
        vault_owner = self.Parameters.vault_owner if self.Parameters.vault_owner else self.owner
        self.tvm2_secret = sdk2.Vault.data(vault_owner, self.Parameters.tvm_secret_token)

        uids = self.Parameters.uids.split('\n')
        uids = filter(None, uids)

        self.modify_script()
        self.get_ticket()

        # noinspection PyUnboundLocalVariable
        cookies_data = [self.get_uid(self.ticket, uid) for uid in uids]
        cookies = ''.join(filter(None, cookies_data))
        logging.info('Cookies are %s', cookies)

        self.save_resource(cookies)
