# coding: utf-8
import logging
import sandbox.common.errors as ce
import sandbox.common.types.notification as ctn

from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary
from .loadtest_report import ReportLoadTestResults


def shoot_types_radio_group(*args, **kwargs):
    with sdk2.parameters.RadioGroup(*args, **kwargs) as setter:
        setter.values['const'] = setter.Value(value='Constant load shooting', default=True)
        setter.values['line'] = setter.Value(value='Line load shooting with imbalance point')
        setter.values['simple_const_link'] = setter.Value(
            value='Constant load shooting, only http link to lunapark results')
        setter.values['simple_line_link'] = setter.Value(
            value='Line load shooting, only http link to lunapark results')
    return setter


class LoadTestResults(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 512
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 10

        only_regression = sdk2.parameters.Bool(
            'Report only failed KPIs in regression?',
            description='Reports will be sent only if regression KPI failed',
            default=False
        )
        shoot_id = sdk2.parameters.Integer(
            'Lunapark test id',
            description='Lunapark test id (only number, i.e. 1633829)',
            required=True
        )

        report_type = shoot_types_radio_group(
            'Type of report',
            description='Report content depends on shoot type',
            required=True
        )

        additional_message = sdk2.parameters.String(
            'Message that will be added to the report',
            multiline=True
        )

        send_comment = sdk2.parameters.Bool(
            'Send comment to Startrek',
            description='Send comment with test report to Startrek ticket',
            default=False
        )
        with send_comment.value[True]:
            ticket_id = sdk2.parameters.String(
                'ST ticket',
                description='Startrek ticket where comment will be added, i.e. LOAD-318',
                required=False
            )
            st_token_name = sdk2.parameters.String(
                'Vault record name with startrek token',
                description='Vault name of record with startrek token',
                default='lunapark-startrek-token',
                required=False
            )

        send_letter = sdk2.parameters.Bool(
            'Send e-mail',
            description='Send letter with test report',
            default=False
        )
        with send_letter.value[True]:
            mail_recipients = sdk2.parameters.List(
                'Mail to',
                sdk2.parameters.String('email', description='Emails where report will be sent to'),
                required=False
            )

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(
            attrs={
                'task_name': 'LoadTestResults',
                'released': 'stable'
            }).first()

    def on_enqueue(self):
        # check parameters
        if self.Parameters.send_comment \
                and not self.Parameters.ticket_id \
                and not self.Parameters.st_token_name:
            raise ce.TaskError('No startrek ticket found')
        if self.Parameters.send_letter \
                and not self.Parameters.mail_recipients:
            raise ce.TaskError('No mail recipients are found')

    def on_execute(self):

        message = self.build_message()
        if message:
            logging.info('Message is %s', message)
            if self.Parameters.send_comment:
                from startrek_client import Startrek

                st_client = Startrek(
                    token=sdk2.Vault.data(self.Parameters.st_token_name),
                    useragent='load-test-results'
                )
                issue = st_client.issues[self.Parameters.ticket_id]
                comment = issue.comments.create(text=message)

            if self.Parameters.send_letter:
                self.server.notification(
                    subject='Sandbox notification for '
                            'https://lunapark.yandex-team.ru/{}'.format(self.Parameters.shoot_id),
                    body=message,
                    recipients=self.Parameters.mail_recipients,
                    transport=ctn.Transport.EMAIL
                )

    def build_message(self):
        from sandbox.projects.tank.lunapark_client.settings import LUNAPARK_URL

        message = u''

        if 'simple' in self.Parameters.report_type:
            if 'const' in self.Parameters.report_type:
                message = u'Стрельба постоянной нагрузкой: https://lunapark.yandex-team.ru/{}\n'.format(
                    self.Parameters.shoot_id)
            elif 'line' in self.Parameters.report_type:
                message = u'Стрельба на разладку: https://lunapark.yandex-team.ru/{}\n'.format(
                    self.Parameters.shoot_id)
            if self.Parameters.additional_message:
                message += u'\n{}'.format(self.Parameters.additional_message)
            return message

        report = ReportLoadTestResults(self.Parameters.shoot_id, self.Parameters.report_type)
        report.get_summary()

        if self.Parameters.only_regression:
            if report.check_resolution():
                logging.info('SLA not broken for %s/%s', LUNAPARK_URL, self.Parameters.shoot_id)
            else:
                message = u'Стрельба {}/{}\n{}'.format(LUNAPARK_URL, self.Parameters.shoot_id, report.message)
        else:
            report.build_message()
            message = report.message
            if self.Parameters.additional_message:
                message += u'\n{}'.format(self.Parameters.additional_message)

        return message
