# coding: utf-8
import logging
import os
import requests
import shutil
import time
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import sandbox.common.config as common_config
import sandbox.projects.common.network as network

from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.tank.load_resources.resources import YANDEX_TANK_LXC_CONTAINER


TANK_STATUS_URL = 'http://[{}]:{}/api/v1/tank/status.json'
TANKAPI_FOLDER = '/var/lib/tankapi/tests'
TANKAPI_PORT = 8083
YAV_LUNAPARK_ID = 'sec-01drefbx0tptqmcamkt7q01trj'
OAUTH_TOKEN = 'lunapark-staff-oauth-token'
TOKEN_FILE = '/home/lunapark/.token'

class SandboxTank(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 1024 * 8
        disk_space = 1024 * 50
        privileged = True
        client_tags = ctc.Tag.SAS

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Context(sdk2.Task.Context):
        tank_dc = ''
        tank_ip = ''
        tank_restarts = 0
        artefacts_folder = ''
        tank_started = 0
        preparing_started = 0
        testing = False

    class Parameters(sdk2.Task.Parameters):
        description = 'Yandex Tank as sandbox task'
        kill_timeout = 60 * 60 * 3
        wait_timeout = 30 * 60

        tankapi_port = sdk2.parameters.Integer('Tankapi port', default_value=TANKAPI_PORT)
        container = sdk2.parameters.Container(
            'LXC Container with tank',
            resource_type=YANDEX_TANK_LXC_CONTAINER,
            required=True
        )

        with sdk2.parameters.Output:
            tank_ip = sdk2.parameters.String('Tank IP address')
            exit_code = sdk2.parameters.Integer('Exit code')

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(
            attrs={
                'task_name': 'SandboxTank',
                'released': 'stable'
            }).first()

    def on_create(self):
        self.Parameters.container = YANDEX_TANK_LXC_CONTAINER.find(
            state=ctr.State.READY,
            attrs={
                'purpose': 'Firestarter',
                'released': 'stable'
            }
        ).order(-YANDEX_TANK_LXC_CONTAINER.id).first().id

    def on_execute(self):
        if self.Parameters.container:
            run_tank(self.Parameters.tankapi_port)
            self.Context.tank_started = time.time()
        else:
            logging.error('No LXC container with the tank was found!')
            self.finalize(self.status, ctt.TaskStatus.FAILURE)

        get_token(YAV_LUNAPARK_ID, OAUTH_TOKEN, TOKEN_FILE)

        self.Context.tank_dc = common_config.Registry().this.dc
        self.Context.tank_ip = network.get_my_ipv6()
        self.Context.save()
        logging.info('Local IP: %s' % str(self.Context.tank_ip))
        self.Parameters.tank_ip = self.Context.tank_ip
        while self.test_is_stil_running():
            self.Context.save()
            time.sleep(5)

    def on_break(self, prev_status, status=ctt.TaskStatus.FAILURE):
        self.finalize(prev_status, status)

    def on_failure(self, prev_status, status=ctt.TaskStatus.FAILURE):
        self.finalize(prev_status, status)

    def on_timeout(self, prev_status, status=ctt.TaskStatus.FAILURE):
        logging.error('Shooting too long!')
        self.finalize(prev_status, status)

    def finalize(self, prev_status, status):
        logging.debug('Finalize from %s to %s', prev_status, status)
        self.save_artefacts()
        self.Parameters.exit_code = 1 if status == ctt.TaskStatus.FAILURE else 0
        super(SandboxTank, self).on_finish(prev_status, status)

    def test_is_stil_running(self):
        tank_status = self.get_tank_status()
        if isinstance(tank_status, tuple):
            preparing_status, testing_status, current_test = tank_status

            if preparing_status and not self.Context.preparing_started:
                self.Context.preparing_started = time.time()
     
            if preparing_status and self.Context.preparing_started:
                self.check_timeout(self.Context.preparing_started)

            if testing_status and not self.Context.testing:
                self.Context.testing = True
                logging.info('Shooting started')

            if current_test and not self.Context.artefacts_folder:
                self.Context.artefacts_folder = '{}/{}'.format(TANKAPI_FOLDER, current_test)
        
            if not testing_status:
                if self.Context.testing:
                    self.Context.testing = False
                    self.save_artefacts()
                    return False
                else:
                    self.check_timeout(self.Context.tank_started)
        else:
            run_tank(self.Parameters.tankapi_port)
            self.Context.tank_restarts += 1
            logging.warning('Error when getting the tank status! Tankapi was restart %d times', self.Context.tank_restarts)
            if self.Context.tank_restarts > 10:
                raise TaskFailure('Tank is not responding!')

        return True

    def check_timeout(self, starttime):
        if time.time() - starttime > self.Parameters.wait_timeout:
            raise TaskFailure('The shooting does not started or failed')

    def save_artefacts(self):
        try:
            for root, _, artefacts in os.walk(self.Context.artefacts_folder):
                for artefact in artefacts:
                    shutil.copyfile(os.path.join(root, artefact), os.path.join(os.getcwd(), 'log1', artefact))
        except Exception:
            logging.error('Can not create resource!', exc_info=True)

    def get_tank_status(self):
        try:
            result = requests.get(TANK_STATUS_URL.format(self.Context.tank_ip, self.Parameters.tankapi_port)).json()
            return (result['is_preparing'], result['is_testing'], result['current_test'])
        except (requests.exceptions.RequestException, IndexError, TypeError):
            logging.exception('Failed to get tank status')


def run_tank(port):
    try:
        os.popen('/usr/bin/tankapi --ipv6 --port={}'.format(port))
        logging.info('Tankapi was started on the port %d.', port)
    except OSError:
        logging.warning('Failed to run tankapi on the port %d!', port, exc_info=True)


def get_token(secret, key, file):
    try:
        dirname = os.path.dirname(file)
        if not os.path.exists(dirname):
            os.makedirs(dirname)
        with open(file, 'w') as token:
            secrets = sdk2.yav.Secret(secret)
            oauth_token = secrets.data(True)[key]
            token.write('OAuth {token}'.format(token=oauth_token))
        logging.info('[GET TOKEN] Token is saved')
    except Exception:
        logging.exception('[GET TOKEN] Failed to create token file %s', file)
