# coding: utf-8

import glob
import logging
import os
import random
import tarfile
import time

import requests
import sandbox.common.errors as ce
import sandbox.common.types.notification as ctn
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.sandboxsdk.ssh import Key
from sandbox.projects.common import file_utils as fu
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.load_resources.resources import YANDEX_TANK_LOGS
from sandbox.projects.tank.Firestarter.external_calls import retry
from .detect_tank import TankFinder, Host


DEFAULT_TANK_CONFIG = """phantom:
  load_profile: {load_type: rps, schedule: 'line(1,10,1m) const(10,1m)'}
  address: 'yandex.ru:443'
  ssl: true
  writelog: proto_warning
uploader:
  operator: lunapark
  task: LOAD-318
  job_name:
  job_dsc: 'started from Sandbox'
  meta: {jenkinsbuild: 'http://sandbox.yandex-team.ru/task/{}/view'}
"""

DEFAULT_TANK_INI = """[phantom]
ammofile=
rps_schedule=
uris=
address=
[meta]
operator=lunapark
task=
ver=
job_dsc=started from sandbox with ShootViaTankapi
job_name=
"""

DEFAULT_MONITORING = """<Monitoring>
    <Host address="[target]">
        <CPU measure="idle,user,system,iowait" />
        <System measure="la1,csw,int,numproc,numthreads" />
        <Memory measure="free,used,cached,buff" />
        <Disk measure="read,write"/>
        <Net measure="recv,send,retransmit,estab,closewait,timewait"/>
    </Host>
    <Host address="localhost">
        <CPU measure="idle,user,system,iowait" />
        <System measure="la1,csw,int,numproc,numthreads" />
        <Memory measure="free,used,cached,buff" />
        <Disk measure="read,write"/>
        <Net measure="recv,send,retransmit,estab,closewait,timewait"/>
    </Host>
</Monitoring>
"""

AUTOSTOP_CODES = {
    '21': 'average response time',
    '22': 'http codes',
    '23': 'network codes',
    '24': 'instance count',
    '25': 'total time codes',
    '26': 'total http codes',
    '27': 'total net codes',
    '28': 'negative http codes',
    '29': 'negative net codes',
    '30': 'http trend',
    '31': 'monitored metric',
    '32': 'monitored metric',
    '33': 'cumulative percentiles',
}

WRONG_RC_MESSAGE = "Tankapi returned exit code {}, check tankapi-cmd.out"


def split_socket(address):
    """
        supported address inputs:
        'ipv4',
        'ipv4:port',
        '[ipv4]:port',
        'ipv6',
        '[ipv6]:port',
        'host',
        'host:port',
        '[host]:port'
    """
    try:
        assert address
        ipv6 = len(address.split(':')) > 2
        if address.find(']:') != -1:
            # [ipv4]:port
            # [ipv6]:port
            # [host]:port
            host = ':'.join(address.split(':')[:-1]).replace('[', '').replace(']', '')
            port = int(address.split(':')[-1])
        elif address.find(':') != -1 and not ipv6:
            # ipv4:port
            # host:port
            host = address.split(':')[0].replace('[', '').replace(']', '')
            port = int(address.split(':')[-1])
        else:
            # ipv4
            # ipv6
            # host
            host = address
            port = 8083
    except Exception:
        logging.exception('')
        raise ValueError
    return host, port


def link_format(href, text, description=''):
    return '{}<a href="{}">{}</a>'.format(description, href, text)


class FailedCloneGit(Exception):
    pass


class ShootViaTankapi(sdk2.Task):
    """
        Task for shooting via tankapi-cmd
    """

    local_repo = ''
    local_configs = ''
    tank_config_path = ''
    monitoring_file = ''
    ammo_file = ''
    tankapi_cmd = ''

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        description = u'Стрельба через танкапи'
        kill_timeout = 60 * 60 * 3

        container = sdk2.parameters.Container(
            'LXC Container with tankapi',
            resource_type=YANDEX_TANKAPI_LXC_CONTAINER,
            required=True
        )

        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            nanny_service = sdk2.parameters.String('Target nanny service name',
                                                   description='Nanny service name where your target is deployed, '
                                                               'for example maps_core_meta_testing. Random container '
                                                               'will be taken as a target. NB! This target overrides '
                                                               'phantom.address set in tank config!')
            override_nanny_port = sdk2.parameters.String('Target nanny service port',
                                                         description='Port for target if nanny service port '
                                                                     'is not used, usually 80. If empty - port from '
                                                                     'nanny service will be used.')
            use_public_tanks = sdk2.parameters.Bool('Use public tanks',
                                                    description='Use public tanks from ALL_RCLOUD_TANKS. First '
                                                                'available tank from the same DC will be used. NB! '
                                                                'Works only if target nanny service name is defined!',
                                                    default=True)
            tanks = sdk2.parameters.List('List of tanks', default=[],
                                         description='One or several tanks in the same DC. The first tank ready '
                                                     'to shoot will be used. Format: lee.tanks.yandex.net:8083. '
                                                     'Also you can provide user nanny service with tanks. '
                                                     'Format: nanny:my_tanks_service '
                                                     '(overrides use_public_tanks parameter)')
            logs_to_save = sdk2.parameters.List('Logs from tank to save after shooting',
                                                default=["answ*.log", "tank*.log", "jobno*.txt"],
                                                description='List of tank logs which will be saved as '
                                                            'YANDEX_TANK_LOGS resource')
            timeout = sdk2.parameters.Integer('Timeout to tank, seconds', default=None, required=False,
                                              description='Increase this if requests to tank are timed out')
            check_tankapi_rc = sdk2.parameters.Bool('Check the return code of the tankapi-cmd?',
                                                    description='Use this field if you want to fail the task if tankapi-cmd fails',
                                                    default=False)

        with sdk2.parameters.Group('Tank config parameters') as tank_config_block:
            with sdk2.parameters.RadioGroup('Tank config source', required=True) as config_source:
                config_source.values['url'] = config_source.Value(value='HTTP url', default=True)
                config_source.values['resource'] = config_source.Value(value='Sandbox resource')
                config_source.values['file'] = config_source.Value(value='Upload config content')
                config_source.values['github'] = config_source.Value(value='Config from github')
                config_source.values['arcadia'] = config_source.Value(value='Config from Arcadia')
                with config_source.value['url']:
                    config_url = sdk2.parameters.Url('Tank config url')
                with config_source.value['resource']:
                    config_resource = sdk2.parameters.LastReleasedResource('Resource with tank config',
                                                                           state=ctr.State.READY)
                with config_source.value['file']:
                    config_content = sdk2.parameters.String('Tank config contents', multiline=True,
                                                            default_value=DEFAULT_TANK_CONFIG)
                with config_source.value['github']:
                    config_git_path = sdk2.parameters.String(
                        'Path to tank config',
                        description='Path to tank config file relative to the root '
                                    'of repo, for example /load/configs/set1_load.ini')
                with config_source.value['arcadia']:
                    config_arc_path = sdk2.parameters.String(
                        "Path to tank config relative to the root",
                        description='for example sandbox/projects/tank/szypulka_test_shoot/configs/load_const.yaml')
                config_add_parameters = sdk2.parameters.String(
                    'Additional tankapi parameters',
                    description='Will be added to cmd, override options set in tank config. '
                                'Format: -o phantom.writelog=all -o autostop.autostop=http(5xx,10%,1m)')
        with sdk2.parameters.Group('Monitoring config parameters') as monitoring_block:
            use_monitoring = sdk2.parameters.Bool('Collect host metrics other then default?',
                                                  description='Add config for metrics collection during the shoot')
            with use_monitoring.value[True]:
                with sdk2.parameters.RadioGroup('Monitoring config source', required=True) as monitoring_source:
                    monitoring_source.values['in_config'] = monitoring_source.Value(value='Already set in tank config',
                                                                                    default=True)
                    monitoring_source.values['url'] = monitoring_source.Value(value='HTTP url')
                    monitoring_source.values['resource'] = monitoring_source.Value(value='Sandbox resource')
                    monitoring_source.values['file'] = monitoring_source.Value(value='Upload config content')
                    monitoring_source.values['github'] = monitoring_source.Value(value='Config from github')
                    monitoring_source.values['arcadia'] = monitoring_source.Value(value='Config from Arcadia')
                with monitoring_source.value['url']:
                    monitoring_url = sdk2.parameters.Url('Monitoring config url',
                                                         description='Should be available from tank host')
                with monitoring_source.value['resource']:
                    monitoring_resource = sdk2.parameters.LastReleasedResource('Resource with monitoring config',
                                                                               state=ctr.State.READY)
                with monitoring_source.value['file']:
                    monitoring_content = sdk2.parameters.String('Monitoring config contents', multiline=True,
                                                                default_value=DEFAULT_MONITORING)
                with monitoring_source.value['github']:
                    monitoring_git_path = sdk2.parameters.String(
                        'Path to monitoring file',
                        description='Path to monitoring config relative to the root of repo, '
                                    'for example /load/configs/monitoring.xml')
                with monitoring_source.value['arcadia']:
                    monitoring_arc_path = sdk2.parameters.String(
                        'Path to monitoring config relative to the root',
                        description='for example  sandbox/projects/tank/szypulka_test_shoot/configs/monitoring.xml')
        with sdk2.parameters.Group('Ammo parameters') as ammo_block:
            with sdk2.parameters.RadioGroup('Ammo source', required=True) as ammo_source:
                ammo_source.values['in_config'] = ammo_source.Value(value='Already set in tank config',
                                                                    default=True)
                ammo_source.values['url'] = ammo_source.Value(value='HTTP url')
                ammo_source.values['resource'] = ammo_source.Value(value='Sandbox resource')
                ammo_source.values['github'] = ammo_source.Value(value='Ammo from github')
                ammo_source.values['arcadia'] = ammo_source.Value(value='Ammo from Arcadia')
            with ammo_source.value['url']:
                ammo_source_url = sdk2.parameters.Url('Ammo url', description='Should be avalaible from tank host')
            with ammo_source.value['resource']:
                ammo_resource = sdk2.parameters.Resource('Resource with ammo', state=ctr.State.READY)
            with ammo_source.value['github']:
                ammo_git_path = sdk2.parameters.String('Path to ammo',
                                                       description='Path to ammo relative to the root of repo, '
                                                                   'for example /load/ammo/urls_list.1')
            with ammo_source.value['arcadia']:
                ammo_arc_path = sdk2.parameters.String('Path to ammo')
        with sdk2.parameters.Group('Git parameters') as git_config_block:
            sdk2.parameters.Info('Required if at least one config is taken from git')
            ssh_vault = sdk2.parameters.String('Vault item name with ssh key for git access')
            vault_owner = sdk2.parameters.String('Vault item owner')
            git_repository = sdk2.parameters.Url('Link to git repo with config',
                                                 description='For example '
                                                             'https://github.yandex-team.ru/media/tickets.git')
            git_branch = sdk2.parameters.String('Branch name', default='master')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    class Context(sdk2.Task.Context):
        tanks = []
        tankapi_port = ''
        target_host = ''
        ammo_option = ''
        monitoring_option = ''
        tankapi_exit_code = 0
        tank_config_file = ''

    @staticmethod
    def _gzip_directory(source, destination):
        with tarfile.open(destination, "w:gz") as tar_archive:
            for item in os.listdir(source):
                tar_archive.add(os.path.join(source, item), arcname=item)

    def _notify_on_task_problems(self, reason='', task_fixer='szypulka'):
        self.server.notification(
            subject='[shoot_via_tankapi] {}'.format(reason),
            body='{}\nhttps://sandbox.yandex-team.ru/task/{}'.format(reason, self.id),
            recipients=task_fixer,
            transport=ctn.Transport.EMAIL
        )

    def on_create(self):
        try:
            self.Parameters.container = YANDEX_TANKAPI_LXC_CONTAINER.find(
                state=ctr.State.READY,
                owner='LOAD-ROBOT',
            ).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
            logging.info('Container %s is taken', self.Parameters.container.id)
        except AttributeError:
            logging.info('No LXC container with tankapi is found, please select it manually.')

    @retry(tries=5, delay=2)
    def _clone_repository(self):
        self.local_repo = str(self.path('repo').as_posix())
        clone_cmd = 'git clone --progress -b {} {} {}'.format(self.Parameters.git_branch,
                                                              self.Parameters.git_repository, self.local_repo)
        with ProcessLog(self, logging.getLogger('git_clone')) as process_log:
            status = sp.Popen(
                clone_cmd,
                shell=True,
                stdout=process_log.stdout,
                stderr=process_log.stderr
            ).wait()
            if status != 0:
                raise FailedCloneGit('Failed to clone git repo')

        logging.info('Repository %s:%s cloned to %s', self.Parameters.git_repository, self.Parameters.git_branch,
                     self.local_repo)

    def _define_target(self):

        tanks_nanny_services = None
        if all(tank.startswith('nanny:') for tank in self.Parameters.tanks):
            tanks_nanny_services = [service.lstrip('nanny:') for service in self.Parameters.tanks]
        tank_finder = TankFinder(target_nanny_service=self.Parameters.nanny_service,
                                 tanks_nanny_services=tanks_nanny_services)
        tank_finder.collect_info()
        if tank_finder.targets:
            random_target = random.choice(tank_finder.targets)
            port = self.Parameters.override_nanny_port if self.Parameters.override_nanny_port else random_target.port
            self.Context.target_host = '{}:{}'.format(random_target.host, port)
        else:
            # TODO: add definition of random target not only from Nanny LOAD-688
            random_target = Host('', '', '')
        if self.Parameters.use_public_tanks or tanks_nanny_services:
            try:
                self.Context.tanks = [tank.host for tank in tank_finder.tanks if random_target.dc in tank.dc]
                self.Context.tankapi_port = set([tank.port for tank in tank_finder.tanks]).pop()
            except IndexError:
                raise ce.TaskError('Public tanks are not found, check common.log for errors')

    def _set_tanks_list(self):
        """
            Defines the list of suitable tanks.
            Should be replaced with lpq when it's ready (https://st.yandex-team.ru/LUNAPARK-3212)
        """
        for tank_param in self.Parameters.tanks:
            try:
                tank, self.Context.tankapi_port = split_socket(tank_param)
                self.Context.tanks.append(tank)
            except ValueError:
                pass

    def _set_tank_config(self):
        """
        Tank config is saved as a file for tankapi call and as a resource for logging purposes
        """
        local_config_path = self.local_configs + '/tank.config'

        if self.Parameters.config_source == 'file':
            if not self.Parameters.config_content:
                raise ce.TaskError('Empty tank config')
            fu.write_file(local_config_path, self.Parameters.config_content)

        if self.Parameters.config_source == 'url':
            try:
                resp = requests.get(self.Parameters.config_url)
                config_contents = resp.text
                logging.debug('Got config file by url, content is %s', config_contents)
            except requests.exceptions.RequestException:
                logging.exception('Failed to get file')
                raise ce.TaskError('Failed to get tank config by url {}'.format(self.Parameters.config_url))
            if not config_contents:
                raise ce.TaskError('Empty tank config')
            fu.write_file(local_config_path, config_contents)

        if self.Parameters.config_source == 'github':
            # noinspection PyTypeChecker
            local_config_path = self.local_repo + self.Parameters.config_git_path

        if self.Parameters.config_source == 'resource':
            tank_config = self.Parameters.config_resource
            tank_config_data = sdk2.ResourceData(tank_config)
            local_config_path = str(tank_config_data.path)

        logging.info('Tank config: %s', fu.read_lines(local_config_path))
        return local_config_path

    def _set_ammo(self):
        """
        Ammo can be given as a file (added to config and transferred to tank host),
        as an url (added to config)
        or can be already included in config
        """
        if self.Parameters.ammo_source == 'in_config':
            return ''
        if self.Parameters.ammo_source == 'url':
            return '-o phantom.ammofile={}'.format(self.Parameters.ammo_source_url)
        if self.Parameters.ammo_source == 'resource':
            # Tank can take only http links now,
            # so the ammo resource should be accessible via http proxy
            ammo_resource = self.Parameters.ammo_resource
            if not ammo_resource.http_proxy:
                raise ce.InvalidResource('Ammo resource has no http_proxy link')
            return '-o phantom.ammofile={}'.format(ammo_resource.http_proxy)
        if self.Parameters.ammo_source == 'github':
            # noinspection PyTypeChecker
            ammo_file = self.local_repo + self.Parameters.ammo_git_path
            return '-o phantom.ammofile={} -f {}'.format(os.path.basename(ammo_file), ammo_file)
        if self.Parameters.ammo_source == 'arcadia':
            ammo_file = self.local_configs + '/ammo'
            return '-o phantom.ammofile={} -f {}'.format(os.path.basename(ammo_file), ammo_file)

    def _set_monitoring(self):
        """
        Monitoring can be already included in config (as a list of monitoring/telegraf directives),
        as a link to external file or sandbox resource (then it's passed as tankapi option and
            file is transferred to tank),
        or as an input parameter (then it's saved to file, passed as tankapi option and
            file is transferred to tank)
        """
        self.monitoring_file = self.local_configs + '/monitoring.config'

        if self.Parameters.monitoring_source == 'in_config':
            return ''

        if self.Parameters.monitoring_source == 'resource':
            monitoring_config = self.Parameters.monitoring_resource
            monitoring_config_data = sdk2.ResourceData(monitoring_config)
            self.monitoring_file = str(monitoring_config_data.path)
            logging.info('Monitoring config: %s', fu.read_lines(self.monitoring_file))
            return '-o telegraf.config={} -f {}'.format(os.path.basename(self.monitoring_file),
                                                        self.monitoring_file)

        if self.Parameters.monitoring_source == 'file':
            if not self.Parameters.monitoring_content:
                raise ce.TaskError('Empty monitoring config')
            fu.write_file(self.monitoring_file, self.Parameters.monitoring_content)
            logging.info('Monitoring config: %s', fu.read_lines(self.monitoring_file))
            return '-o telegraf.config={} -f {}'.format(os.path.basename(self.monitoring_file),
                                                        self.monitoring_file)

        if self.Parameters.monitoring_source == 'url':
            try:
                resp = requests.get(self.Parameters.monitoring_url)
                config_contents = resp.text
            except requests.exceptions.RequestException:
                raise ce.TaskError('Failed to get monitoring config by url {}'.format(
                    self.Parameters.monitoring_url))
            if not config_contents:
                raise ce.TaskError('Empty monitoring config')
            fu.write_file(self.monitoring_file, config_contents)
            logging.info('Monitoring config: %s', fu.read_lines(self.monitoring_file))
            return '-o telegraf.config={} -f {}'.format(os.path.basename(self.monitoring_file),
                                                        self.monitoring_file)

        if self.Parameters.monitoring_source == 'github':
            # noinspection PyTypeChecker
            self.monitoring_file = self.local_repo + self.Parameters.monitoring_git_path
            logging.info('Monitoring config: %s', fu.read_lines(self.monitoring_file))
            return '-o telegraf.config={} -f {}'.format(os.path.basename(self.monitoring_file),
                                                        self.monitoring_file)

        if self.Parameters.monitoring_source == 'arcadia':
            return '-o telegraf.config={} -f {}'.format(os.path.basename(self.monitoring_file),
                                                        self.monitoring_file)

    def _set_tankapi_cmd(self):
        tank_opts = []
        for tank in self.Context.tanks:
            tank_opts.extend(['-t', tank])
        tank_opts.extend(['-p', str(self.Context.tankapi_port)])
        tank_opts.extend(['-o', 'meta.use_tank_port={}'.format(self.Context.tankapi_port)])
        if self.Context.target_host:
            tank_opts.extend(['-o', 'phantom.address={}'.format(self.Context.target_host)])
        for log in self.Parameters.logs_to_save:
            tank_opts.extend(['-d', log])
        tank_opts.extend(['-j', 'jobno.txt'])
        tank_opts.extend([self.Context.ammo_option, ])
        if self.Context.monitoring_option:
            tank_opts.extend([self.Context.monitoring_option, ])
        if self.Parameters.config_add_parameters:
            tank_opts.extend([self.Parameters.config_add_parameters])
        tankapi_cmd = ' '.join(
            [
                'tankapi-cmd',
                '-v',
                '-c {}'.format(self.tank_config_path),
                '--timeout {}'.format(self.Parameters.timeout) if self.Parameters.timeout else ''
            ] + tank_opts
        )
        logging.info('TANKAPI-CMD is "%s"', tankapi_cmd)
        return tankapi_cmd

    def _is_test_online(self, retries=5):
        result = []

        for attempt in range(retries):
            try:
                response = requests.get(
                    'https://lunapark.yandex-team.ru/api/job/online.json?job={}'.format(
                        self.Parameters.lunapark_job_id
                    ),
                    verify=False
                )
                result = response.json()
            except Exception:
                logging.warning("Got exception requesting Lunapark api, retry", exc_info=True)
                time.sleep(1)

        for test in result:
            if self.Parameters.lunapark_job_id == str(test['n']):
                return True

        return False

    def _return_test_exit_code(self, retries=5):

        exit_code = None

        for attempt in range(retries):
            try:
                response = requests.get(
                    'https://lunapark.yandex-team.ru/api/job/offline.json?job={}'.format(
                        self.Parameters.lunapark_job_id
                    ),
                    verify=False
                )
                exit_code = response.json()[0]['quit_status']
                if not exit_code:
                    pass
                else:
                    break
            except Exception:
                logging.warning("Got exception requesting Lunapark api, retry", exc_info=True)
                time.sleep(1)

        return str(exit_code)

    def _save_tank_logs(self):
        tanklogs_dir_name = '{}_tanklogs'.format(self.Parameters.lunapark_job_id)
        os.makedirs(tanklogs_dir_name, 0o755)
        tanklogs_path = self.path(tanklogs_dir_name)
        for log in self.Parameters.logs_to_save:
            try:
                logname = glob.glob(log)[0]
                logging.info('Created link for %s in %s', logname, tanklogs_dir_name)
                os.link(str(self.path(logname)), os.path.join(str(tanklogs_path), logname))
            except IndexError:
                logging.error('No logs found for %s', log)
            except OSError:
                logging.error('Can\'t save log %s', log)
        tanklogs_gzip = self.path('{}.tar.gz'.format(tanklogs_dir_name))
        self._gzip_directory(str(tanklogs_path), str(tanklogs_gzip))
        folder_resource_data = sdk2.ResourceData(
            YANDEX_TANK_LOGS(self,
                             'Yandex tank logs for shoot {}'.format(self.Parameters.lunapark_job_id),
                             str(tanklogs_gzip)
                             )
        )
        folder_resource_data.ready()

    def _define_test_id(self):
        jobno_file = glob.glob('jobno*.txt')[0]
        job_no = fu.read_lines(jobno_file)[-1]
        logging.info('Task id from jobno.txt: %s', job_no)

        if 'None' in job_no:
            self.set_info(
                'Seems that shooting was done but results were not uploaded to Lunapark, '
                'please check tankapi-cmd.out.log')

        else:
            lunapark_link = 'https://lunapark.yandex-team.ru/{}'.format(job_no)
            self.Parameters.lunapark_link = lunapark_link
            self.Parameters.lunapark_job_id = job_no
            self.set_info(
                link_format(lunapark_link, lunapark_link, description='Lunapark link: '),
                do_escape=False
            )

        return job_no

    def on_prepare(self):
        try:
            os.makedirs('local_configs', 0o755)
        except OSError:
            pass
        self.local_configs = str(self.path('local_configs'))

        if 'github' in self.Parameters.config_source \
                or 'github' in self.Parameters.ammo_source \
                or 'github' in self.Parameters.monitoring_source:
            if self.Parameters.ssh_vault:
                with Key(self, self.Parameters.vault_owner, self.Parameters.ssh_vault):
                    self._clone_repository()
            else:
                self._clone_repository()

        if 'arcadia' in self.Parameters.config_source:
            arcadia_url = sdk2.svn.Arcadia.trunk_url(self.Parameters.config_arc_path)
            sdk2.svn.Arcadia.export(arcadia_url, self.local_configs + '/tank.config', depth='empty')
        if 'arcadia' in self.Parameters.monitoring_source:
            arcadia_url = sdk2.svn.Arcadia.trunk_url(self.Parameters.monitoring_arc_path)
            sdk2.svn.Arcadia.export(arcadia_url, self.local_configs + '/monitoring.config', depth='empty')
        if 'arcadia' in self.Parameters.ammo_source:
            arcadia_url = sdk2.svn.Arcadia.trunk_url(self.Parameters.ammo_arc_path)
            sdk2.svn.Arcadia.export(arcadia_url, self.local_configs + '/ammo', depth='empty')

    def on_execute(self):

        with self.memoize_stage['define_tank_list']:
            if self.Parameters.nanny_service:
                self._define_target()
            if self.Parameters.tanks:
                self._set_tanks_list()

            if self.Context.tanks:
                logging.info('List of tanks: %s, tankapi port %s', self.Context.tanks, self.Context.tankapi_port)
            else:
                raise ce.TaskError('No available tanks found.\n '
                                   'Please check that you\'ve switched on "Use public tanks" and defined the name '
                                   'of target nanny service. Either the list of tanks should be given in parameters.')

        self.tank_config_path = self._set_tank_config()
        self.Context.ammo_option = self._set_ammo()
        self.Context.monitoring_option = self._set_monitoring()
        self.tankapi_cmd = self._set_tankapi_cmd()

        # Start shooting via tankapi
        with ProcessLog(self, 'tankapi-cmd') as process_log:
            try:
                tank_call = sp.Popen(
                    self.tankapi_cmd, shell=True,
                    stdout=process_log.stdout, stderr=process_log.stdout
                )
                self.Context.tankapi_exit_code = tank_call.wait()
            except sp.CalledProcessError as process_error:
                raise ce.TaskError('Tankapi failed: {e}'.format(e=str(process_error)))

            if self.Context.tankapi_exit_code == 127:
                raise ce.TaskError('Tankapi not found, check LXC container')
            elif self.Context.tankapi_exit_code != 0:
                self.set_info(WRONG_RC_MESSAGE.format(self.Context.tankapi_exit_code))

        try:
            self._define_test_id()
        except (IndexError, IOError):
            logging.info('No file with job number found, check tankapi-cmd.out.log')

        if not self.Parameters.lunapark_job_id:
            line = ''
            try:
                for line in fu.read_line_by_line(str(process_log.stdout.path)):
                    if ('Another test is running with pid' in line or
                            'Another test is running' in line or
                            'Another test is already running' in line):
                        raise ce.TemporaryError(
                            'Lock file found on tank when initiating the shooting. Status set to temporary'
                        )
            except IOError:
                logging.info('No tankapi-cmd.out')
            finally:
                logging.debug('Log is %s with last line %s', str(process_log.stdout.path), line)
            raise ce.TaskError('Job number file not found or results were not uploaded to Lunapark, '
                               'please check tankapi-cmd.out')
        while self._is_test_online():
            logging.info('Test is still online. Waiting for test stop...')
            time.sleep(30)

    def on_finish(self, prev_status, status):
        super(ShootViaTankapi, self).on_finish(prev_status, status)
        tankapi_log = 'tankapi-cmd.out.log'

        self._save_tank_logs()

        shooting_result = self._return_test_exit_code()

        autostop = AUTOSTOP_CODES.get(shooting_result)
        if autostop:
            if self.Parameters.check_tankapi_rc and self.Context.tankapi_exit_code != 0:
                raise ce.TaskError('{}. Autostop by {}'.format(WRONG_RC_MESSAGE.format(self.Context.tankapi_exit_code), autostop))
            else:
                raise ce.TaskError('Tankapi returned exit code {}: autostop by {}'.format(shooting_result, autostop))
        elif shooting_result == '10':
            raise ce.TaskError(
                'Tankapi returned exit code not mentioned in RC assert list, please check {}'.format(tankapi_log))
        elif self.Context.tankapi_exit_code != 0 and self.Parameters.check_tankapi_rc:
            raise ce.TaskError(WRONG_RC_MESSAGE.format(self.Context.tankapi_exit_code))
