"""
Utility to find suitable tank from runtime cloud common tanks
"""

import collections
import logging
import re

import requests

TANKS_GROUP = 'production_yandex_tank'

Host = collections.namedtuple('Host', ['dc', 'host', 'port'])


class TankFinder(object):

    def __init__(self,
                 target_nanny_service='',
                 tanks_nanny_services=None,
                 target_host=None,
                 gencfg_group='ALL_RCLOUD_TANKS'):
        self.cloud_type = 'nanny'
        self.target_nanny_service = target_nanny_service
        self.tanks_nanny_services = tanks_nanny_services
        # TODO: add dc detection for given target LOAD-688
        self.target_host = Host(target_host.get('dc'),
                                target_host.get('host'),
                                target_host.get('port')) \
            if target_host else None
        self.gencfg_group = gencfg_group  # Used for special tanks in RTC
        self.targets = []
        self.tanks = []

    # @retry(tries=5, delay=5)
    def define_hosts_in_nanny_group(self, group_name, tank=False):
        logging.info('Trying to get info about nanny group %s', group_name)
        url = 'https://nanny.yandex-team.ru/v2/services/{}/current_state/instances/'.format(group_name)
        response = requests.get(url, verify=False)
        response.raise_for_status()
        assert not response.text.startswith('<html>')

        group_hosts = []
        data = [item for item in response.json()['result'] if item]
        for entry in data:

            container = 'container_hostname' if 'MTN_ENABLED' in entry['network_settings'] else 'hostname'
            dc = [tag[-3:] for tag in entry['itags'] if 'a_dc_' in tag][0]

            is_yp = False
            if re.match(r'.+\.yp-[a-z]\.yandex\.net', entry[container]):  # <pod_id>.<dc>.yp-c.yandex.net
                try:
                    dc = entry[container].split('.')[-4]
                    logging.info('Detected YP host, extracted DC %s from container name', dc)
                    is_yp = True
                except Exception:
                    logging.warning('It seems we\'ve got YP host but name %s does not match expected format',
                                    entry[container], exc_info=True)
            if tank and not is_yp:  # Filters gencfg tanks group.
                gencfg_filter = [tag for tag in entry['itags'] if 'a_topology_group-{}'.format(self.gencfg_group) in tag]
                if gencfg_filter:
                    group_hosts.append(Host(dc, entry[container], entry['port']))
            else:
                group_hosts.append(Host(dc, entry[container], entry['port']))

        logging.info('Hosts for group %s are %s', group_name, group_hosts)
        return group_hosts

    def define_tank_by_target(self):
        logging.info('Trying to get info about tanks in runtime cloud')
        # Search for tanks in either user-provided tanks nanny services or public TANKS_GROUP.
        if self.tanks_nanny_services:
            all_tanks = [tank for service in self.tanks_nanny_services
                         for tank in self.define_hosts_in_nanny_group(service, tank=True)]
        else:
            all_tanks = self.define_hosts_in_nanny_group(TANKS_GROUP, tank=True)
        possible_dc = list(set([host.dc for host in self.targets]))
        tanks = []

        def filter_dc(item):
            for datacenter in possible_dc:
                if item.dc == datacenter:
                    return True
            return False

        try:
            tanks = list(filter(filter_dc, all_tanks))
        except TypeError:
            tanks = []
        except Exception as exc:
            logging.error('Exception occured during tanks detection: %s', exc)

        if not tanks:
            logging.warning('No tanks were found in runtime cloud')
        else:
            logging.info('Tanks that can be used from runtime cloud: %s', tanks)

        return tanks

    def collect_info(self):
        if self.target_nanny_service:
            logging.info('Targets will be taken from nanny group %s', self.target_nanny_service)
            self.targets = self.define_hosts_in_nanny_group(self.target_nanny_service)
        if self.target_host:
            logging.info('Target %s added to targets', self.target_host)
            self.targets.append(self.target_host)

        if self.targets:
            self.tanks = self.define_tank_by_target()
        else:
            logging.warning('No targets were found by nanny service {}'.format(self.target_nanny_service))
