import re
import logging


TASK_ID_REGEX = r'^[a-zA-Z]+\-\d+$'
IPv4 = r'[1-2]?\d?\d\.[1-2]?\d?\d\.[1-2]?\d?\d\.[1-2]?\d?\d'
IPv6 = r'\[?[a-f0-9]{1,4}\:[a-f0-9:]+\]?'
FQDN = r'[a-z0-9]+[a-z-0-9.]*\.[a-z]+'
logger = logging.getLogger('CONFIG PARSER')

class ConfigManager:

    def __init__(self, task_id, config, monitoring_config_path=None):
        self.config = compile_config(config)
        logger.debug('ConfigManager config:\n%s', self.config)
        self.monitoring_config_path = monitoring_config_path or ''
        self.sandbox_task_id = task_id
        self.custom = False

    def __get_metaconf_option(self, option):
        """ Parse option value from metaconf/firestarter section
        :param option: option name
        :type option: str
        :return: option value or empty string
        :rtype: str
        """
        if self.config.get('metaconf', {}).get('enabled', True):
            value = self.config.get('metaconf', {}).get('firestarter', {}).get(option, '')
            return value.strip() if isinstance(value, str) else value

    def __get_meta_option(self, section, option):
        """ Parse option value from meta in given section
        :param section: section name
        :type section: str
        :param option: option name
        :type option: str
        :return: option value or empty string
        :rtype: str
        """
        value = self.config.get(section, {}).get('meta', {}).get(option, '')
        return value.strip() if isinstance(value, str) else value

    def __get_option(self, section, option):
        """ Parse option value from given section
        :param section: section name
        :type section: str
        :param option: option name
        :type option: str
        :return: option value or empty string
        :rtype: str
        """
        if self.config.get(section, {}).get('enabled', True):
            value = self.config.get(section, {}).get(option, '')
            return value.strip() if isinstance(value, str) else value

    def __get_pandora_option(self, section, option):
        """ Parse option value from the first pool in pandora.config_content section
        :param section: section name
        :type section: str
        :param option: option name
        :type option: str
        :return: option value or empty string
        :rtype: str
        """
        if self.config.get('pandora', {}).get('enabled', True):
            value = self.config.get('pandora', {}).get('config_content', {}).get('pools', [{}])[0].get(section, {}).get(option, '')
            return value.strip() if isinstance(value, str) else value

    def get_tank(self):
        """
        Parses config and takes tank and port settings from uploader meta.
        :return: Tank and port values
        :rtype: set(str, str)
        """
        tank = (self.__get_metaconf_option('tank') or self.__get_meta_option('uploader', 'use_tank'))
        port = (self.__get_metaconf_option('tank_port') or self.__get_meta_option('uploader', 'use_tank_port'))
        tank, port = self._parse_shooting_point_(tank, port)
        logger.debug('Parsed tank: %s, parsed port: %s', tank, port)
        return tank, port

    def get_target(self):
        """
        Parses config and takes tank and port settings from generator section.
        By default takes phantom target, otherwise checks pandora section.
        Ipv6 should be in brackets, or it's impossible to differ last part from port.
        If no port is set, 80 is taken by default
        :return: Target and port values
        :rtype: set(str, str)
        """
        meta_target = self.__get_metaconf_option('target')
        meta_target_port = self.__get_metaconf_option('target_port')
        phantom_target = self.__get_option('phantom', 'address')
        pandora_target = self.__get_pandora_option('gun', 'target')
        raw_target = meta_target or pandora_target or phantom_target
        try:
            target, port = self._parse_shooting_point_(raw_target, meta_target_port)
        except (ValueError, AttributeError):
            target, port = raw_target, meta_target_port
        return target, port

    def _parse_shooting_point_(self, value, port=''):
        """
        Parses value for tank or target defined in config.
        :return: Target and port values
        :rtype: set(str, str)
        """
        if isinstance(value, str):

            if value.startswith('nanny:') or value.startswith('deploy:') or value.startswith('sandbox'):
                value, port = ':'.join(value.split(':')[:2]), (value.split(':') + ['', ''])[2] or port

            elif value.startswith('['):
                if ']' in value:
                    value, port = re.findall(IPv6, value)[0].strip('[]'), value.split(']', 1)[-1].strip(':') or port
                else:
                    logger.debug('Wrong value %s', value)
                    value, port = '', ''

            elif ':' in value and not re.match(IPv6, value):
                value, port = value.rsplit(':', 1)

        return value, str(port)

    def get_operator(self):
        """
        Parses config and takes operator from uploader section.
        :return: operator or logged_user or empty string
        :rtype: str
        """
        return self.__get_option('uploader', 'operator')

    def get_task(self):
        """
        Parses config and takes task from uploader section
        :return: task name
        :rtype: string or None
        :raise: TaskNotFound if no valid task detected
        """
        return self.__get_option('uploader', 'task')

    def get_datacenter(self):
        """
        Parses config and return value of the datacenter from metaconf.firestarter section if it exist else return empty string.
        return: dc
        rtype: string
        """
        return self.__get_metaconf_option('dc')

    def pandora_is_custom(self):
        """ Shows whether pandora is used in shooting
        :return: enabled flag from pandora section
        :rtype: bool
        """
        pandora_cmd = self.__get_option('pandora', 'pandora_cmd') 
        return pandora_cmd != '/usr/local/bin/pandora' if pandora_cmd else False

    def check_pandora_config_file(self):
        """
        Parses config and return config_file option from pandora section if it exist else return empty string.
        return: config_file availability
        rtype: bool
        """
        return bool(self.__get_option('pandora', 'config_file'))

    def check_bfg_used(self):
        """ 
        Shows whether BFG is used in shooting
        :return: enabled flag from bfg section
        :rtype: bool
        """
        return bool(self.config.get('bfg', {}).get('enabled', False)) or bool(self.config.get('bfg2020', {}).get('enabled', False))

    def check_jmeter_used(self):
        """ 
        Shows whether jMeter is used in shooting
        :return: enabled flag from jmeter section
        :rtype: bool
        """
        return bool(self.config.get('jmeter', {}).get('enabled', False))

    def check_for_custom_gun(self):
        """ 
        Check config for special options
        :rtype: bool
        """
        return self.pandora_is_custom() or self.check_pandora_config_file() or self.check_bfg_used() or self.check_jmeter_used()

    def set_launched_from(self):
        """
        Adds firestarter sandbox task id to parameter 'launched_from'
        """
        self.config['uploader'].setdefault('meta', {})['launched_from'] = \
            'https://sandbox.yandex-team.ru/task/{}'.format(self.sandbox_task_id)

    def set_target(self, host, port):
        """
        Rewrites address set to nanny group with target host and target port.
        At the moment only phantom amd pandora are supported.
        :param host: hostname, can be fqdn or ip
        :type host: str
        :param port: port value
        :type port: str
        """
        if re.match(r'^[a-f0-9][a-f0-9:]+[a-f0-9]$', host):
            host = '[{}]'.format(host)
        if self.__get_option('phantom', 'enabled'):
            self.config['phantom']['address'] = '{host}:{port}'.format(host=host, port=port)
        elif self.__get_option('pandora', 'enabled'):
            for pool in self.config['pandora']['config_content']['pools']:
                pool['gun']['target'] = '{host}:{port}'.format(host=host, port=port)

    def set_tank(self, host, port):
        """
        Rewrites address of the usable tank into uploader/meta section.
        :param host: hostname, can be fqdn or ip
        :type host: str
        :param port: port value
        :type port: str
        """
        logger.info("Set tank %s:%s", host, port)
        if self.__get_option('uploader', 'enabled'):
            self.config['uploader']['meta'] = {
                'use_tank': host,
                'use_tank_port': port
            }

    def set_monitoring_path(self, path):
        """
        Adds monitoring path if monitoring config exists
        :param path: relative path to monitoring config
        :type path: str
        """
        self.config.setdefault('telegraf', {})['config'] = path
        self.config['telegraf']['enabled'] = True

    def set_use_tank_port(self):
        """
        Adds use_tank_port for the most obvious mistake with use_tank:
        when group nanny:production_yandex_tank is defined but no use_tank_port set
        """
        tank_port = self.config['uploader']['meta'].get('use_tank_port')
        if not tank_port:
            self.config['uploader']['meta']['use_tank_port'] = 30169

    def set_operator(self, operator):
        """
        Rewrite operator value in uploader section of the config:
        :param operator: operator value
        :type operator: str
        """
        self.config['uploader']['operator'] = operator


def parse_tank(tank):
    """
    :param tank: tank name taken from config
    :type tank: str
    :return: hostname, service name and group name. Every string can be empty.
    :rtype: set(str, str, str)
    """
    host, service, group = '', '', ''
    if tank and (tank.startswith('nanny') or tank.startswith('deploy')):
        try:
            service, group = tank.split('#')
        except ValueError:
            service = tank
    else:
        host = tank
    return host, service, group


def compile_config(config):
    """
    Compile config with the !inject options
    :param config:
    :type config: str
    :return: config
    :rtype: dict
    """
    import tempfile
    import yaml
    import os
    from load.tools.yaml_injection import InjectionLoader
    from load.tools.yaml_injection.resource_loader import BaseResourceLoader
    from sandbox.projects.tank.Firestarter.status import FirestarterError, InternalStatus

    logger = logging.getLogger('COMPILE CONFIG')

    class SandboxArcadiaLoader(BaseResourceLoader):
        """
        Class for reading files from arcadia when compiling the shooting configuration
        """

        from sandbox.projects.tank.common import retry

        def key(self):
            return 'arcadia'

        @retry(tries=3)
        def download_resource(self, arc_path):

            from sandbox.sdk2.vcs.svn import Arcadia

            temp_file = os.path.join(tempfile.gettempdir(), next(tempfile._get_candidate_names()))
            try:
                Arcadia.export(Arcadia.trunk_url(arc_path), temp_file, depth='empty')
                with open(temp_file, 'r') as config:
                    read = config.read()
                    logger.info('Downloaded from arcadia: ')
                    logging.info(read)
                    return read
            finally:
                os.remove(temp_file)

    try:
        config = yaml.load(config, InjectionLoader.with_resources(SandboxArcadiaLoader()))
        assert type(config) == dict
        return config

    except (yaml.parser.ParserError, yaml.scanner.ScannerError, AssertionError):
        logger.debug('Shooting config has syntax error or wrong format.\n%s', config, exc_info=True)
        raise FirestarterError(
            status=InternalStatus.FAILED_TO_START,
            error='Wrong shooting config format or syntax error',
            section='compile_config',
        )
