import logging
import requests
import re

from requests.exceptions import ConnectionError
from sandbox.projects.tank.common import retry
from .config_parser import parse_tank


TANK_FINDER_URL = 'https://tank-finder.yandex-team.ru:443'
VALIDATOR_URL = 'https://tank-validator.yandex-team.ru'
SANDBOX_URL = 'https://sandbox.yandex-team.ru'
STATUS_URL = 'http://{}/api/v1/tank/status.json'
DCS = ['iva', 'man', 'myt', 'sas', 'vla']
logger = logging.getLogger('EXTERNAL CALL')

def parse_tf_response(tank_finder_reply):
    tank_list = []
    err = ''

    if isinstance(tank_finder_reply, dict) and tank_finder_reply.get('result') == 'error':
        err = tank_finder_reply.get('error_msg')

    if isinstance(tank_finder_reply, list):
        for item in tank_finder_reply:
            tank_list.append('{host}:{port}'.format(host=item.get('tank'), port=item.get('port')))

    return tank_list, err


def external_call(service, method, url, payload=None):
    error = ''
    result = {}

    @retry(tries=3)
    def _request(_method, _url, _payload=None):
        resp = requests.request(method=_method, url=_url, json=_payload, verify=False)
        return resp

    try:
        response = _request('GET', url) if method == 'GET' else _request('POST', url, payload)
        result = response.json()
        if response.text:
            logger.debug('Response from %s: %s', service, response.text)
    except (ConnectionError, ValueError):
        error = 'Reply from {} is not available, check backend logs'.format(service)
        logger.debug('Reply from %s is not valid', service, exc_info=True)
    logger.info('Result %s', result)
    if error:
        logger.debug('Error %s', error)
    return result, error


def call_validator(input_config):
    """
        External call to validator
        :param: input_config - Input config
        :type: Dict
        :return validated config, errors
        :rtype set(string, dict)
        Possible errors:
            -  Validator is not available:
              - ConnectionError
            - Validator reply is not valid:
              - validator reply is not 200
              - validator reply is not valid json
              - validator reply doesn't have keys 'config' and 'errors'
    """

    url = '{base}/config/validate.json'.format(base=VALIDATOR_URL)
    res, message = external_call('validator', 'POST', url, input_config)
    output_config = res.get('config', {})
    errors = ['{}: {}'.format(k, v) for k, v in res.setdefault('errors', {}).items()] if not message else []
    return output_config, errors


def call_tank_finder(target, tank, dc=''):
    """
    External call to tank finder
    Checks if tank and target are located in the same DC
    :param target: target hostname, ip or nanny group
    :type target: str
    :param tank: tank hostname, ip or nanny group
    :type tank: str
    :param dc: tank dc, can be empty
    :type dc: str
    :return: tank_list
    :rtype: list
    """
    if dc:
        if tank.startswith('nanny:'):
            tank_list = call_tank_finder_nanny(tank, dc)
        elif tank.startswith('deploy:'):
            tank_list = call_tank_finder_deploy(tank, dc)
        else:
            tank_list = []
    else:
        tank_value, tank_service, tank_group = parse_tank(tank)
        url = '{base}/tanks/list.json?target={target}&service={service}&group={group}'.format(
            base=TANK_FINDER_URL, target=target.strip('[]'), service=tank_service, group=tank_group
        )
        logger.info('Call %s', url)
        res, message = external_call('tank_finder', 'GET', url)
        tank_list, err = parse_tf_response(res)
        error = message or err
        if error:
            logger.debug('Error received:  %s', error)
        if not isinstance(tank_list, list):
            logger.debug('Wrong tank list was received:  %s', tank_list)
            tank_list = []
    return tank_list


def call_tank_finder_deploy(project, dc=''):
    """
    Gets list of a instances from the deploy stage
    :param project: deploy params
    :type project: str
    :param dc: advanced datacenter value, optional
    :type dc: str
    :return: hosts
    :rtype: list
    """

    stage, deploy_unit, datacenter = (re.sub(r'^deploy:', '', project).split('.') + ['', ''])[:3]
    datacenter = datacenter or dc.lower()
    datacenter = datacenter if datacenter in DCS else ''
    url = '{base}/deploy?stage={stage}&unit={deploy_unit}&dc={datacenter}'.format(base=TANK_FINDER_URL, stage=stage, deploy_unit=deploy_unit, datacenter=datacenter)
    logger.info('Call %s', url)
    res, error = external_call('tank_finder', 'GET', url)
    if error:
        logger.debug('[CALL TANK-FINDER DEPLOY] Error received: %s', error)
    return res.get('fqdn', [])


def call_tank_finder_nanny(service, dc=''):
    """
    Gets list of a instances from the nanny service
    :param service: nanny service_name and group
    :type service: str
    :param dc: datacenter value, optional
    :type dc: str
    :return: hosts
    :rtype: list
    """
    datacenter = dc.lower() if dc.lower() in DCS else ''
    service, group = (re.sub(r'^nanny:', '', service).split('#') + [''])[:2]
    url = '{base}/nanny?service={service}&group={group}&dc={datacenter}'.format(base=TANK_FINDER_URL, service=service, group=group, datacenter=datacenter)
    logger.info('Call %s', url)
    res, error = external_call('tank_finder', 'GET', url)
    if error:
        logger.debug('Error received: %s', error)
    return res.get('fqdn', [])


def define_host_dc(host):
    """
    Gets the datacenter of the host
    :param host: hostname, ip, nanny group or deploy params
    :type host: str
    :return: dc, error
    :rtype: str, str
    """
    url = '{base}/target_dc?target={target}'.format(base=TANK_FINDER_URL, target=str(host).strip('[]'))
    logger.info('Call %s', url)
    resp, error = external_call('tank_finder', 'GET', url)
    if isinstance(resp, dict):
        error = error or resp.get('error_msg')
        if error:
            logger.debug('Unable to get DC for host %s. %s', host, error)
        return resp.get('datacenter')
    else:
        logger.debug('Wrong response data %s for request %s.', resp, url)


def get_tank_version(tank):
    """
    Getting the version of the tank
    :param tank: tank hostname, ipv4, ipv6 with the port
    :type tank: str
    :return: version
    :rtype: str
    """
    try:
        return requests.get(STATUS_URL.format(tank)).json()['version'].split('/')[1]
    except Exception:
        logger.debug('Failed to find a version for the tank %s.', tank, exc_info=True)


@retry(tries=3)
def get_sandbox_output(task_type, output_parameter, task_tags=''):
    """
    Geting the output parameter from the specified sandbox task
    :param task_type: Sandbox Task Type 
    :type task_type: str
    :param task_tags: Sandbox Task Tags
    :type task_tags: str
    :param output_parameter: Sandbox Task output parameter
    :type output_parameter: str
    :return: output_value
    :rtype: str
    example: curl -X GET -H "accept: application/json; charset=utf-8"
            "https://sandbox.yandex-team.ru/api/v1.0/task?limit=10&offset=0&order=-id&type=GENERATE_YAPPY_BETA&status=SUCCESS&tags=RM_CI,RELEASE%3ARELEASE_GOODS_BACKEND,RM_COMPONENT%3AGOODS_BACKEND,RM_COMPONENT%3AGOODS_BACKEND"
    """
    from json.decoder import JSONDecodeError

    sandbox_url = '{base}/api/v1.0/task?limit=5&offset=0&order=-id&type={task_type}&status=SUCCESS&tags={task_tags}'.format(base=SANDBOX_URL, task_type=task_type, task_tags=task_tags.upper().replace(':', '%3A').replace('=', '%3A'))
    logger.info('Call %s', sandbox_url)
    headers = {'Accept': 'application/json', 'Accept-Charset': 'utf-8'}
    response = requests.get(sandbox_url, headers=headers)

    try:
        output_value = response.json().get('items', [{}])[0].get('output_parameters', {}).get(output_parameter, '')
        logger.info('The value %s was found for the %s parameter', output_value, output_parameter)
        return str(output_value)
    except (TypeError, JSONDecodeError, IndexError):
        logger.debug('Wrong sandbox answer. %s', response, exc_info=True)
