""" https://st.yandex-team.ru/LUNAPARK-3417 """
import os
import re
import shutil
import logging
import fileinput

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox.projects.tank.Firestarter.external_calls import retry
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.BuildDockerImageFromGit import DockerContentResourse
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class FailedCloneGit(Exception):
    pass


default_repo = 'bb.yandex-team.ru/scm/load/luna.git'


class BuildLunaFromBB(sdk2.Task):
    """
       Metajob to build Luna docker image (cf. https://bb.yandex-team.ru/projects/LOAD/repos/luna/browse)
       Outdated
    """
    version = ''

    class Requirements(sdk2.Requirements):
        disk_space = 512

    class Context(sdk2.Task.Context):
        version = ''

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Luna parameters') as tasks_block:
            build_docker_luna = sdk2.parameters.Bool(
                'Build docker image of Luna',
                default=True
            )
            tag = sdk2.parameters.String(
                label='Docker tag',
                default='dev'
            )
            repo = sdk2.parameters.String(
                label='Git repo (https)',
                default=default_repo,
            )
            branch = sdk2.parameters.String(
                label='Git branch',
                description='Git branch to checkout from',
                default='dev'
            )
            luna_registry = sdk2.parameters.String(
                'Registry.yandex.net path',
                default='load/volta-back'
            )
        with sdk2.parameters.Group('Vault parameters') as general_block:
            vault_login = sdk2.parameters.String('Robot name for vault records', default='robot-lunapark')
            vault_owner = sdk2.parameters.String('Vault item owner', default='LOAD')

    @retry(tries=5, delay=2)
    def clone_repository(self, repo, branch, service):
        checkout_dir = self.path(service).as_posix()
        clone_cmd = 'git clone -b {} {} {}'.format(branch, repo, checkout_dir)
        with ProcessLog(self, logging.getLogger('git_clone')) as process_log:
            status = sp.Popen(
                clone_cmd,
                shell=True,
                stdout=process_log.stdout,
                stderr=process_log.stderr
            ).wait()
            if status != 0:
                raise FailedCloneGit('Failed to clone git repo')
        logging.info('Repository %s/%s cloned to %s', repo, branch, checkout_dir)
        return checkout_dir

    def prepare_resource(self, checkout_dir):
        """
        Saves Dockerfile and repo around to sandbox resource for using in BuildDockerImageV6 task
        :return: sdk2.Resource
        """
        docker_file = os.path.join(checkout_dir, 'Dockerfile')
        if not os.path.exists(docker_file):
            raise ce.TaskError("Dockerfile {} not found inside the repository".format(
                docker_file)
            )
        for line in fileinput.input(docker_file, inplace=1):
            # change Luna version
            line = line.replace(
                'ARG LUNA_BRANCH=master',
                'ARG LUNA_BRANCH={}'.format(self.Parameters.branch)
            ).rstrip()
            # change supervisord.conf and yandex.list location since Dockerfile should be placed in root
            line = line.replace(
                'COPY supervisord.conf /etc/supervisord.conf',
                'COPY docker/supervisord.conf /etc/supervisord.conf'
            ).rstrip()
            line = line.replace(
                'COPY ./yandex.list /etc/apt/sources.list.d/yandex.list',
                'COPY docker/yandex.list /etc/apt/sources.list.d/yandex.list'
            ).rstrip()
            print(line)
        # make sure dockerfile is named 'Dockerfile' and is located in repo root
        # so that the child task could fetch it
        shutil.move(docker_file, os.path.join(checkout_dir, 'Dockerfile'))
        resource = DockerContentResourse(
            self,
            'Dockerfile and contents from {repo}'.format(repo=os.path.basename(checkout_dir)),
            checkout_dir
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_docker_image(self, resource, registry_tags):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: registry_tags,
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.vault_login,
            BuildDockerImageV6.VaultItemName.name: '{}-token'.format(self.Parameters.vault_login),
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.vault_owner,
        }
        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description="Building and publishing Docker image from Git source (by {})".format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _check_subtasks_status(self):
        sub_tasks = self.find()
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                raise ce.TaskFailure("Subtask {} {} is failed with status {}".format(task.type, task.id, task.status))

    @staticmethod
    def _get_cmd_output(cmd, cwd):
        return sp.check_output([cmd], shell=True, cwd=cwd).strip()

    def on_execute(self):
        bb_token = sdk2.Vault.data('robot-lunapark-bb-oauth')
        # ssh://git@bb.yandex-team.ru/~fomars/luna.git
        repo = re.compile(r'bb\.ya..+\.git$').search(str(self.Parameters.repo)).group(0)
        auth = 'x-oauth-token:{}@'.format(bb_token) \
            if self.Parameters.repo == default_repo else ''
        self.Context.internal_luna_dir = self.clone_repository(
            'https://{}{}'.format(auth, repo),
            self.Parameters.branch,
            'volta'
        )
        self.Context.version = self._get_cmd_output("grep version setup.py | awk -F\\' '{print $2}'",
                                                    self.Context.internal_luna_dir)
        with self.memoize_stage.build_luna:
            luna_git_resource = self.prepare_resource(self.Context.internal_luna_dir)
            if self.Parameters.build_docker_luna:
                # noinspection PyTypeChecker
                self.build_docker_image(
                    resource=luna_git_resource,
                    registry_tags=[
                        '{}:{}-{}'.format(self.Parameters.luna_registry,
                                          self.Parameters.branch.replace('/', '_'),
                                          self.Context.version)
                    ] + [
                        '{}:{}'.format(self.Parameters.luna_registry, self.Parameters.tag)
                    ] * (len(self.Parameters.tag) > 0),
                )
        with self.memoize_stage.check_status:
            self._check_subtasks_status()

    def get_docker_tag(self, version):
        if self.Parameters.branch == 'master' and self.Parameters.repo == default_repo:
            return version
        else:
            return '{}-{}'.format(version, self.Parameters.branch)
