# coding: utf-8

import os
import logging
import requests
import re

from sandbox import sdk2
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.sdk2.ssh import Key
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.projects.common.build.YaPackage as YaPackage
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.tank.Firestarter.external_calls import retry
from sandbox.projects.tank.build.TankDebuilder import TankDebuilder
from sandbox.projects.tank.build.DeployValidator import DeployValidator
from sandbox.projects.BuildDockerImageFromGit import DockerContentResourse
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, UbuntuRelease

from .build_tank_lxc import (
    build_tank_lxc_container,
    release_lxc_sandbox_resource,
    test_sandbox_tank_shooting,
    check_test_tank_shooting_result
)


class FailedCloneGit(Exception):
    pass


# noinspection PyTypeChecker
class BuildTankFromArcadia(sdk2.Task):
    """
       Metajob to build internal Yandex.Tank from Arcadia (cf. https://a.yandex-team.ru/arc/trunk/arcadia/load/projects/yandex-tank-internal-pkg)
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 1024
        kill_timeout = 60 * 10
        client_tags = ctc.Tag.LXC

    class Context(sdk2.Task.Context):
        # pylint: disable=too-few-public-methods
        local_repo = ''
        deb_version = ''
        build_task_id = 0
        shoot_task_id = 0

    class Parameters(sdk2.Task.Parameters):
        # pylint: disable=too-few-public-methods
        with sdk2.parameters.Group('Source branch') as source_branch_block:
            use_branches = sdk2.parameters.Bool('Use branches to build tank?')

            with use_branches.value[True]:
                checkout_arcadia_from_url = sdk2.parameters.String(
                    label='Arc url for arcadia',
                    description='Arc branch.\nE.g. arcadia-arc:/#users/<username>/branch',
                )

        with sdk2.parameters.Group('Tasks') as tasks_block:
            build_docker_tank = sdk2.parameters.Bool(
                'Build docker image of Yandex.Tank with YA_PACKAGE',
                default=True
            )

            with use_branches.value[True]:
                with build_docker_tank.value[True]:
                    docker_version = sdk2.parameters.String(
                        label='Custom docker version',
                        description='If you build from personal branch, result will be uploaded to load/test/yandex-tank-internal with version from this field',
                        default=''
                    )

            build_debian_tank = sdk2.parameters.Bool(
                'Build debian package of Yandex.Tank with YA_PACKAGE',
                default=True
            )

            build_docker_validator = sdk2.parameters.Bool(
                'Build docker image of validator',
                default=True
            )
            deploy_qloud_validator = sdk2.parameters.Bool(
                'Deploy validator on qloud',
                default=True
            )
            build_tankapi_lxc = sdk2.parameters.Bool(
                'Build LXC container with tankapi for SHOOT_VIA_TANKAPI',
                default=False
            )
            build_tank_lxc = sdk2.parameters.Bool(
                'Build LXC container with tank for SandboxTank',
                default=True
            )

        with sdk2.parameters.Group('Vault parameters') as general_block:
            ssh_vault_name = sdk2.parameters.String(
                'Vault item with ssh key for git access',
                default='robot-lunapark-github-ssh'
            )
            ssh_vault_owner = sdk2.parameters.String('Vault item owner', default='LOAD')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for registry.yandex.net (vault item name)',
                default='robot-lunapark-token'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner for oauth token', default='LOAD')
            robot_login = sdk2.parameters.String('Robot login', default='robot-lunapark')

    @retry(tries=5, delay=2)
    def clone_repository(self, repo, branch='master', service='yandex_tank_internal'):
        checkout_dir = self.path(service).as_posix()
        clone_cmd = 'git clone -b {} {} {}'.format(branch, repo, checkout_dir)
        with Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            with ProcessLog(self, logging.getLogger('git_clone')) as process_log:
                status = sp.Popen(
                    clone_cmd,
                    shell=True,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr
                ).wait()
                if status != 0:
                    raise FailedCloneGit('Failed to clone git repo')
        self.Context.local_repo = checkout_dir
        logging.info('Repository %s/%s cloned to %s', repo, branch, checkout_dir)
        return checkout_dir

    def prepare_resource(self, repo_dir, branch):
        """
        Saves Dockerfile and repo around to sandbox resource for using in BuildDockerImageV6 task
        :return: sdk2.Resource
        """
        logging.info('Dockerfile and contents from {}'.format(os.path.basename(repo_dir)))
        logging.info('Branch {}'.format(branch))
        resource = DockerContentResourse(
            self,
            'Dockerfile and contents from {}, branch {}'.format(os.path.basename(repo_dir), branch),
            repo_dir
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_with_yapackage(self, pkg_type, version, desc, checkout_arcadia_path, **kwargs):
        task_class = sdk2.Task['YA_PACKAGE']
        ya_package_params = {
            YaPackage.parameters.ArcadiaUrl.name: checkout_arcadia_path,
            YaPackage.CustomVersionParameter.name: version,
            YaPackage.PackageTypeParameter.name: pkg_type
        }
        ya_package_params.update(kwargs)
        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description=desc,
            owner='LOAD',
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **ya_package_params
        ).enqueue()
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def debuild_tank(self):
        for distro in self.Parameters.ubuntu_version:
            debuild_tank_subtask = TankDebuilder(
                self,
                description='Build yandex-tank-internal for {}'.format(distro),
                owner=self.owner,
                git_repo=self.Parameters.source_pkg_repo,
                tank_revision=self.Parameters.revision,
                tankapi_revision=self.Parameters.tankapi_revision,
                distro_version=distro,
                vault_items_owner=self.Parameters.oauth_vault_owner,
                robot_login=self.Parameters.robot_login,
                create_resources=self.Parameters.create_resource,
                dupload=self.Parameters.dupload_package
            )
            debuild_tank_subtask.enqueue()

    def deploy_to_qloud(self):
        deploy_qloud_subtask = DeployValidator(
            self,
            description='Deploy validator to qloud',
            owner=self.owner,
            registry_url='registry.yandex.net/load/validator',
            tag=self.Parameters.py_version
        )
        self.Context.qloud_subtask = deploy_qloud_subtask.id
        deploy_qloud_subtask.enqueue()

    def build_lxc_with_tankapi(self):
        # rewrite to debian
        install_script = """#!/usr/bin/env bash
set -ex

### install tankapi-client
apt-get update -q && \
apt-get install --no-install-recommends -yq tankapi-client

### check tvm2 lib
apt-cache policy  libticket-parser2-python

### clean apt archives
apt-get clean
apt-get autoclean
rm -f /var/cache/apt/*.bin
"""

        build_tankapi_lxc_subtask = SandboxLxcImage(
            self,
            owner=self.owner,
            resource_type=YANDEX_TANKAPI_LXC_CONTAINER.name,
            resource_description='Container with tankapi and other stuff for loadtests. '
                                 'Tank version {}'.format(self.Context.deb_version),
            ubuntu_release=UbuntuRelease.XENIAL,
            custom_image=True,
            custom_repos='deb http://dist.yandex.ru/yandex-xenial stable/all/\n'
                         'deb http://dist.yandex.ru/yandex-xenial stable/amd64/\n'
                         'deb http://load-xenial.dist.yandex.ru/load-xenial stable/all/\n'
                         'deb http://load-xenial.dist.yandex.ru/load-xenial stable/amd64/',
            custom_script=install_script,
            custom_packages='git software-properties-common python-pip libticket-parser2-python',
            custom_attrs={'ttl': '365'}
        ).enqueue()
        self.Context.build_lxc_subtask = build_tankapi_lxc_subtask.id

    def get_version_from_trunk(self):
        sdk2.svn.Arcadia.export('arcadia:/arc/trunk/arcadia/load/projects/yandex-tank/yandextank/version.py', './version.py', depth='empty')
        with open('./version.py') as fd:
            current_version = re.findall("VERSION = '(.+)'", fd.read())[0]
        return current_version

    def make_json_request(self, uri, oauth=False):
        session = requests.Session()
        if oauth:
            session.headers.update({'Authorization': 'OAuth {}'.format(sdk2.Vault.data(self.Parameters.oauth_vault_owner, self.Parameters.oauth_vault_name))})
        response = getattr(session, 'get')(uri)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            logging.exception('Api Error')
            logging.error('Response body: %s', response.text)
        return response.json()

    def get_tags(self, repo):
        url = 'https://registry.yandex.net/v2/{}/tags/list'.format(repo)
        return self.make_json_request(url, oauth=True)['tags']

    def is_docker_tag_used(self, tank_version, repo):
        tags = self.get_tags(repo)
        for tag in tags:
            if len(re.findall(tank_version, tag)) > 0:
                logging.info('find version {} docker registry'.format(tag))
                return True
        return False

    def get_deb_versions(self):
        url = 'http://dist.yandex.ru/find?env=stable&pkg=yandex-tank-internal&repo=load-xenial'
        return self.make_json_request(url)['result']

    def is_debian_version_used(self, tank_version):
        meta = self.get_deb_versions()
        versions = []
        for item in meta:
            if len(re.findall(tank_version, item['version'])) > 0:
                logging.info('find version {} at load-xenial.dist.yandex.ru in {} environment'.format(item['version'], item['environment']))
                return True
            versions.append(item['version']+'\n')
        logging.info('versions in dist:\n{}'.format(versions))
        return False

    def _wait_for_subtasks(self):
        sub_tasks = self.find()
        if sub_tasks:
            raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _check_subtasks_status(self):
        sub_tasks = self.find()
        task_errors = ''
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                task_errors += 'Subtask {} {} is failed with status {}\n'.format(task.type, task.id, task.status)
        if task_errors:
            raise ce.TaskFailure(task_errors)

    def get_current_version(self):
        sdk2.svn.Arcadia.export('arcadia:/arc/trunk/arcadia/load/projects/yandex-tank/yandextank/version.py', './version.py', depth='empty')
        with open('./version.py') as fd:
            current_version = re.findall("VERSION = '(.+)'", fd.read())[0]
        logging.info('current version is: {}'.format(current_version))
        return current_version

    # noinspection PyTypeChecker
    def on_execute(self):
        if not self.Parameters.use_branches:
            self.Context.checkout_arcadia_path = 'arcadia-arc:/#trunk'
            self.Context.version = self.get_current_version()
            docker_image_repository = 'load'
        else:
            self.Context.checkout_arcadia_path = self.Parameters.checkout_arcadia_from_url
            self.Context.version = self.Parameters.docker_version
            docker_image_repository = 'load/test'
            if self.Parameters.build_debian_tank:
                self.Parameters.description += '\nYou building from branch, in this mode debian building prohibited. Use YA_PACKAGE to build custom package'

        # pylint: disable=no-member
        if self.Parameters.build_docker_tank:
            with self.memoize_stage.build_docker_tank:
                if self.is_docker_tag_used(self.Context.version, '{}/yandex-tank-internal'.format(docker_image_repository)):
                    self.Parameters.description += 'Current version: {} already in docker registry'.format(self.Context.version)
                else:
                    description = 'Building and publishing yandex-tank Docker image, version: {}'.format(self.Context.version)
                    kwargs = {
                        YaPackage.PackagesParameter.name: 'load/projects/yandex-tank-internal-pkg/docker_pkg.json',
                        YaPackage.DockerImageRepositoryParameter.name: docker_image_repository,
                        YaPackage.DockerUserParameter.name: 'LOAD',
                        YaPackage.DockerPushImageParameter.name: True,
                        YaPackage.DockerTokenVaultName.name: 'robot-lunapark-token'
                    }
                    self.build_with_yapackage(
                        YaPackage.DOCKER,
                        self.Context.version,
                        description,
                        self.Context.checkout_arcadia_path,
                        **kwargs
                    )

        # pylint: disable=no-member
        if self.Parameters.build_debian_tank and self.Parameters.checkout_arcadia_from_url == '':
            with self.memoize_stage.build_debian_tank:
                if self.is_debian_version_used(self.Context.version):
                    self.Parameters.description += 'Current version: {} already in debian dist'.format(self.Context.version)
                else:
                    description = 'Building and publishing yandex-tank Debian package, version: {}'.format(self.Context.version)
                    kwargs = {
                        YaPackage.PackagesParameter.name: 'load/projects/yandex-tank-internal-pkg/debian_pkg.json',
                        YaPackage.CompressPackageArchiveParameter.name: True,
                        YaPackage.PublishPackageParameter.name: True,
                        YaPackage.KeyUserParameter.name: 'robot-lunapark',
                        YaPackage.MultiplePublishMappingParameter.name: {'load/projects/yandex-tank-internal-pkg/debian_pkg.json': 'load-xenial'}
                    }
                    self.build_with_yapackage(
                        YaPackage.DEBIAN,
                        self.Context.version,
                        description,
                        self.Context.checkout_arcadia_path,
                        **kwargs
                    )

        # pylint: disable=no-member
        if self.Parameters.build_docker_validator:
            with self.memoize_stage.build_validator:
                if self.is_docker_tag_used(self.Context.version, '{}/validator'.format(docker_image_repository)):
                    self.Parameters.description += 'Current version: {} already in docker registry'.format(self.Context.version)
                else:
                    description = 'Building and publishing validator Docker image, version: {}'.format(self.Context.version)
                    kwargs = {
                        YaPackage.PackagesParameter.name: 'load/projects/validator/pkg.json',
                        YaPackage.DockerImageRepositoryParameter.name: docker_image_repository,
                        YaPackage.DockerUserParameter.name: 'LOAD',
                        YaPackage.DockerPushImageParameter.name: True,
                        YaPackage.DockerTokenVaultName.name: 'robot-lunapark-token'
                    }
                    self.build_with_yapackage(
                        YaPackage.DOCKER,
                        self.Context.version,
                        description,
                        self.Context.checkout_arcadia_path,
                        **kwargs
                    )

        # pylint: disable=no-member
        if self.Parameters.build_tank_lxc:
            with self.memoize_stage.build_tank_lxc:
                build_subtask = build_tank_lxc_container(self)
                self.Context.build_task_id = build_subtask.id
                raise sdk2.WaitTask(build_subtask, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=3600)
            release_lxc_sandbox_resource(self.Context.build_task_id)    

            with self.memoize_stage.test_sandbox_shooting:
                shoot_subtask = test_sandbox_tank_shooting(self)
                self.Context.shoot_task_id = shoot_subtask.id
                raise sdk2.WaitTask(shoot_subtask, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=3600)
            build_tank_lxc.check_test_tank_shooting_result(self.Context.shoot_task_id, self.Context.build_task_id)


        # pylint: disable=no-member
        with self.memoize_stage.deploy_validator:
            if self.Parameters.deploy_qloud_validator:
                self.deploy_to_qloud()

        with self.memoize_stage.build_tankapi_lxc:
            if self.Parameters.build_tankapi_lxc:
                self.build_lxc_with_tankapi()

        # pylint: disable=no-member
        with self.memoize_stage.check_status:
            self._wait_for_subtasks()
        self._check_subtasks_status()
