# coding: utf-8

import os
import re
import logging

from sandbox import sdk2
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.sdk2.ssh import Key
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.projects.tank.Firestarter.external_calls import retry
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.tank.build.TankDebuilder import TankDebuilder
from sandbox.projects.tank.build.DeployValidator import DeployValidator
from sandbox.projects.BuildDockerImageFromGit import DockerContentResourse
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, UbuntuRelease


class FailedCloneGit(Exception):
    pass


# noinspection PyTypeChecker
class BuildTankFromGit(sdk2.Task):
    """
       Metajob to build internal Yandex.Tank (cf. https://github.yandex-team.ru/load/yandex-tank-internal-pkg)
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 1024
        kill_timeout = 60 * 10
        client_tags = ctc.Tag.LXC

    class Context(sdk2.Task.Context):
        # pylint: disable=too-few-public-methods
        local_repo = ''
        deb_version = ''

    class Parameters(sdk2.Task.Parameters):
        # pylint: disable=too-few-public-methods

        with sdk2.parameters.Group('Tasks') as tasks_block:
            build_docker_tank = sdk2.parameters.Bool(
                'Build docker image of Yandex.Tank',
                default=True
            )
            debuild_tank = sdk2.parameters.Bool(
                'Build deb-package of Yandex.Tank',
                default=True
            )
            build_docker_validator = sdk2.parameters.Bool(
                'Build docker image of validator',
                default=True
            )
            deploy_qloud_validator = sdk2.parameters.Bool(
                'Deploy validator on qloud',
                default=True
            )
            build_tankapi_lxc = sdk2.parameters.Bool(
                'Build LXC container with tankapi for SHOOT_VIA_TANKAPI',
                default=False
            )

        with sdk2.parameters.Group('Vault parameters') as general_block:
            ssh_vault_name = sdk2.parameters.String(
                'Vault item with ssh key for git access',
                default='robot-lunapark-github-ssh'
            )
            ssh_vault_owner = sdk2.parameters.String('Vault item owner', default='LOAD')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for registry.yandex.net (vault item name)',
                default='robot-lunapark-token'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner for oauth token', default='LOAD')
            robot_login = sdk2.parameters.String('Robot login', default='robot-lunapark')

        with sdk2.parameters.Group('Yandex.Tank parameters') as tank_block:
            with sdk2.parameters.RadioGroup('Bump debian package version') as bump_version:
                bump_version.values['rc'] = bump_version.Value('rc', default=True)
                bump_version.values['stable'] = bump_version.Value('stable')
                bump_version.values['none'] = False
            source_pkg_repo = sdk2.parameters.String(
                label='Git repo with source package',
                default='git@github.yandex-team.ru:load/yandex-tank-internal-pkg.git'
            )
            revision = sdk2.parameters.String(
                label='yandextank repo revision',
                description='branch or tag or short hash',
                default='release'
            )
            tankapi_revision = sdk2.parameters.String(
                label='tankapi repo revision',
                description='branch or tag or short hash',
                default='master'
            )
            docker_tags = sdk2.parameters.List(
                label='docker registry tags',
                default=['rc']
            )
            py_version = sdk2.parameters.String(
                'Tank py_version',
                description='Tank version taken from setup.py in yandex-tank main repo',
                default=''
            )
            changelog_message = sdk2.parameters.String(
                label='Changes in new version',
                default=''
            )
            with debuild_tank.value[True]:
                ubuntu_version = sdk2.parameters.List(
                    'Distro version list', default=['precise'],
                    description='Deb package will be built for this Ubuntu versions'
                )
                create_resource = sdk2.parameters.Bool(
                    'Create resource with deb package?',
                    default=True
                )
                dupload_package = sdk2.parameters.Bool(
                    'Dupload deb package?',
                    default=True
                )

    @retry(tries=5, delay=2)
    def clone_repository(self, repo, branch='master', service='yandex_tank_internal'):
        checkout_dir = self.path(service).as_posix()
        clone_cmd = 'git clone -b {} {} {}'.format(branch, repo, checkout_dir)
        with Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            with ProcessLog(self, logging.getLogger('git_clone')) as process_log:
                status = sp.Popen(
                    clone_cmd,
                    shell=True,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr
                ).wait()
                if status != 0:
                    raise FailedCloneGit('Failed to clone git repo')
        self.Context.local_repo = checkout_dir
        logging.info('Repository %s/%s cloned to %s', repo, branch, checkout_dir)
        return checkout_dir

    def get_deb_version(self, version=''):
        """
        Gets latest version with given mask from changelog
        :return: string with full deb version
        """
        latest_version = sp.check_output(
            ["grep '{}' debian/changelog | head -1 | sed -e 's/.*[(]//' | sed -e 's/[)].*//'".format(version)],
            shell=True,
            cwd=self.Context.local_repo
        ).strip()
        logging.info('Version %s is found in changelog', latest_version)
        return latest_version

    # noinspection PyTypeChecker
    def bump_master_version(self):
        """
        Bump version built from master branch.
        Version format: major.minor.patch[.hotfix]
        :return: string with new deb_version
        """

        py_version = self.Parameters.py_version
        # noinspection PyTypeChecker
        full_deb_version = self.get_deb_version(py_version)

        if re.search(r'hot\s?fix', self.Parameters.changelog_message, re.IGNORECASE):
            # Hotfix: Take the latest record from changelog with the same patch
            # and increase hotfix number
            version = full_deb_version.split('.')
            try:
                version[3] = str(int(version[3]) + 1)
            except IndexError:
                version.append('1')
            assert len(version) == 4
            return '.'.join(version)
        else:
            # New patch: the latest patch is increased by one and is equal to py_version
            return py_version

    def bump_release_version(self):
        """
        Bump version built from release branch
        Version format: major.minor.patch~prerelease
        :return: string with new deb_version
        """
        py_version = self.Parameters.py_version
        deb_version = self.get_deb_version(py_version)

        if deb_version:
            # Prerelease for given patch is found, so increase prerelease number by one
            try:
                deb_major_version, rc = deb_version.split('~')
            except ValueError:
                # The patch equal to py_version is found in changelog.
                # It means that probably we have forgotten to bump py_version manually before build
                raise ce.TaskError(
                    'Can\'t calculate prerelease version: {} is given in setup.py, but release {} '
                    'has been already built according to debian.changelog'.format(py_version, deb_version)
                )
            return '{}~{}'.format(py_version, int(rc) + 1)
        else:
            # The first prerelease for this patch
            return '{}~1'.format(py_version)

    def add_record_to_changelog(self):
        """
        Adds new entry in changelog with bumped version and message with changes
        """
        # Change env values for changelog entries
        for env_value in ['TEMP', 'TMPDIR', 'TMP']:
            if env_value in os.environ:
                del os.environ[env_value]
        os.environ['DEBFULLNAME'] = self.Parameters.robot_login
        os.environ['EMAIL'] = '{}@yandex-team.ru'.format(self.Parameters.robot_login)

        if self.Parameters.bump_version == 'rc':
            logging.info('Calculating version for release branch...')
            new_version = self.bump_release_version()
        elif self.Parameters.bump_version == 'stable':
            logging.info('Calculating version for master branch...')
            new_version = self.bump_master_version()
        else:
            return

        self.Context.deb_version = new_version
        # Check that calculated version is not in changelog already
        if new_version == self.get_deb_version(new_version):
            logging.info('New version %s already added to changelog', new_version)
        else:
            dch_cmd = ' '.join([
                'dch -b',
                '--package "$package"',
                '--newversion "{version}"'.format(version=new_version),
                '-c debian/changelog "{message}"'.format(message=self.Parameters.changelog_message),
                '--urgency low',
                '--distribution "precise"'
            ])

            with GpgKey2(self.Parameters.oauth_vault_owner,
                         '{}-gpg-private'.format(self.Parameters.robot_login),
                         '{}-gpg-public'.format(self.Parameters.robot_login)):
                logging.info('Dch_cmd is %s', dch_cmd)
                with ProcessLog(self, logger=logging.getLogger('dch')) as process_log:
                    status = sp.Popen(
                        dch_cmd,
                        shell=True,
                        stdout=process_log.stdout,
                        stderr=process_log.stderr,
                        cwd=self.Context.local_repo
                    ).wait()
                    if status != 0:
                        raise ce.TaskError('Failed to change debian/changelog')

            with Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
                with ProcessLog(self, logger=logging.getLogger('ps1')) as process_log:
                    ps1 = sp.Popen('git add debian/changelog', shell=True,
                                   stdout=process_log.stdout, stderr=sp.STDOUT, cwd=self.Context.local_repo
                                   ).wait()
                    if ps1 != 0:
                        raise ce.TaskError('Failed to add debian/changelog')
                    ps2 = sp.Popen(
                        'git commit --message="Bump version {}" --author="{} <{}@yandex-team.ru>"'.format(
                            new_version, self.Parameters.robot_login, self.Parameters.robot_login
                        ),
                        shell=True, stdout=process_log.stdout, stderr=sp.STDOUT, cwd=self.Context.local_repo
                    ).wait()
                    if ps2 != 0:
                        raise ce.TaskError('Failed to commit')
                    ps3 = sp.Popen('git push', shell=True,
                                   stdout=process_log.stdout, stderr=sp.STDOUT, cwd=self.Context.local_repo
                                   ).wait()
                    if ps3 != 0:
                        raise ce.TaskError('Failed to push')

    def prepare_resource(self, repo_dir, branch):
        """
        Saves Dockerfile and repo around to sandbox resource for using in BuildDockerImageV6 task
        :return: sdk2.Resource
        """
        logging.info('Dockerfile and contents from {}'.format(os.path.basename(repo_dir)))
        logging.info('Branch {}'.format(branch))
        resource = DockerContentResourse(
            self,
            'Dockerfile and contents from {}, branch {}'.format(os.path.basename(repo_dir), branch),
            repo_dir
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_docker_image(self, resource, registry_tags, build_args):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: registry_tags,
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.robot_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.oauth_vault_name,
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.oauth_vault_owner,
            BuildDockerImageV6.DockerBuildArgs.name: build_args
        }
        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description='Building and publishing Docker image from Git source (by {}). \n Tags {}'.format(
                self.id, registry_tags
            ),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def debuild_tank(self):
        for distro in self.Parameters.ubuntu_version:
            debuild_tank_subtask = TankDebuilder(
                self,
                description='Build yandex-tank-internal for {}'.format(distro),
                owner=self.owner,
                git_repo=self.Parameters.source_pkg_repo,
                tank_revision=self.Parameters.revision,
                tankapi_revision=self.Parameters.tankapi_revision,
                distro_version=distro,
                vault_items_owner=self.Parameters.oauth_vault_owner,
                robot_login=self.Parameters.robot_login,
                create_resources=self.Parameters.create_resource,
                dupload=self.Parameters.dupload_package
            )
            debuild_tank_subtask.enqueue()

    def deploy_to_qloud(self):
        deploy_qloud_subtask = DeployValidator(
            self,
            description='Deploy validator to qloud',
            owner=self.owner,
            registry_url='registry.yandex.net/load/validator',
            tag=self.Parameters.py_version
        )
        self.Context.qloud_subtask = deploy_qloud_subtask.id
        deploy_qloud_subtask.enqueue()

    def build_lxc_with_tankapi(self):

        install_script = """#!/usr/bin/env bash
set -ex

### install tankapi-client
pip install -i https://pypi.yandex-team.ru/simple/ tankapi-client requests pyyaml

### check tvm2 lib
apt-cache policy  libticket-parser2-python

### clean apt archives
apt-get clean
apt-get autoclean
rm -f /var/cache/apt/*.bin
"""

        build_tankapi_lxc_subtask = SandboxLxcImage(
            self,
            owner=self.owner,
            resource_type=YANDEX_TANKAPI_LXC_CONTAINER.name,
            resource_description='Container with tankapi and other stuff for loadtests. '
                                 'Tank version {}'.format(self.Context.deb_version),
            ubuntu_release=UbuntuRelease.XENIAL,
            custom_image=True,
            custom_repos='deb http://dist.yandex.ru/yandex-xenial stable/all/\n'
                         'deb http://dist.yandex.ru/yandex-xenial stable/amd64/',
            custom_script=install_script,
            custom_packages='git software-properties-common python-pip libticket-parser2-python',
            custom_attrs={'ttl': '365'}
        ).enqueue()
        self.Context.build_lxc_subtask = build_tankapi_lxc_subtask.id

    def _wait_for_subtasks(self):
        sub_tasks = self.find()
        if sub_tasks:
            raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _check_subtasks_status(self):
        sub_tasks = self.find()
        task_errors = ''
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                task_errors += 'Subtask {} {} is failed with status {}\n'.format(task.type, task.id, task.status)
        if task_errors:
            raise ce.TaskFailure(task_errors)

    # noinspection PyTypeChecker
    def on_execute(self):
        branch = self.Parameters.revision
        registry_tags = self.Parameters.docker_tags

        if not self.Context.local_repo:
            self.clone_repository(self.Parameters.source_pkg_repo)

        # pylint: disable=no-member
        with self.memoize_stage.collect_changelog:
            if self.Parameters.bump_version:
                self.add_record_to_changelog()
            else:
                # The last record in changelog is taken if we don't want to bump new version
                self.Context.deb_version = self.get_deb_version()

        # pylint: disable=no-member
        with self.memoize_stage.build_docker_tank:
            if self.Parameters.build_docker_tank:
                tank_git_resource = self.prepare_resource(self.Context.local_repo, branch)
                self.build_docker_image(
                    resource=tank_git_resource,
                    registry_tags=['load/yandex-tank-pip:{}'.format(tag) for tag in registry_tags],
                    build_args=['BRANCH={}'.format(branch)]
                )

        # pylint: disable=no-member
        if self.Parameters.debuild_tank:
            with self.memoize_stage.debuild_tank:
                self.debuild_tank()

        # pylint: disable=no-member
        with self.memoize_stage.build_validator:
            if self.Parameters.build_docker_validator:
                self.Context.validator_dir = self.clone_repository(
                    'https://github.com/yandex-load/controlcenter',
                    'master',
                    'validator'
                )
                validator_git_resource = self.prepare_resource(self.Context.validator_dir, 'master')
                self.build_docker_image(
                    resource=validator_git_resource,
                    registry_tags=['load/validator:{}'.format(tag) for tag in registry_tags],
                    build_args=['TANK_BRANCH={}'.format(branch)]
                )
        # pylint: disable=no-member
        with self.memoize_stage.deploy_validator:
            if self.Parameters.deploy_qloud_validator:
                self.deploy_to_qloud()

        with self.memoize_stage.build_tankapi_lxc:
            if self.Parameters.build_tankapi_lxc:
                self.build_lxc_with_tankapi()

        # pylint: disable=no-member
        with self.memoize_stage.check_status:
            self._wait_for_subtasks()
        self._check_subtasks_status()
