# coding: utf-8

import os
import shutil
import logging
import fileinput

from sandbox import sdk2
import sandbox.common.errors as ce
from sandbox.sdk2.vcs.git import Git
import sandbox.common.types.task as ctt
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.BuildDockerImageFromGit import DockerContentResourse


class BuildVoltaFromGit(sdk2.Task):
    """
       Metajob to build internal Volta docker image (cf. https://github.yandex-team.ru/load/mobile-load-test)
    """
    version = ''

    class Requirements(sdk2.Requirements):
        disk_space = 512

    class Context(sdk2.Task.Context):
        version = ''

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Volta parameters') as tasks_block:
            build_docker_volta = sdk2.parameters.Bool(
                'Build docker image of Volta',
                default=True
            )
            version = sdk2.parameters.String(
                label='Volta version',
                required=False
            )
            commit = sdk2.parameters.String(
                label='Commit hash',
                description='Commit that initiated the build',
                default=''
            )
            volta_registry = sdk2.parameters.String(
                'Registry.yandex.net path',
                default='load/yandex-tank-android'
            )

        with sdk2.parameters.Group('Vault parameters') as general_block:
            vault_login = sdk2.parameters.String('Robot name for vault records', default='robot-lunapark')
            vault_owner = sdk2.parameters.String('Vault item owner', default='LOAD')

    def clone_repository(self, repo, branch, service):
        checkout_dir = self.path(service).as_posix()
        Git(repo).clone(checkout_dir, branch)
        logging.info('Repository for %s (%s) cloned to %s', service, repo, checkout_dir)
        return checkout_dir

    def prepare_resource(self, checkout_dir):
        """
        Saves Dockerfile and repo around to sandbox resource for using in BuildDockerImageV6 task
        :return: sdk2.Resource
        """

        docker_file = os.path.join(checkout_dir, 'docker/Dockerfile')
        if not os.path.exists(docker_file):
            raise ce.TaskError("Dockerfile {} not found inside the repository".format(
                docker_file)
            )
        for line in fileinput.input(docker_file, inplace=1):
            # change Volta version
            line = line.replace(
                'ARG VOLTA_BRANCH=master',
                'ARG VOLTA_BRANCH={}'.format(self.Context.version)
            ).rstrip()
            # change supervisord.conf and yandex.list location since Dockerfile should be placed in root
            line = line.replace(
                'COPY supervisord.conf /etc/supervisord.conf',
                'COPY docker/supervisord.conf /etc/supervisord.conf'
            ).rstrip()
            line = line.replace(
                'COPY ./yandex.list /etc/apt/sources.list.d/yandex.list',
                'COPY docker/yandex.list /etc/apt/sources.list.d/yandex.list'
            ).rstrip()
            print(line)

        # make sure dockerfile is named 'Dockerfile' and is located in repo root
        # so that the child task could fetch it
        shutil.move(docker_file, os.path.join(checkout_dir, 'Dockerfile'))

        resource = DockerContentResourse(
            self,
            'Dockerfile and contents from {repo}'.format(repo=os.path.basename(checkout_dir)),
            checkout_dir
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_docker_image(self, resource, registry_tags):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: registry_tags,
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.vault_login,
            BuildDockerImageV6.VaultItemName.name: '{}-token'.format(self.Parameters.vault_login),
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.vault_owner,
        }
        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description="Building and publishing Docker image from Git source (by {})".format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _check_subtasks_status(self):
        sub_tasks = self.find()
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                raise ce.TaskFailure("Subtask {} {} is failed with status {}".format(task.type, task.id, task.status))

    def on_execute(self):
        self.Context.internal_volta_dir = self.clone_repository(
            'https://github.yandex-team.ru/load/mobile-load-test.git',
            'master',
            'volta_internal'
        )

        self.Context.version = self.Parameters.version

        with self.memoize_stage.build_volta:
            volta_git_resource = self.prepare_resource(self.Context.internal_volta_dir)
            if self.Parameters.build_docker_volta:
                # noinspection PyTypeChecker
                self.build_docker_image(
                    resource=volta_git_resource,
                    registry_tags=['{}:{}'.format(self.Parameters.volta_registry, self.Context.version)],
                )

        with self.memoize_stage.check_status:
            self._check_subtasks_status()
