# coding: utf-8

import re
import os
import pytz
import time
import logging

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.projects.tank.build.BuildTankFromGit import BuildTankFromGit
from sandbox.projects.tank.build.BuildVoltaFromGit import BuildVoltaFromGit


def utc_to_local(utc_dt):
    local_tz = pytz.timezone('Europe/Moscow')
    local_dt = utc_dt.replace(tzinfo=pytz.utc).astimezone(local_tz)
    return local_tz.normalize(local_dt)


class CheckFreshTag(sdk2.Task):
    """
    Checks if there are any changes for given repository since last run
    """
    repo = ''
    repo_path = ''
    tags = {}
    child_task = None
    subtask_details = {}

    class Context(sdk2.Context):
        pass

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 512
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        # 10 minutes is enough, usually takes less than a minute
        kill_timeout = 60 * 10

        with sdk2.parameters.RadioGroup("Project to build", required=True) as project_to_build:
            project_to_build.values['tank'] = project_to_build.Value(value='tank', default=True)
            project_to_build.values['volta'] = project_to_build.Value(value='volta')
        with project_to_build.value['tank']:
            tank_repo = sdk2.parameters.String(
                'Yandex.Tank repository',
                default='https://github.com/yandex/yandex-tank/',
                required=True
            )
            with sdk2.parameters.RadioGroup("Branch to check", required=True) as branch_to_check:
                branch_to_check.values['master'] = branch_to_check.Value(value='master', default=True)
                branch_to_check.values['release'] = branch_to_check.Value(value='release')
        with project_to_build.value['volta']:
            volta_repo = sdk2.parameters.String(
                'Volta repository',
                default='https://github.com/yandex-load/volta',
                required=True
            )

        dry_run = sdk2.parameters.Bool(
            'Dry run',
            description='Only check for changes, do not start build',
            default=False
        )

        with sdk2.parameters.Output:
            run_new_build = sdk2.parameters.Bool(
                'Build started', default=False
            )

    @staticmethod
    def check_status(exit_code, message):
        if exit_code:
            raise ce.TaskError(message)

    def _run_cmd(self, cmd, logger_name, err_message, cwd=os.getcwd(), shell=True):
        status = 0
        for i in range(5):
            logging.info('Run [%s], attempt #%s', cmd, i+1)
            with ProcessLog(self, logger=logging.getLogger(logger_name)) as process_log:
                status = sp.Popen(
                    cmd,
                    shell=shell,
                    stdout=process_log.stdout,
                    stderr=process_log.stdout,
                    cwd=cwd
                ).wait()
            if not status:
                break
        else:
            self.check_status(status, err_message)

    def _get_cmd_output(self, cmd):
        return sp.check_output([cmd], shell=True, cwd=self.repo_path).strip()

    def clone_repo(self):
        self.repo_path = self.path(self.Parameters.project_to_build).as_posix()
        branch = self.Parameters.branch_to_check
        git_link = self.Parameters.tank_repo if 'tank' in self.Parameters.project_to_build else self.Parameters.volta_repo
        clone_cmd = 'git clone --progress -b {} {} {}'.format(branch, git_link, self.repo_path)
        self._run_cmd(cmd=clone_cmd, logger_name='git_clone', err_message='Failed to clone git')
        logging.info('Repository %s cloned to %s', git_link, self.repo_path)

    def find_previous_task(self):
        """
        Finds previous successful sb task with the same repo and branch
        :return: sdk2.Task object or None
        """
        return sdk2.Task.find(
            self.type,
            status=ctt.Status.SUCCESS,
            input_parameters={'project_to_build': self.Parameters.project_to_build,
                              'branch_to_check': self.Parameters.branch_to_check}
        ).order(-sdk2.Task.id).first()

    def check_existent_task(self, commit):
        """
        Checks existence of active tasks with same commit to avoid several build tasks for the same tag/commit.
        :param commit: commit hash which triggered the build.
            Tag commit in case of master branch, freshest commit in release branch.
        :return: False if no tasks found, True if found
        """
        if not commit:
            return False
        # noinspection PyCallByClass
        # noinspection PyTypeChecker
        # PyCharm inspection and assert in sdk2.Task find contradict here
        existent_task = sdk2.Task.find(
            self.child_task,
            status=(ctt.Status.Group.DRAFT, ctt.Status.Group.QUEUE, ctt.Status.Group.EXECUTE, ctt.Status.Group.WAIT),
            input_parameters={'commit': commit}
        ).order(-sdk2.Task.id).first()
        if existent_task:
            logging.info('Task %s already started for commit %s. This task will be stopped',
                         existent_task, self.subtask_details['commit'])
            return True
        else:
            return False

    def prepare_subtask_info(self):
        """
        Define the type of child task and its parameters
        """
        if 'tank' in self.Parameters.project_to_build:
            self.set_info('Collect info about tank')
            self.child_task = BuildTankFromGit
            py_version = self._get_cmd_output('grep version setup.py | grep -o "[0-9.]*"')
            logging.info('Version in setup.py is %s', py_version)
            self.subtask_details = {
                'build_docker_tank': True,
                'debuild_tank': True,
                'dupload_package': True,
                'build_docker_validator': True,
                'deploy_qloud_validator': True,
                'build_tankapi_lxc': True,
                'ubuntu_version': ['xenial', 'default'],
                'py_version': py_version
            }
            if 'release' in self.Parameters.branch_to_check:
                self.subtask_details = {
                    'prerelease': True,
                    'build_docker_tank': True,
                    'ubuntu_version': ['xenial', 'default'],
                    'build_docker_validator': False,
                    'deploy_qloud_validator': False,
                    'build_tankapi_lxc': True,
                    'py_version': py_version
                }
        elif 'volta' in self.Parameters.project_to_build:
            self.set_info('Collect info about volta')
            self.child_task = BuildVoltaFromGit
        else:
            raise ce.TaskFailure('No service selected')
        self.Context.subtask_details = self.subtask_details

    @staticmethod
    def latest_commit(commits_data):
        """
            Sorts input_data (fresh_commits) by timestamp in reverse order, i.e. from newest to oldest,
            defines freshest entry and takes its hash as an subtask parameter
        """
        try:
            assert 'ts' in commits_data.values()[0]
        except AssertionError:
            logging.warning('Error when sorting list by timestamps since no \'ts\' found.')
            logging.debug('Input: %s', commits_data)
        sorted_tuple = sorted(list(commits_data.items()), key=lambda l: l[1]['ts'], reverse=True)
        return sorted_tuple[0][1]['ts']

    def find_fresh_commits(self, control_ts):
        """
            Finds fresh commits made after given timestamp
        :param control_ts: Timestamp of last successful run for the same repo and branch
        :return: dict with commits or empty dict
        """
        fresh_commits = {}

        # We take only last change if no previous tasks were found
        # TODO: decide what to do in this case
        if not control_ts:
            fresh_log = self._get_cmd_output('git log -1 --pretty=format:"%h %s; %b%n"')
        else:
            fresh_log = self._get_cmd_output('git log --pretty=format:"%h %s; %b%n" --since {ts}'.format(ts=control_ts))

        if not fresh_log:
            return fresh_commits

        logging.info('Fresh log contents: %s', fresh_log)

        for entry in fresh_log.split('\n\n'):
            commit_hash, message = entry.split(' ', 1)
            commit_info = self._get_cmd_output('git cat-file commit {} '.format(commit_hash))
            commit_ts = set(re.findall(r'\s\d+\s', commit_info))
            if len(commit_ts) != 1:
                logging.warning('Timestamps for %s are different: %s', commit_hash, commit_ts)
            else:
                fresh_commits[commit_hash] = {'ts': commit_ts.pop().strip(' '),
                                              'message': message.rstrip().replace('\r\n', '; ')}

        raw_messages = ';'.join([messge['message'] for messge in list(fresh_commits.values())])
        changelog_message = self.collect_changelog_message(raw_messages)
        self.subtask_details['changelog_message'] = changelog_message

        return fresh_commits

    @staticmethod
    def collect_changelog_message(raw):
        """ Prepare changelog message """
        def clear_list(line):
            return ('pep8' not in line) and ('Merge' not in line) and ('Tagger: ' not in line) and \
                   (not re.match(r'v.+\d', line)) and line

        messages = raw.split(';')
        messages = [x.lstrip(' ') for x in messages]
        messages = list(filter(clear_list, messages))
        changelog_message = '; '.join(set(messages))

        return changelog_message

    def get_current_tags(self):
        """
        Collects info about tags in repository: name, md5 and commit timestamp
        :return: dict with tag name as a key and dict with tag data (hash, timestamp) as a value
        :rtype: Dict[str, Dict[str, str]]
        """
        with ProcessLog(self, 'git_tags'):
            take_tags = self._get_cmd_output('git ls-remote --tags')
            tags_data = dict(line.split('\t') for line in take_tags.rstrip().split('\n'))
            self.tags = {y.split('/')[-1]: {'md5': x} for x, y in list(tags_data.items())}

            logging.info('Tags_data: %s', self.tags)

            for tag in self.tags:
                if '^{}' not in tag:
                    tag_info = self._get_cmd_output('git cat-file commit {} '.format(self.tags[tag]['md5']))
                    logging.debug('Tag %s info: %s', tag, tag_info)

                    tag_ts = set(re.findall(r'\s\d{10}\s', tag_info))
                    if len(tag_ts) != 1:
                        logging.warning('Timestamps for %s are different: %s', tag, tag_ts)
                    self.tags[tag]['ts'] = tag_ts.pop().strip(' ')

                    try:
                        tag_message = self._get_cmd_output('git show {} --format=%B'.format(tag))
                        logging.debug('Tag %s commit message is %s', tag, tag_message)
                        if tag_message:
                            self.tags[tag]['message'] = tag_message
                    except sp.CalledProcessError:
                        pass

                else:
                    self.tags.pop(tag)

    def find_fresh_tag(self, control_ts):
        """
        Detects all the tags created later than given timestamp.
        :return: dict with fresh tags or empty dict
        """
        fresh_tags = {}

        for tag, tag_data in sorted(list(self.tags.items()), reverse=True):
            if int(tag_data['ts']) > int(control_ts):
                fresh_tags[tag] = tag_data
                logging.info('Tag %s is created after previous task run at %s', tag, control_ts)
            else:
                logging.info('Tag %s is created before previous task run at %s', tag, control_ts)

        logging.info('Fresh tags are %s', fresh_tags)

        return fresh_tags

    def run_from_master(self):
        self.get_current_tags()
        previous = self.find_previous_task()
        if previous:
            # Convert utc to local tz for timestamp
            previous_created_local = utc_to_local(previous.created)
            logging.info('Previous successful task for this repository is #%s created at %s',
                         previous.id, previous_created_local)
            previous_ts = int(time.mktime(previous_created_local.timetuple()))
            new_tags = self.find_fresh_tag(previous_ts)
        else:
            logging.info('Previous task not found')
            new_tags = self.tags

        if new_tags:
            self.set_info('Fresh tags found: {}'.format(new_tags))
            self.Parameters.description = self.Parameters.description + '. \nFresh tags found'

        if self.Parameters.dry_run or not new_tags:
            return

        for tag in new_tags:

            version = tag[1:] if tag.startswith('v') else tag
            self.subtask_details['commit'] = new_tags[tag]['md5']

            raw_message = new_tags[tag]['message'].replace('\n\n', ';')
            changelog_message = self.collect_changelog_message(raw_message)
            self.subtask_details['changelog_message'] = changelog_message

            # Check that there are no tasks started with this tag
            if not self.check_existent_task(self.subtask_details['commit']):
                logging.info('Starting %s for %s', self.child_task.__name__, tag)
                child = self.child_task(
                    self,
                    description='Started by sandbox task #{}. \nBuild from master branch, tag {}'.format(self.id, tag),
                    owner=self.owner,
                    version=version,
                    **self.subtask_details
                )
                # Do not start another build from master branch until current is finished
                child.Requirements.semaphores = ctt.Semaphores(
                    acquires=[ctt.Semaphores.Acquire(name='tank-build-from-master')],
                    release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
                )
                child.enqueue()
                if not self.Parameters.run_new_build:
                    self.Parameters.run_new_build = True

    def run_from_release(self):

        # find fresh commits
        previous = self.find_previous_task()
        if previous:
            # Convert utc to local tz for timestamp
            previous_created_local = utc_to_local(previous.created)
            logging.info('Previous successful task for this repository is #%s created at %s',
                         previous.id, previous_created_local)
            previous_ts = int(time.mktime(previous_created_local.timetuple()))
        else:
            previous_ts = 0
        new_commits = self.find_fresh_commits(previous_ts)

        if new_commits:
            self.set_info('Fresh commits found: {}'.format(new_commits))
            self.Parameters.description = self.Parameters.description + '. \nFresh commits found'
            latest_commit = self.latest_commit(new_commits)
            self.subtask_details['commit'] = latest_commit

        if self.Parameters.dry_run or not new_commits:
            return

        if not self.check_existent_task(self.subtask_details['commit']):

            logging.info('Starting %s for changes in release branch', self.child_task.__name__)
            child = self.child_task(
                self,
                description='Started by sandbox task #{}. \nBuild from release branch, commit {}'.format(self.id, self.subtask_details['commit']),
                owner=self.owner,
                **self.subtask_details
            )
            # Do not start another build from release branch until current is finished
            child.Requirements.semaphores = ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(name='tank-build-from-release')],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            )
            child.enqueue()
            if not self.Parameters.run_new_build:
                self.Parameters.run_new_build = True

    def on_execute(self):
        self.clone_repo()

        self.prepare_subtask_info()

        if 'release' not in self.Parameters.branch_to_check:
            self.run_from_master()
        else:
            self.run_from_release()
