# coding: utf-8

import os

import datetime
import pytz
import logging

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.tank.ShootViaTankapi import FailedCloneGit
from sandbox.projects.tank.Firestarter.external_calls import retry
from sandbox.projects.tank.build.TestTank import TestTank
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.projects.tank.build.BuildTankFromGit import BuildTankFromGit
from sandbox.sdk2.ssh import Key


def utc_to_local(utc_dt):
    local_tz = pytz.timezone('Europe/Moscow')
    local_dt = utc_dt.replace(tzinfo=pytz.utc).astimezone(local_tz)
    return local_tz.normalize(local_dt)


class CheckFreshTank(sdk2.Task):
    """
    Checks if there are any changes for given repository since last run
    """

    class Context(sdk2.Context):
        pass

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 512
        cores = 1
        kill_timeout = 30

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        expires = datetime.timedelta(minutes=60)

        with sdk2.parameters.Group('SSH parameters') as ssh_block:
            ssh_vault_name = sdk2.parameters.String(
                'Vault item with ssh key for git access',
                default='robot-lunapark-github-ssh'
            )
            ssh_vault_owner = sdk2.parameters.String('Vault item owner', default='LOAD')
            ssh_user = sdk2.parameters.String('SSH username', default='robot-lunapark')

        tank_repo = sdk2.parameters.String(
            'Yandex.Tank repository',
            default='https://github.com/yandex/yandex-tank/',
            required=True
        )
        tank_branch = sdk2.parameters.String(
            'Yandextank repo branch',
            default='release',
        )
        dry_run = sdk2.parameters.Bool(
            'Dry run',
            description='Only check for changes, do not start build',
            default=False
        )
        rerun = sdk2.parameters.Bool(
            'Rerun',
            description='Run even if revision is not updated',
            default=False
        )
        with sdk2.parameters.Output:
            last_tested_commit = sdk2.parameters.String(
                'Latest tested commit', default='Unknown', required=True
            )
            last_tested_commit_ts = sdk2.parameters.Integer(
                "Latest tested commit timestamp", default=42, required=True
            )
            last_tested_commit_datetime = sdk2.parameters.String(
                'Latest tested commit datetime', default='Unknown', required=True
            )

    @staticmethod
    def check_status(exit_code, message):
        if exit_code:
            raise ce.TaskError(message)

    def _run_cmd(self, cmd, logger_name, err_message, cwd=os.getcwd(), shell=True):
        status = 0
        for i in range(5):
            logging.info('Run [%s], attempt #%s', cmd, i + 1)
            with ProcessLog(self, logger=logging.getLogger(logger_name)) as process_log:
                status = sp.Popen(
                    cmd,
                    shell=shell,
                    stdout=process_log.stdout,
                    stderr=process_log.stdout,
                    cwd=cwd
                ).wait()
            if not status:
                break
        else:
            self.check_status(status, err_message)

    @staticmethod
    def _get_cmd_output(cmd, cwd):
        return sp.check_output([cmd], shell=True, cwd=cwd).strip()

    def find_previous_task(self, branch):
        """
        Finds previous finished sb task with the same repo and branch
        :return: sdk2.Task object or None
        """
        return sdk2.Task.find(
            self.type,
            status=(ctt.Status.SUCCESS, ctt.Status.FAILURE),
            input_parameters={'tank_branch': branch}
        ).order(-sdk2.Task.id).first()

    def get_previous_tested_commit_info(self, branch):
        previous_task = self.find_previous_task(branch)
        if previous_task is not None:
            previous_ts = previous_task.Parameters.last_tested_commit_ts \
                if previous_task.Parameters.last_tested_commit_ts else 0
            previous_hash = previous_task.Parameters.last_tested_commit \
                if previous_task.Parameters.last_tested_commit else ''
        else:
            previous_ts, previous_hash = 0, ''
        logging.info('Previous task: {}\nPrevious ts: {}\nPrevious hash {}'.format(previous_task,
                                                                                   previous_ts,
                                                                                   previous_hash))
        return previous_ts, previous_hash

    def get_last_revision_info(self, repo_dir):
        """
            :returns (timestamp, short hash) of latest commit
        """
        git_log_cmd = 'git log -1 --pretty=format:"%ct %h"'
        latest_ts, latest_hash = self._get_cmd_output(git_log_cmd, repo_dir).split()
        logging.info('Latest commit {} was on {}'.format(latest_hash,
                                                         self.format_ts(latest_ts)))
        return int(latest_ts), latest_hash

    @staticmethod
    def format_ts(ts):
        return datetime.datetime.fromtimestamp(int(ts)).strftime('%Y-%m-%d %H:%M:%S')

    def check_subtasks_status(self):
        sub_tasks = self.find()
        task_errors = ''
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                task_errors += 'Subtask {} {} is failed with status {}\n'.format(task.type, task.id, task.status)
        if task_errors:
            raise ce.TaskFailure(task_errors)

    @retry(tries=5, delay=2)
    def clone_repository(self, repo, branch, service='yandextank'):
        checkout_dir = self.path(service).as_posix()
        clone_cmd = 'git clone -b {} {} {}'.format(branch, repo, checkout_dir)
        with Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            with ProcessLog(self, logging.getLogger('git_clone')) as process_log:
                status = sp.Popen(
                    clone_cmd,
                    shell=True,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr
                ).wait()
                if status != 0:
                    raise FailedCloneGit('Failed to clone git repo')
        logging.info('Repository %s/%s cloned to %s', repo, branch, checkout_dir)
        return checkout_dir

    def on_execute(self):
        def set_output_params(ts, commit_hash, is_fresh):
            self.Parameters.last_tested_commit_ts = ts
            self.Parameters.last_tested_commit_datetime = self.format_ts(ts)
            self.Parameters.last_tested_commit = commit_hash
            message = "no "*(not is_fresh) + 'fresh commits ' + 'found'*is_fresh
            self.set_info(message)
            self.Parameters.description = '{dsc}\n:: {msg}'.format(dsc=self.Parameters.description,
                                                                   msg=message)

        with self.memoize_stage.only_one:
            running = sdk2.Task.find(self.type,
                                     status=tuple(ctt.Status.Group.EXECUTE) +
                                            tuple(ctt.Status.Group.WAIT)
                                     ).order(-sdk2.Task.id).limit(5)
            running = [task.id for task in running if task.id != self.id]
            if running:
                logging.info('Running task(s) found {}'.format(running))
                logging.info('Quiting')
                self.Parameters.description = 'Another task(s) already running: {}'.format(running)
                raise Exception('Another task(s) already running: {}'.format(running))

        with self.memoize_stage.check_repo:
            # clone repo
            repo_dir = self.clone_repository(self.Parameters.tank_repo, self.Parameters.tank_branch)
            # check last commit
            previous_ts, previous_hash = self.get_previous_tested_commit_info(self.Parameters.tank_branch)
            latest_ts, latest_hash = self.get_last_revision_info(repo_dir)
            self.Context.fresh_commits_found = latest_ts > previous_ts
            set_output_params(latest_ts, latest_hash, self.Context.fresh_commits_found)
            # save build params
            self.Context.py_version = self._get_cmd_output("grep version setup.py | awk -F\\' '{print $2}'", repo_dir)
            self.Context.revision = latest_hash

        with self.memoize_stage.test_tank:
            if self.Context.fresh_commits_found or self.Parameters.rerun:
                test_tank_subtask = TestTank(self,
                                             description='Autotest fresh commit',
                                             branch=self.Context.revision)
                subtask_id = test_tank_subtask.id
                test_tank_subtask.enqueue()
                raise sdk2.WaitTask([subtask_id],
                                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                    wait_all=True)
        self.check_subtasks_status()

        with self.memoize_stage.release_tank:
            if self.Context.fresh_commits_found or self.Parameters.rerun:
                if self.Parameters.tank_branch in 'master':
                    build_params = {
                        'build_docker_tank': True,
                        'debuild_tank': True,
                        'dupload_package': True,
                        'build_docker_validator': True,
                        'deploy_qloud_validator': True,
                        'build_tankapi_lxc': True,
                        'ubuntu_version': ['xenial', 'default'],
                        'py_version': self.Context.py_version,
                        'bump_version': 'stable',
                        'revision': self.Context.revision,
                        'docker_tags': ['latest', str(self.Context.py_version)]
                    }
                else:
                    build_params = {
                        'bump_version': 'rc',
                        'build_docker_tank': True,
                        'ubuntu_version': ['xenial', 'default'],
                        'build_docker_validator': False,
                        'deploy_qloud_validator': False,
                        'build_tankapi_lxc': True,
                        'py_version': self.Context.py_version,
                        'revision': self.Context.revision,
                        'docker_tags': ['rc']
                    }
                build_subtask = BuildTankFromGit(
                    self,
                    description='Started by sandbox task #{}.'.format(self.id),
                    owner=self.owner,
                    **build_params
                )
                build_subtask_id = build_subtask.id
                build_subtask.enqueue()
                raise sdk2.WaitTask([build_subtask_id],
                                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                    wait_all=True)

        self.check_subtasks_status()
