import json
import logging

from sandbox import sdk2
from sandbox.projects.sandbox_ci.qloud.qloud import QloudApi
from sandbox.projects.sandbox_ci.utils.request import send_request


class DeployValidator(sdk2.Task):
    """
        Deploys tank-hub validator to qloud
    """
    qloud_env_conf = {}

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512

    class Parameters(sdk2.Task.Parameters):
        env_dump = sdk2.parameters.Resource(
            'Json with environment dump',
            required=False
        )
        registry_url = sdk2.parameters.String(
            'Registry url with docker image',
            default='registry.yandex.net/load/validator'
        )
        tag = sdk2.parameters.String(
            'Registry tag to deploy',
            default='latest'
        )
        qloud_token = sdk2.parameters.String(
            'Vault name with qloud token',
            default='robot-lunapark-token'
        )

    def get_docker_image_hash(self):
        url = 'https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={registry_url}&tag={tag}'.format(
            registry_url=self.Parameters.registry_url,
            tag=self.Parameters.tag
        )
        res = send_request('get', url, headers={}, verify=False)
        if res.ok:
            d = json.loads(res.text)
            return d['hash']
        else:
            return None

    def _get_qloud_environment_config(self):

        if self.Parameters.env_dump:
            env_dump = sdk2.resource.ResourceData(self.Parameters.env_dump)
            logging.info('Dump is taken from resource %s', self.Parameters.env_dump.id)
        else:
            env_dump = sdk2.resource.ResourceData(sdk2.Resource.find(id=512396841).first())
            logging.info('Dump is taken from resource #512396841')

        conf = env_dump.path.read_bytes()
        self.qloud_env_conf = json.loads(conf)

        # TODO: add deploy_component to qloud client
        for num, component in enumerate(self.qloud_env_conf['components']):
            logging.info("Update qloud target image repository and hash")
            component['properties']['repository'] = '{url}:{tag}'.format(
                url=self.Parameters.registry_url,
                tag=self.Parameters.tag
            )
            component['properties']['hash'] = self.get_docker_image_hash()
            self.qloud_env_conf['comment'] += 'Update {component} from {image}, hash {md5_hash}'.format(
                component=component['componentName'],
                image=self.Parameters.registry_url,
                md5_hash=component['properties']['hash']
            )
            logging.info("Target image for deploy: %s; hash: %s",
                         component['properties']['repository'], component['properties']['hash'])

    def on_execute(self):
        qloud_auth_token = sdk2.Vault.data(self.Parameters.qloud_token)
        qloud_client = QloudApi(qloud_auth_token)

        self._get_qloud_environment_config()
        logging.debug('Config is %s', self.qloud_env_conf)

        qloud_client.upload_environment(self.qloud_env_conf)
