""" https://st.yandex-team.ru/LUNAPARK-3194 """
import os

from sandbox import sdk2
from sandbox.sandboxsdk.ssh import Key
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.projects.tank.load_resources.resources import LUNAPARK_DEB_PACKAGE
from sandbox.projects.tank.build.base_load_debuilder import BaseLoadDebuilder


class LunaparkDebuilder(sdk2.Task, BaseLoadDebuilder):
    """
    Builds and uploads deb-package for Lunapark and Overload
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.RadioGroup('Project to build', required=True) as project_to_build:
            project_to_build.values['lunapark'] = project_to_build.Value(value='lunapark', default=True)
            project_to_build.values['overload'] = project_to_build.Value(value='overload')
        with sdk2.parameters.Group('Build parameters'):
            distro_version = sdk2.parameters.String('Ubuntu version', required=True, default='precise')
            source_repo_branch = sdk2.parameters.String('Repo branch', required=False, default='')
            vault_items_owner = sdk2.parameters.String('Vault items owner', default='LOAD')
            robot_login = sdk2.parameters.String('Robot login (without @yandex-team.ru)', default='robot-lunapark')
        with sdk2.parameters.Group('Tasks'):
            create_resources = sdk2.parameters.Bool('Create resources?', required=True, default=True)
            dupload = sdk2.parameters.Bool('Dupload deb package?', required=True, default=False)
            build_porto_layer = sdk2.parameters.Bool('Build porto layer?', required=True, default=False)

    def set_attrs(self):
        self.robot_login = self.Parameters.robot_login
        self.vault_items_owner = self.Parameters.vault_items_owner
        self.github_ssh_key = Key(self, self.vault_items_owner, '{}-github-ssh'.format(self.robot_login))
        self.cacos_ssh_key = Key(self, self.vault_items_owner, '{}-ssh'.format(self.robot_login))
        self.gpg_key = GpgKey2(self.vault_items_owner, '{}-gpg-private'.format(self.robot_login),
                               '{}-gpg-public'.format(self.robot_login))

        bb_token = sdk2.Vault.data('robot-lunapark-bb-oauth')
        if 'lunapark' in self.Parameters.project_to_build:
            self.source_repo_link = 'https://x-oauth-token:{}@bb.yandex-team.ru/scm/load/lunapark.git'.format(bb_token)
            self.source_repo_branch = self.Parameters.source_repo_branch or 'master'
        else:
            self.source_repo_link = 'https://x-oauth-token:{}@bb.yandex-team.ru/scm/load/overload.git'.format(bb_token)
            self.source_repo_branch = self.Parameters.source_repo_branch or 'public'

        self.local_path = self.path()
        self.local_hooks_dir = ''

        self.cacos_repo = 'load'

        self.create_resources = self.Parameters.create_resources
        self.output_resource_type = LUNAPARK_DEB_PACKAGE

        self.output_ubuntu_version = self.Parameters.distro_version
        self.dupload_result_message = ''

    def modify_source(self):
        rules_file = os.path.join(self.local_repo_dir, 'debian/rules')
        data = '#!/usr/bin/make -f\n%:\n\tdh $@ -v\noverride_dh_update_autotools_config:\n'
        with open(rules_file, 'w') as rules:
            rules.write(data)

    def build_porto_layer(self):
        subtask = None

        if 'lunapark' in self.Parameters.project_to_build:
            subtask = sdk2.Task['BUILD_PORTO_LAYER'](
                self,
                description='Lunapark Front Porto-layer, ver. {}'.format(self.output_deb_version),
                compress='tar.xz',
                layer_name='lunapark-front',
                layer_type='PORTO_LAYER_LUNAPARK_FRONT',
                parent_layer=236752924,
                script='arcadia:/robots/trunk/genconf/PORTOVM/lunapark/lunapark_front_production.sh',
                start_os=True
            ).enqueue()
        elif 'overload' in self.Parameters.project_to_build:
            subtask = sdk2.Task['BUILD_PORTO_LAYER'](
                self,
                description='Overload Front Porto-layer, ver. {}'.format(self.output_deb_version),
                compress='tar.xz',
                layer_name='overload-front',
                layer_type='PORTO_LAYER_OVERLOAD_FRONT',
                parent_layer=547491851,
                script='arcadia:/robots/trunk/genconf/PORTOVM/overload/overload_front_production.sh',
                start_os=True
            ).enqueue()

        if subtask:
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_prepare(self):
        self.set_attrs()
        self.Context.local_repo_dir = self.git_clone(self.Parameters.source_repo_branch)

    def on_execute(self):
        self.set_attrs()
        self.local_repo_dir = self.Context.local_repo_dir
        self.modify_source()
        self.Parameters.description = self.set_package_info()

        with self.memoize_stage.build:
            self.build()

        with self.memoize_stage.create_resource:
            if self.Parameters.create_resources:
                self.create_resource(self.local_result_dir)

        with self.memoize_stage.dupload:
            if self.Parameters.dupload:
                self.dupload()
                if self.check_packages_in_dist():
                    self.set_info(self.dupload_result_message)
                # else:
                #     raise ce.TaskFailure(self.dupload_result_message)
                self.dmove('testing')
                self.dmove('stable')

        with self.memoize_stage.build_porto_layer():
            if self.Parameters.build_porto_layer:
                self.build_porto_layer()
