import os
import fileinput

from sandbox import sdk2
from sandbox.sandboxsdk.ssh import Key
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.projects.tank.load_resources.resources import TANK_DEB_PACKAGE
from sandbox.projects.tank.build.base_load_debuilder import BaseLoadDebuilder


class TankDebuilder(sdk2.Task, BaseLoadDebuilder):
    """
        Builds and uploads deb-package with yandex-tank-internal
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Git parameters") as git_block:
            git_repo = sdk2.parameters.String(
                'Git repository', required=True,
                default_value='https://github.yandex-team.ru/load/yandex-tank-internal-pkg'
            )
            tank_revision = sdk2.parameters.String('Specify tank branch or tag or commit hash', default='master')
            tankapi_revision = sdk2.parameters.String('Specify tankapi branch or tag or commit hash', default='master')
        distro_version = sdk2.parameters.String('Ubuntu version', required=True, default='precise')
        vault_items_owner = sdk2.parameters.String('Vault items owner', required=True, default="LOAD")
        robot_login = sdk2.parameters.String('Robot login (without @yandex-team.ru)', required=True,
                                             default='robot-lunapark')
        create_resources = sdk2.parameters.Bool('Create resources?', required=True, default=True)
        dupload = sdk2.parameters.Bool('Dupload deb package?', required=True, default=False)

    def set_attrs(self):
        self.robot_login = self.Parameters.robot_login
        self.vault_items_owner = self.Parameters.vault_items_owner
        self.github_ssh_key = Key(self, self.vault_items_owner, '{}-github-ssh'.format(self.robot_login))
        self.cacos_ssh_key = Key(self, self.vault_items_owner, '{}-ssh'.format(self.robot_login))
        self.gpg_key = GpgKey2(self.vault_items_owner, '{}-gpg-private'.format(self.robot_login),
                               '{}-gpg-public'.format(self.robot_login))

        self.source_repo_link = self.Parameters.git_repo
        self.source_repo_branch = 'master'  # if (self.Parameters.distro_version == 'default' or
        #                                        self.Parameters.distro_version == 'precise')\
        #     else self.Parameters.distro_version

        self.local_path = self.path()

        self.create_resources = self.Parameters.create_resources
        self.output_resource_type = TANK_DEB_PACKAGE

        if 'default' in self.Parameters.distro_version:
            self.output_ubuntu_version = 'precise'
        else:
            self.output_ubuntu_version = self.Parameters.distro_version
        self.cacos_repo = 'load-{}'.format(self.output_ubuntu_version)

        self.dupload_result_message = ''

    def modify_source(self):

        hook_prepare = os.path.join(self.local_hooks_dir, 'D10prepare')
        rules_file = os.path.join(self.local_repo_dir, 'debian/rules')
        reqs_file = os.path.join(self.local_repo_dir, 'requirements.txt')
        control_file = os.path.join(self.local_repo_dir, 'debian/control')
        preinst_file = os.path.join(self.local_repo_dir, 'debian/yandex-tank-internal.preinst')
        postinst_file = os.path.join(self.local_repo_dir, 'debian/yandex-tank-internal.postinst')

        # Remove python 2.7.11 req
        for line in fileinput.input(control_file, inplace=1):
            line = line.replace('python2.7 (>=2.7.11),', 'python2.7,').rstrip()
            print(line)

        # add python-pip
        for line in fileinput.input(hook_prepare, inplace=1):
            line.replace('pip install', 'pip install -vvv ').rstrip()
            old_deps = 'apt-get install -y ca-certificates python2.7 python-minimal git'
            old_wget = 'wget -O get-pip.py https://proxy.sandbox.yandex-team.ru/472584907'
            new_wget = 'wget https://bootstrap.pypa.io/get-pip.py || true'
            line = line.replace(old_deps, old_deps + ' libssl-dev openssl').rstrip()
            line = line.replace(old_wget, new_wget).rstrip()
            print(line)

        # Build from specified branch
        for line in fileinput.input(reqs_file, inplace=1):
            line = line.replace('yandex-tank/tarball/master',
                                'yandex-tank/tarball/{}'.format(self.Parameters.tank_revision)).rstrip()
            line = line.replace('old-yatank-internal-api.git@master',
                                'old-yatank-internal-api.git@{}'.format(self.Parameters.tankapi_revision)).rstrip()
            print(line)

        if 'trusty' in self.Parameters.distro_version:
            # Change sources.list
            for line in fileinput.input(hook_prepare, inplace=1):
                line = line.replace("$(ARCH)/", "$(ARCH)/' > /etc/apt/sources.list.d/load.list;").rstrip()
                if 'load.dist.yandex.ru' not in line:
                    print(line)

        if 'xenial' in self.Parameters.distro_version:

            # Change sources list and add dh-systemd as a dependency
            for line in fileinput.input(hook_prepare, inplace=1):
                old_deps = 'apt-get install -y ca-certificates python2.7 python-minimal git'
                line = line.replace('precise', 'xenial').rstrip()
                line = line.replace(old_deps, old_deps + ' dh-systemd').rstrip()
                print(line)

            # Modify debhelper calls for systemd
            with open(rules_file, 'r') as rules:
                old_rules = rules.read()
            new_rules = old_rules.replace('dh $@', 'dh $@ -v --with systemd')
            new_rules = '\n'.join([new_rules,
                                   'override_dh_systemd_enable:',
                                   '\tdh_systemd_enable --name=tankapi',
                                   '\tdh_systemd_enable --name=bdk',
                                   # Dirty hack, should be overwritten
                                   'override_dh_update_autotools_config:'])
            with open(rules_file, 'w') as rules:
                rules.write(new_rules)

            # Check on preinst if there is upstart at all to stop tankapi
            stop_tankapi = 'stop tankapi || echo No tank API was running'
            check_stop_tankapi = 'if [ "$(which stop)" != "" ]; then\n    {}\nfi'.format(stop_tankapi)
            for line in fileinput.input(preinst_file, inplace=1):
                line = line.replace(stop_tankapi, check_stop_tankapi).rstrip()
                print(line)

            # Rewrite postinst without upstart
            def clear_upstart(entry):
                return (('upstart' not in entry) and ('start tankapi' not in entry) and ('start bdk' not in entry)
                        and ('stop bdk' not in entry) and ('stop tankapi' not in entry))
            with open(postinst_file, 'r') as post_inst:
                old_post_inst = post_inst.readlines()
            new_post_inst = list(filter(clear_upstart, old_post_inst))
            with open(postinst_file, 'w') as post_inst:
                post_inst.write(''.join(new_post_inst))

    def on_prepare(self):
        self.set_attrs()
        self.Context.repo_dir = self.git_clone()

    def on_execute(self):
        self.set_attrs()
        self.local_repo_dir = self.Context.repo_dir
        self.local_hooks_dir = self.local_repo_dir + '/hooks.d'
        self.modify_source()
        self.Parameters.description = self.set_package_info()

        # with self.memoize_stage.debuild:
        self.build()

        with self.memoize_stage.create_resource:
            if self.Parameters.create_resources:
                self.create_resource(self.local_result_dir)

        if self.Parameters.dupload:
            self.dupload()
            if self.check_packages_in_dist():
                self.set_info(self.dupload_result_message)
            else:
                raise ce.TaskFailure(self.dupload_result_message)
