import os
import logging
import tempfile

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.ssh import Key
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.tank.load_resources.resources import TANK_PY_PACKAGE


class UploadTankToPypi(sdk2.Task):
    """
    Builds python package for yandextank from repo and uploads it to global pypi
    """

    local_repo_dir = ''
    package_name = ''
    py_version = ''

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Git parameters'):
            git_link = sdk2.parameters.String(label='Git repo with source package',
                                              default='https://github.com/yandex/yandex-tank.git')
            git_branch = sdk2.parameters.String(label='Git branch', default='master')
        with sdk2.parameters.Group('Vault parameters'):
            vault_pypi_user = sdk2.parameters.String('Vault owner', default='LOAD')
            vault_access_key_record = sdk2.parameters.String('Pypi access-key record in vault',
                                                             default='')
            vault_secret_key_record = sdk2.parameters.String('Pypi secret-key record in vault',
                                                             default='')
            ssh_vault_name = sdk2.parameters.String(
                'Vault item with ssh key for git access',
                default='robot-lunapark-github-ssh'
            )
            ssh_vault_owner = sdk2.parameters.String('Owner of ssh key with git access', default='LOAD')

    def _run_cmd(self, cmd, logger_name, err_message, cwd=os.getcwd(), shell=True):
        """
        Run given commandline in subprocess.Popen and check exit code.
        All stdout and stderr redirected to common log
        """
        with ProcessLog(self, logger=logging.getLogger(logger_name)) as process_log:
            status = sp.Popen(
                cmd,
                shell=shell,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()
        if not status:
            logging.error(err_message)
            # raise ce.TaskError(err_message)

    def _get_cmd_output(self, cmd):
        return sp.check_output([cmd], shell=True, cwd=self.local_repo_dir).strip()

    def create_resource(self, source):
        resource = TANK_PY_PACKAGE(
            self,
            'Package {}={}'.format(self.package_name, self.py_version),
            source
        )
        resource.version = self.py_version
        resource.package = self.package_name
        sdk2.ResourceData(resource).ready()

    def git_clone(self):
        root_dir = tempfile.mkdtemp()
        checkout_dir = root_dir + '/repo'
        clone_cmd = 'git clone --progress -b {} {} {}'.format(
            self.Parameters.git_branch, self.Parameters.git_link, checkout_dir)

        try:
            with Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
                self._run_cmd(cmd=clone_cmd, logger_name='git_clone',
                              err_message='Failed to clone git')
        except ce.VaultNotFound:
            self._run_cmd(cmd=clone_cmd, logger_name='git_clone',
                          err_message='Failed to clone git')
        logging.info('Repository for %s:%s cloned to %s',
                     self.Parameters.git_link, self.Parameters.git_branch, checkout_dir)
        return checkout_dir

    def on_execute(self):
        self.local_repo_dir = self.git_clone()

        self._run_cmd('python setup.py sdist',
                      cwd=self.local_repo_dir, logger_name='pyi_build', err_message='Can\'t build pypi')
        self.set_info('Successfully built yandextank for PyPI')

        dist_dir = self.local_repo_dir + '/dist'
        self.package_name = self._get_cmd_output('grep "name=" setup.py | cut -d "\'" -f2')
        self.py_version = self._get_cmd_output('grep "version" setup.py | cut -d "\'" -f2')
        self.create_resource(dist_dir)
