import os
import time
import tempfile
import logging

from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.resource_types import DEB_PACKAGE_TAR
from sandbox.projects.tank.load_resources.resources import TANK_PBUILDER_IMAGE
from sandbox.projects.market.sre.MarketDebuilder.dist import Dist


class BaseLoadDebuilder(object):
    """
        Base class for debuild tasks used in LOAD projects (tank/lunapark/overload/etc).
        Should not be initiated
    """

    def __init__(self):
        self.task = sdk2.Task

        self.robot_login = ''
        self.vault_items_owner = ''
        self.github_ssh_key = None
        self.cacos_ssh_key = None
        self.gpg_key = None

        self.source_repo_link = ''
        self.source_repo_branch = ''

        self.local_path = os.path.dirname(os.path.abspath(__file__))
        self.local_result_dir = ''
        self.local_repo_dir = ''
        self.local_hooks_dir = ''

        self.create_resources = True
        self.task_description = ''
        self.cacos_repo = ''
        self.output_package = ''
        self.output_deb_version = ''
        self.output_resource_type = DEB_PACKAGE_TAR
        self.output_ubuntu_version = ''
        self.dupload_result_message = ''

    # @retry(tries=5, delay=10)
    def _run_cmd(self, cmd, logger_name, err_message, cwd=os.getcwd(), shell=True):
        """
        Run given commandline in subprocess.Popen and check exit code.
        All stdout and stderr redirected to separate log
        :param cmd: Bash commandline
        :param cwd: Directory where cmd should be run
        :param logger_name: Name of log
        :param err_message: error message that will be risen if exit code is not 0
        :param shell: switches on shell arg for Popen
        :return:
        """
        with ProcessLog(self, logger=logging.getLogger(logger_name)) as process_log:
            status = sp.Popen(
                cmd,
                shell=shell,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()
        if not status:
            logging.error(err_message)
            # raise ce.TaskError(err_message)

    def _get_cmd_output(self, cmd):
        return sp.check_output([cmd], shell=True, cwd=self.local_repo_dir).strip()

    # @retry(tries=5, delay=2)
    def git_clone(self, branch=None):
        branch = self.source_repo_branch if branch is None else branch
        root_dir = tempfile.mkdtemp()
        checkout_dir = root_dir + '/repo'
        clone_cmd = 'git clone --progress -b {} {} {}'.format(
            branch, self.source_repo_link, checkout_dir)

        try:
            with self.github_ssh_key:
                self._run_cmd(cmd=clone_cmd, logger_name='git_clone',
                              err_message='Failed to clone git')
        except ce.VaultNotFound:
            self._run_cmd(cmd=clone_cmd, logger_name='git_clone',
                          err_message='Failed to clone git')
        logging.info('Repository for %s:%s cloned to %s', self.source_repo_link, self.source_repo_branch, checkout_dir)
        return checkout_dir

    def set_package_info(self, changelog=''):

        self.output_deb_version = self._get_cmd_output(
            cmd="head -n 1 debian/changelog | sed -e 's/.*[(]//' | sed -e 's/[)].*//'")
        self.output_package = self._get_cmd_output(
            cmd="grep 'Package' debian/control | sed 's/Package: //'")
        changes = changelog or self._get_cmd_output(
            cmd="dpkg-parsechangelog | sed -n '/Changes:/,$p' | tail -n +4 | sed 's/^..//'")

        return "{}={}\n{}\n{}".format(
            self.output_package,
            self.output_deb_version,
            self.output_ubuntu_version,
            changes
        )

    @staticmethod
    def _get_basetgz(ubuntu_version='precise'):
        """
            Sync sandbox resource with pbuilder image
        self._get_basetgz(Str) -> Str
        :param ubuntu_version: precise, trusty or xenial
        :return: local path with pbuilder image
        """
        images = sdk2.Resource[TANK_PBUILDER_IMAGE].find(attrs={'release': ubuntu_version})
        pbimage = images.order(-sdk2.Resource.id).first()
        pbimage_data = sdk2.ResourceData(pbimage)
        pbimage_path = pbimage_data.path
        logging.info('pbimage %s with release %s synced to %s', pbimage.id, ubuntu_version, pbimage_path)
        return str(pbimage_path)

    def _install_pbuilder(self):
        pbuilder_cmd = "export DEBIAN_FRONTEND=noninteractive && " \
                       "apt-get update && apt-get install " \
                       "pbuilder -q0 --assume-yes --force-yes -o Dpkg::Options::='--force-confdef'"
        logging.info('Install pbuilder')

        self._run_cmd(cmd=pbuilder_cmd, logger_name='install_pbuild',
                      cwd=self.local_repo_dir, err_message='Failed to install pbuilder')

    def build(self):
        """
        Build deb package with pdebuilder
        """

        self._install_pbuilder()

        result_dir = str(self.local_path) + '/result_deb'
        try:
            os.mkdir(result_dir)
        except OSError:
            raise ce.TaskError('Failed to make result dir')
        self.local_result_dir = result_dir

        build_cmd = ' '.join([
            'pdebuild',
            '--use-pdebuild-internal',
            '--auto-debsign --debsign-k {}'.format(self.robot_login),
            '--buildresult {}'.format(self.local_result_dir),
            '--',
            # '--allow-untrusted',
            '--basetgz {}'.format(self._get_basetgz(self.output_ubuntu_version)),
            '--buildplace {}'.format(self.local_repo_dir),
            '--aptcache {}'.format(self.local_repo_dir),
        ])
        if self.local_hooks_dir:
            build_cmd += ' --hookdir {}'.format(self.local_hooks_dir)

        with self.gpg_key:
            self._run_cmd(cmd=build_cmd, cwd=self.local_repo_dir,
                          logger_name='pbuilder', err_message='Failed to build deb')

    def create_resource(self, source):
        resource = self.output_resource_type(
            self,
            'Package {}={}'.format(self.output_package, self.output_deb_version),
            source
        )
        resource.version = self.output_deb_version
        resource.package = self.output_package
        resource.distro = self.output_ubuntu_version
        sdk2.ResourceData(resource).ready()

    def dupload(self):
        """
            Dupload deb package to Cacos
        """
        dupload_conf = {
            self.cacos_repo: {
                'fqdn': '{}.dupload.dist.yandex.ru'.format(self.cacos_repo),
                'method': 'scpb',
                'incoming': '/repo/{}/mini-dinstall/incoming/'.format(self.cacos_repo),
                'dinstall_runs': 0,
                'login': self.robot_login,
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            }
        }

        upload_cmd = "set -x; chmod 0644 {abs_path}/* && " \
                     "dupload --force --to {repo} " \
                     "{abs_path}/*.changes".format(abs_path=self.local_result_dir, repo=self.cacos_repo)

        with DebRelease(dupload_conf):
            with self.cacos_ssh_key:
                self._run_cmd(cmd=upload_cmd, cwd=self.local_result_dir,
                              logger_name='dupload', err_message='Failed to dupload')

    def check_packages_in_dist(self):
        wait_repo_timeout = 300
        start_time = time.time()
        packages_were_found = False
        dist = Dist()

        # Check if package appears in Cacos every 10 seconds for 5 minutes
        while not packages_were_found and time.time() < start_time + wait_repo_timeout:
            time.sleep(10)
            logging.debug('Checking packages in repo ...')
            packages_were_found = dist.if_exist(self.output_package, self.output_deb_version,
                                                self.cacos_repo)

        if packages_were_found:
            self.dupload_result_message = 'Package {name}={version} ({distro}) successfully uploaded to {cacos}'.format(
                name=self.output_package, version=self.output_deb_version, distro=self.output_ubuntu_version,
                cacos=self.cacos_repo
            )
            return True
        else:
            self.dupload_result_message = 'Packages are not found in {} after {} secs from dupload'.format(
                self.cacos_repo, wait_repo_timeout
            )
            return False

    def dmove(self, to_branch):

        dmove_cmd = 'ssh dupload.dist.yandex.ru dmove {repo} {to_branch} {package} {version}'.format(
            repo=self.cacos_repo, to_branch=to_branch,
            package=self.output_package, version=self.output_deb_version)

        with self.cacos_ssh_key:
            self._run_cmd(cmd=dmove_cmd, logger_name='dmove', err_message='Failed to dmove to {}'.format(to_branch))
