# coding: utf-8
import logging
import requests

from sandbox import sdk2
from sandbox.projects.tank.common import retry
from sandbox.sdk2.service_resources import SandboxTasksBinary


LUNAPARK = 'https://lunapark.yandex-team.ru'


class ShootingRegression(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024 * 2
        privileged = True

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Context(sdk2.Task.Context):
        regression_link = ''

    class Parameters(sdk2.Task.Parameters):

        shooting_id = sdk2.parameters.Integer('ID of the shooting in the lunapark', default_value=0)

        with sdk2.parameters.Output:
            status = sdk2.parameters.Bool('Exit code', default_value=False)
            regression = sdk2.parameters.String('Regression URL', default_value='')

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(
            attrs={
                'task_name': 'ShootingRegression',
                'released': 'stable'
            }).first()

    def on_execute(self):
        result = self.check_sla()
        self.Parameters.status = result['resolution'] if result.get('success', 0) else False
        self.Parameters.regression = self.get_regression_url()

        if not self.Parameters.status:
            logging.error('FAILED KPIS:')
            for failed_kpi in result.get('failed_kpis', {}).values():
                logging.error('%s: %s', failed_kpi.get('dsc', ''), failed_kpi.get('failed_values', {}))

    def check_sla(self):
        url = '{}/api/job/{}/sla.json'.format(LUNAPARK, self.Parameters.shooting_id)
        logging.info('Call %s', url)
        try:
            return requests.get(url).json()[0]
        except (requests.RequestException, TypeError, IndexError):
            logging.error('Failed to check SLA', exc_info=True)

    def get_regression_url(self):
        summary_url = '{}/api/job/{}/summary.json'.format(LUNAPARK, self.Parameters.shooting_id)
        try:
            summary = requests.get(summary_url).json()[0]
            project = summary['project']
            component = summary['component']
            return '{}/regress/{}/{}'.format(LUNAPARK, project, component)
        except (KeyError, TypeError, requests.RequestException):
            logging.debug('Failed to get regression url', exc_info=True)
