import json
import logging

import time

from sandbox import sdk2
from sandbox.projects.sandbox_ci.qloud.qloud import QloudApi
from sandbox.projects.sandbox_ci.utils.request import send_request


class DeployOnQloud(sdk2.Task):
    """
        Deploys docker images to qloud
    """
    qloud_env_conf = {}

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512

    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            'Qloud environment id ( {project}.{application}.{environment} )',
            required=True
        )
        qloud_token = sdk2.parameters.String(
            'Vault name with qloud token',
            default='robot-lunapark-token'
        )
        components_to_deploy = sdk2.parameters.Dict(
            'Components to deploy along with their repository url:tag; '
            'e.g. "tank": "registry.yandex.net/load/yandex-tank-pip:latest"',
            required=True
        )

    def get_docker_image_hash(self, registry_url, tag):
        url = 'https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={registry_url}&tag={tag}'.format(
            registry_url=registry_url,
            tag=tag
        )
        res = send_request('get', url, headers={}, verify=False)
        if res.ok:
            d = json.loads(res.text)
            return d['hash']
        else:
            return None

    def on_execute(self):
        qloud_auth_token = sdk2.Vault.data(self.Parameters.qloud_token)
        qloud_client = QloudApi(qloud_auth_token)

        qloud_env_dump = qloud_client.dump_environment(self.Parameters.environment)

        for component in qloud_env_dump['components']:
            component_name = component['componentName']
            repo = self.Parameters.components_to_deploy.get(component_name)
            if repo is not None:
                logging.info('Component {} is discovered and doomed to deploy'.format(component_name))
                try:
                    url, tag = repo.split(':')
                except ValueError:
                    raise Exception('Invalid repository name: {}\nShould match {{url}}:{{tag}}'.format(repo))
                image_hash = self.get_docker_image_hash(url, tag)
                logging.info("Image to deploy: {}\nhash: {}".format(repo, image_hash))
                component['properties']['repository'] = repo
                component['properties']['hash'] = image_hash
                self.Parameters.components_to_deploy.pop(component_name)
        for undefined_component in self.Parameters.components_to_deploy:
            logging.warning('Component {} is not defined in Qloud environment dump '
                            'and not going to be deployed'.format(undefined_component))

        logging.debug('Qloud env config to be deployed:\n%s', qloud_env_dump)

        qloud_client.upload_environment(qloud_env_dump)
        while not qloud_client.is_environment_deployed(self.Parameters.environment):
            logging.info("Waiting for deploy")
            time.sleep(8)
