import glob
import copy
import json
import os
import os.path
import tarfile

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.common import dolbilka
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.projects.tank import offline

TANK_CPU_LIMIT = "cpuset:tank_cpu_limit"

TANK_GROUP = "Tank settings"
LUNAPARK_GROUP = "Lunapark settings"
PHANTOM_GROUP = "Phantom settings"
AUTOSTOP_GROUP = "Autostop settings"
OFFLINE_GROUP = "Offline report settings"

_JOBNO_FILE = "jobno.txt"
_PHOUT_FILE = "shoot.tsv"
_PLUGIN_KEY_PREFIX = "plugin_"


# Lunapark plugin


class LunaparkAddressParameter(parameters.SandboxStringParameter):
    name = "lunapark_address"
    description = "Lunapark URL"
    default_value = "https://lunapark.yandex-team.ru/"
    group = LUNAPARK_GROUP


class LunaparkTaskParameter(parameters.SandboxStringParameter):
    name = "tank_task_name"
    description = "Open task name with load testing attribute"
    default_value = ""
    required = True
    group = LUNAPARK_GROUP


class LunaparkOperatorParameter(parameters.SandboxStringParameter):
    name = "tank_operator"
    description = "Tank operator for feedback (task author by default)"
    default_value = ""
    group = LUNAPARK_GROUP


class EnableLunaparkParameter(parameters.SandboxBoolParameter):
    name = 'enable_lunapark'
    description = 'Enable lunapark'
    default_value = True
    group = LUNAPARK_GROUP
    sub_fields = {
        'true': [LunaparkAddressParameter.name, LunaparkTaskParameter.name, LunaparkOperatorParameter.name]
    }


class LunaparkPlugin:
    """Set of options for lunapark plugin"""

    @classmethod
    def create_params(cls, enabled=False):
        class _EnableLunaparkParameter(EnableLunaparkParameter):
            default_value = enabled

        return (
            _EnableLunaparkParameter,
            LunaparkAddressParameter,
            LunaparkTaskParameter,
            LunaparkOperatorParameter
        )

    def __init__(self, task):
        if not task.ctx.get(EnableLunaparkParameter.name, False):
            self.config = {}
            return
        self.config = {
            "tank": {
                "plugin_meta": "yandextank.plugins.DataUploader",
            },
            "meta": {
                "task": task.ctx[LunaparkTaskParameter.name],
                "operator": task.ctx[LunaparkOperatorParameter.name] or task.author,
                "api_address": task.ctx[LunaparkAddressParameter.name],
                "jobno_file": _JOBNO_FILE,
                "ignore_target_lock": True,  # our target is 'localhost', so we need this option to avoid conflict
            }
        }

    # TODO: search inside artifacts directory for tank 1.9
    @staticmethod
    def get_job_url(task):
        if not task.ctx.get(EnableLunaparkParameter.name, False):
            return None
        if not os.path.exists(_JOBNO_FILE):
            return None
        with open(_JOBNO_FILE) as f:
            return "https://lunapark.yandex-team.ru/{}".format(f.read())


# Dolbilo plugin


class DolbiloPlugin:
    """Set of options for dolbilo plugin"""

    _latency_quantiles = (0.5, 0.95, 0.99)
    _resp_size_quantiles = (0.5, 0.95)

    @classmethod
    def create_params(cls):
        return (
            dolbilka.DolbilkaExecutorResource,
        ) + dolbilka.DolbilkaExecutor.dolbilka_args_params

    def __init__(self, task, plan, target_host, target_port, rps_schedule=None, augment_url=None):
        executor_resource_id = task.ctx.get(dolbilka.DolbilkaExecutorResource.name)
        if not executor_resource_id:
            executor_resource_id = utils.get_and_check_last_released_resource_id(
                dolbilka_resources.DEXECUTOR_EXECUTABLE
            )

        # basic parameters
        executor_path = task.sync_resource(executor_resource_id)
        plan_path = task.sync_resource(plan)

        if os.path.getsize(plan_path) == 0:
            raise Exception("Plan from resource {} is empty.".format(plan))

        cmdline = [
            ("plan-file", plan_path),
            ("replace-host", target_host),
            ("replace-port", target_port),
            ("output", _PHOUT_FILE),
            "circular",
            "phantom",
        ]

        # additional parameters
        for param in dolbilka.DolbilkaExecutor.dolbilka_args_params:
            value = utils.get_or_default(task.ctx, param)
            if value:
                if isinstance(value, bool):
                    cmdline.append(param.dolbilka_name)
                else:
                    cmdline.append((param.dolbilka_name, value))

        self.config = {
            "tank": {
                "plugin_shootexec": "yandextank.plugins.ShootExec",
            },
            "shootexec": {
                "cmd": executor_path + " " + " ".join(
                    "--{}='{}'".format(k[0], k[1]) if isinstance(k, tuple) else "--{}".format(k)
                    for k in cmdline
                ),
                "output_path": _PHOUT_FILE,
            }
        }

    @staticmethod
    def get_phout_dump_path(artifacts_dir):
        """Search for phout dump inside artifacts directory"""

        for dump_pattern in ("dolbilo_dump_*.log", _PHOUT_FILE):
            dump_path = glob.glob("{}/*/{}".format(artifacts_dir, dump_pattern))
            if dump_path:
                return dump_path[0]

        eh.check_failed("Failed to find dolbilo dump inside {} directory".format(artifacts_dir))

    @classmethod
    def get_dumper_stats(cls, artifacts_dir):
        """
            Returns statistics compatible with dumper results
        """

        stats = cls.get_stats(artifacts_dir)
        result = copy.deepcopy(stats)

        # Convert to old style stats
        for key, value in stats.iteritems():
            if key.startswith("dumper."):
                result[key[len("dumper."):]] = value
            elif key.startswith("shooting.latency_"):
                prefix, suffix = key.rsplit("_", 1)
                result["latency_" + suffix] = value
            elif key.startswith("shooting.response_size_"):
                prefix, suffix = key.rsplit("_", 1)
                if suffix == "1":
                    result["max_size"] = value
                else:
                    result["resp_size_quantile_" + suffix] = value

        return result

    @classmethod
    def get_stats(cls, artifacts_dir):
        stats_path = glob.glob("{}/*/{}".format(artifacts_dir, "offline_stats_*.json"))
        if not stats_path:
            eh.check_failed("Failed to find shooting stats inside {} directory".format(artifacts_dir))
        with open(stats_path[0]) as stats_file:
            stats_data = json.load(stats_file)
            # flatten
            for flatten_key in ("shooting", "monitoring", "dumper", "old_dumper"):
                if flatten_key not in stats_data:
                    continue
                for k, v in stats_data[flatten_key].iteritems():
                    stats_data["{}.{}".format(flatten_key, k)] = v
                del stats_data[flatten_key]
            return stats_data


# Phantom plugin


class PhantomRpsScheduleParameter(parameters.SandboxStringParameter):
    name = 'phantom_rps_schedule'
    description = 'RPS schedule'
    group = PHANTOM_GROUP


class PhantomWriteLogParameter(parameters.SandboxStringParameter):
    name = 'phantom_writelog'
    description = 'Phantom logging'
    group = PHANTOM_GROUP
    default_value = '0'
    choices = [
        ('Disabled', '0'),
        ('All messages', 'all'),
        ('4xx+5xx+network errors', 'proto_warning'),
        ('5xx+network errors', 'proto_error')
    ]


class EnablePhantomParameter(parameters.SandboxBoolParameter):
    name = 'enable_phantom'
    description = 'Enable phantom'
    default_value = True
    group = PHANTOM_GROUP


class PhantomPlugin:
    """Set of options for phantom plugin"""

    default_params = (
        PhantomRpsScheduleParameter,
        PhantomWriteLogParameter,
    )

    @classmethod
    def create_params(cls, enabled=False, plan_params=[]):
        class _EnablePhantomParameter(EnablePhantomParameter):
            default_value = enabled
            sub_fields = {
                'true': [p.name for p in plan_params] + [p.name for p in cls.default_params]
            }

        return (_EnablePhantomParameter,) + cls.default_params + tuple(plan_params)

    def __init__(self, task, uris, target_host, target_port, rps_schedule=None):
        if not task.ctx.get(EnablePhantomParameter.name, False):
            self.config = {}
            return
        phantom_resource_id = utils.get_and_check_last_released_resource_id(
            tank_resources.PHANTOM_EXECUTABLE,
            arch=sandboxapi.ARCH_LINUX
        )
        if rps_schedule is None:
            rps_schedule = task.ctx[PhantomRpsScheduleParameter.name]
        self.config = {
            "tank": {
                "plugin_phantom": "yandextank.plugins.Phantom",
            },
            "phantom": {
                "address": "{}:{}".format(target_host, target_port),
                "rps_schedule": rps_schedule,
                "ammofile": task.sync_resource(uris),
                "headers": ["[Host: {}]".format(target_host), "[Connection: close]"],
                "phantom_path": task.sync_resource(phantom_resource_id),
                "writelog": task.ctx[PhantomWriteLogParameter.name],
                # TODO: add headers parameter as a space separated list 'header1=value1 header2=value2'
            }
        }


# Autostop plugin


class AutostopConditionParameter(parameters.SandboxStringParameter):
    name = 'autostop_condition'
    description = 'Autostop condition'
    group = AUTOSTOP_GROUP
    required = True


class EnableAutostopParameter(parameters.SandboxBoolParameter):
    name = 'enable_autostop'
    description = 'Enable autostop'
    default_value = True
    group = AUTOSTOP_GROUP


class AutostopPlugin:
    """Set of options for autostop plugin"""

    @classmethod
    def create_params(cls, enabled=False):
        class _EnableAutostopParameter(EnableAutostopParameter):
            default_value = enabled
            sub_fields = {
                'true': [AutostopConditionParameter.name]
            }

        return _EnableAutostopParameter, AutostopConditionParameter

    def __init__(self, task):
        if not task.ctx.get(EnableAutostopParameter.name, False):
            self.config = {}
            return
        self.config = {
            "tank": {
                "plugin_autostop": "yandextank.plugins.Autostop",
                "plugin_totalautostop": "yandextank.plugins.TotalAutostop",
            },
            "autostop": {
                "autostop": task.ctx[AutostopConditionParameter.name],
            },
        }


# Offline plugin


class OfflineWarmupTimeParameter(parameters.SandboxIntegerParameter):
    name = offline.WARMUP_TIME_KEY
    description = 'Warmup time (seconds):'
    group = OFFLINE_GROUP
    default_value = 0


class OfflineShutdownTimeParameter(parameters.SandboxIntegerParameter):
    name = offline.SHUTDOWN_TIME_KEY
    description = 'Shutdown time (seconds):'
    group = OFFLINE_GROUP
    default_value = 0


class OfflinePlugin:
    """Set of options for offline plugin"""

    @classmethod
    def create_params(cls):
        return (OfflineWarmupTimeParameter, OfflineShutdownTimeParameter)

    def __init__(self, task):
        self.config = {
            "tank": {
                "plugin_offline": "yandextank.plugins.OfflineReport",
            },
            "offline": {
                "warmup_time": utils.get_or_default(task.ctx, OfflineWarmupTimeParameter),
                "shutdown_time": utils.get_or_default(task.ctx, OfflineShutdownTimeParameter),
                "phout_path": _PHOUT_FILE,
            },
        }


# Telegraf


class TelegrafPlugin:
    _telegraf_config = [
        '<Monitoring>',
        '<Host address="[target]" telegraf="{telegraf_path}">',
        '<CPU/>',
        '<System/>',
        '<Memory/>',
        '<Disk/>',
        '<Net/>',
        '<KernelVmstat/>',
        '</Host>',
        '</Monitoring>',
    ]

    @classmethod
    def create_params(cls):
        return []

    def __init__(self, task):
        telegraf_path = os.path.join(task.abs_path(TankExecutor._virtualenv_dir), "bin", "telegraf")
        telegraf_config = [item.format(telegraf_path=telegraf_path) for item in self._telegraf_config]

        self.config = {
            "tank": {
                "plugin_telegraf": "yandextank.plugins.Telegraf",
            },
            "telegraf": {
                "config": telegraf_config,
                "disguise_hostnames": False,
            }
        }


# JsonReport plugin


class JsonReportPlugin:
    """Set of options for online plugin"""

    @classmethod
    def create_params(cls):
        return []

    def __init__(self, task):
        self.config = {
            "tank": {
                "plugin_jsonreport": "yandextank.plugins.JsonReport",
            },
        }


# Common parameters


class TankVirtualenvParameter(parameters.ResourceSelector):
    name = 'tank_virtualenv_resource_id'
    description = 'Tank virtualenv'
    resource_type = (
        tank_resources.YANDEX_TANK_VIRTUALENV_18,
        tank_resources.YANDEX_TANK_VIRTUALENV_19,
    )
    group = TANK_GROUP
    required = False


class TankExecutor:
    """
        Yandex tank executor
    """

    _config = {
        "tank": {
            "plugin_aggregator": "yandextank.plugins.Aggregator",
        },
        "aggregator": {
            "verbose_histogram": True,
        },
    }

    _virtualenv_dir = "tank-venv"

    @classmethod
    def create_params(cls):
        return (TankVirtualenvParameter,)

    def __init__(self, *plugins):
        self.__config = copy.deepcopy(self._config)
        for plugin in plugins:
            self.__merge_config(plugin.config)

    def __merge_config(self, config):
        for key, value in config.iteritems():
            self.__config.setdefault(key, value).update(value)

    def __dump_config(self, path, job_name):
        config = copy.deepcopy(self.__config)
        if "meta" in config and job_name is not None:
            config["meta"]["job_name"] = job_name.replace("\n", " ")

        with open(path, "w") as fileobj:
            for section_name, section_data in config.iteritems():
                fileobj.write("[{}]\n".format(section_name))
                for key, value in section_data.iteritems():
                    if isinstance(value, bool):
                        value = int(value)
                    if isinstance(value, (list, tuple)):
                        value = "\n".join(value[:1] + ["   {}".format(v) for v in value[1:]])
                    fileobj.write("{}={}\n".format(key, value))

    def __dump_yaml(self, path, job_name):

        config = copy.deepcopy(self.__config)

        if "meta" in config and job_name is not None:
            # List to use '|' representation and avoid problems with special characters
            config["meta"]["job_name"] = [job_name.replace("\n", " ")]

        # convert plugin definition to new schema
        for key, value in config["tank"].iteritems():
            if key.startswith(_PLUGIN_KEY_PREFIX):
                plugin_section = key[len(_PLUGIN_KEY_PREFIX):]
                config.setdefault(plugin_section, {}).update({
                    "package": value,
                    "enabled": "true",
                })
        del config["tank"]

        prefix = "  "
        with open(path, "w") as yaml_file:
            for section_name, section_data in config.iteritems():
                yaml_file.write("{}:\n".format(section_name))
                for key, value in section_data.iteritems():
                    if isinstance(value, bool):
                        value = str(value).lower()
                    if isinstance(value, (list, tuple)):
                        value = "|\n" + "\n".join("{}{}".format(prefix * 2, v) for v in value)
                    yaml_file.write("{}{}: {}\n".format(prefix, key, value))

    def fire(self, work_dir, job_name=None, cgroup=None, autostop_expected=False):
        next_path = os.path.join(self._virtualenv_dir, "next")

        if os.path.exists(next_path):
            config_path = os.path.join(work_dir, "load.yaml")
            self.__dump_yaml(config_path, job_name=job_name)
        else:
            config_path = os.path.join(work_dir, "load.ini")
            self.__dump_config(config_path, job_name=job_name)

        artifact_path = os.path.join(work_dir, "artifacts")

        if cgroup is not None:
            if not os.path.exists("/sys/fs/cgroup/{}/{}/tasks".format(*cgroup.split(":"))):
                raise Exception("Failed to find required cgroup {} on host".format(cgroup))
            cmd = ["cgexec", "-g", cgroup]
        else:
            cmd = []

        cmd += [
            os.path.join(self._virtualenv_dir, "bin", "python"),
            os.path.join(self._virtualenv_dir, "bin", "yandex-tank"),
            "-n",
            "-c", config_path,
            "-k", work_dir,
            "-o", "tank.artifacts_base_dir={}".format(artifact_path)
        ]
        proc = process.run_process(cmd, log_prefix='yandex-tank', wait=False)
        retcode = proc.wait()

        if autostop_expected:
            if retcode not in (21, 22, 23, 24, 31, 32, 33):
                eh.check_failed("Autostop condition was not triggered (retcode={})".format(retcode))
        else:
            if retcode != 0:
                eh.check_failed("Tank execution failed (retcode={})".format(retcode))

        return artifact_path

    @staticmethod
    def get_report_path(artifacts_dir):
        report_path = glob.glob("{}/*/{}".format(artifacts_dir, "report*"))
        if not report_path:
            raise Exception("Failed to find html report inside {} directory".format(artifacts_dir))

        return report_path[0]

    @classmethod
    def init_virtualenv(cls, task, work_dir=".", tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19):
        """
            Initialize virtualenv for tank
            Returns true for next version of tank (1.8)
        """

        virtualenv_resource_id = utils.get_or_default(task.ctx, TankVirtualenvParameter)
        if not virtualenv_resource_id:
            virtualenv_resource_id = utils.get_and_check_last_released_resource_id(tank_resource_type)
        virtualenv_archive = task.sync_resource(virtualenv_resource_id)
        virtualenv_path = os.path.join(work_dir, cls._virtualenv_dir)
        paths.make_folder(virtualenv_path, delete_content=True)
        tarfile.open(virtualenv_archive, "r:*").extractall(virtualenv_path)

        return os.path.exists(os.path.join(virtualenv_path, "bin", "telegraf"))
