from sandbox.projects.common.apihelpers import get_last_released_resource
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.common import dolbilka2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox import sdk2

import copy
import glob
import json
import os
import tarfile


_JOBNO_FILE = "jobno.txt"
_PHOUT_FILE = "shoot.tsv"
_PLUGIN_KEY_PREFIX = "plugin_"


class LunaparkPlugin:
    """Set of options for lunapark plugin"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Lunapark settings") as lunapark_settings:
            enable_lunapark = sdk2.parameters.Bool("Enable lunapark", default=True)
            with enable_lunapark.value[True]:
                lunapark_address = sdk2.parameters.String("Lunapark URL", default="https://lunapark.yandex-team.ru/")
                tank_task_name = sdk2.parameters.String("Open task name with load testing attribute", required=True)
                tank_operator = sdk2.parameters.String("Tank operator for feedback (task author by default)")

    @classmethod
    def create_params(cls, enabled=False):
        return cls.Parameters

    def __init__(self, task):
        if not task.Parameters.enable_lunapark:
            self.config = {}
            return
        self.config = {
            "tank": {
                "plugin_meta": "yandextank.plugins.DataUploader",
            },
            "meta": {
                "task": task.Parameters.tank_task_name,
                "operator": task.Parameters.tank_operator or task.author,
                "api_address": task.Parameters.lunapark_address,
                "jobno_file": _JOBNO_FILE,
                "ignore_target_lock": True,  # our target is 'localhost', so we need this option to avoid conflict
            }
        }

    # TODO: search inside artifacts directory for tank 1.9
    @staticmethod
    def get_job_url(task):
        if not task.Parameters.enable_lunapark:
            return None
        if not os.path.exists(_JOBNO_FILE):
            return None
        with open(_JOBNO_FILE) as f:
            return "https://lunapark.yandex-team.ru/{}".format(f.read())


class DolbiloPlugin:
    """Set of options for dolbilo plugin"""

    _latency_quantiles = (0.5, 0.95, 0.99)
    _resp_size_quantiles = (0.5, 0.95)

    @classmethod
    def create_params(cls):
        return dolbilka2.DolbilkaExecutor2.Parameters

    def __init__(self, task, plan, target_host, target_port, rps_schedule=None, augment_url=None):
        executor_resource = task.Parameters.dolbilo_executor_resource_id
        if not executor_resource:
            executor_resource = get_last_released_resource(dolbilka_resources.DEXECUTOR_EXECUTABLE)

        # basic parameters
        executor_path = str(sdk2.ResourceData(executor_resource).path)
        plan_path = str(sdk2.ResourceData(plan).path)

        if os.path.getsize(plan_path) == 0:
            raise Exception("Plan from resource {} is empty.".format(plan))

        cmdline = [
            ("plan-file", plan_path),
            ("replace-host", target_host),
            ("replace-port", str(target_port)),
            ("output", _PHOUT_FILE),
            "circular",
            "phantom",
        ]

        if augment_url is not None:
            cmdline.append(('augmenturl', augment_url))

        # additional parameters
        for param in dolbilka2.DolbilkaExecutor2.dolbilka_args_params:
            value = getattr(task.Parameters, str(param.name))
            if value:
                if isinstance(value, bool):
                    cmdline.append(param.dolbilka_name)
                else:
                    cmdline.append((param.dolbilka_name, value))

        self.config = {
            "tank": {
                "plugin_shootexec": "yandextank.plugins.ShootExec",
            },
            "shootexec": {
                "cmd": executor_path + " " + " ".join(
                    "--{}='{}'".format(k[0], k[1]) if isinstance(k, tuple) else "--{}".format(k)
                    for k in cmdline
                ),
                "output_path": _PHOUT_FILE,
            }
        }

    @staticmethod
    def get_phout_dump_path(artifacts_dir):
        """Search for phout dump inside artifacts directory"""

        for dump_pattern in ("dolbilo_dump_*.log", _PHOUT_FILE):
            dump_path = glob.glob("{}/*/{}".format(artifacts_dir, dump_pattern))
            if dump_path:
                return dump_path[0]

        eh.check_failed("Failed to find dolbilo dump inside {} directory".format(artifacts_dir))

    @classmethod
    def get_dumper_stats(cls, artifacts_dir):
        """
            Returns statistics compatible with dumper results
        """

        stats = cls.get_stats(artifacts_dir)
        result = copy.deepcopy(stats)

        # Convert to old style stats
        for key, value in stats.iteritems():
            if key.startswith("dumper."):
                result[key[len("dumper."):]] = value
            elif key.startswith("shooting.latency_"):
                prefix, suffix = key.rsplit("_", 1)
                result["latency_" + suffix] = value
            elif key.startswith("shooting.response_size_"):
                prefix, suffix = key.rsplit("_", 1)
                if suffix == "1":
                    result["max_size"] = value
                else:
                    result["resp_size_quantile_" + suffix] = value

        return result

    @classmethod
    def get_stats(cls, artifacts_dir):
        stats_path = glob.glob("{}/*/{}".format(artifacts_dir, "offline_stats_*.json"))
        if not stats_path:
            eh.check_failed("Failed to find shooting stats inside {} directory".format(artifacts_dir))
        with open(stats_path[0]) as stats_file:
            stats_data = json.load(stats_file)
            # flatten
            for flatten_key in ("shooting", "monitoring", "dumper", "old_dumper"):
                if flatten_key not in stats_data:
                    continue
                for k, v in stats_data[flatten_key].iteritems():
                    stats_data["{}.{}".format(flatten_key, k)] = v
                del stats_data[flatten_key]
            return stats_data


class TelegrafPlugin:
    _telegraf_config = [
        '<Monitoring>',
        '<Host address="[target]" telegraf="{telegraf_path}">',
        '<CPU/>',
        '<System/>',
        '<Memory/>',
        '<Disk/>',
        '<Net/>',
        '<KernelVmstat/>',
        '</Host>',
        '</Monitoring>',
    ]

    @classmethod
    def create_params(cls):
        return []

    def __init__(self, task):
        telegraf_path = os.path.join(str(task.path(TankExecutor2._virtualenv_dir)), "bin", "telegraf")
        telegraf_config = [item.format(telegraf_path=telegraf_path) for item in self._telegraf_config]

        self.config = {
            "tank": {
                "plugin_telegraf": "yandextank.plugins.Telegraf",
            },
            "telegraf": {
                "config": telegraf_config,
                "disguise_hostnames": False,
            }
        }


class JsonReportPlugin:
    """Set of options for online plugin"""

    @classmethod
    def create_params(cls):
        return []

    def __init__(self, task):
        self.config = {
            "tank": {
                "plugin_jsonreport": "yandextank.plugins.JsonReport",
            },
        }


class OfflinePlugin:
    """Set of options for offline plugin"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Offline report settings") as offline_report_settings:
            offline_warmup_time = sdk2.parameters.Integer("Warmup time (seconds)", default=0)
            offline_shutdown_time = sdk2.parameters.Integer("Shutdown time (seconds)", default=0)

    @classmethod
    def create_params(cls):
        return cls.Parameters

    def __init__(self, task):
        self.config = {
            "tank": {
                "plugin_offline": "yandextank.plugins.OfflineReport",
            },
            "offline": {
                "warmup_time": task.Parameters.offline_warmup_time,
                "shutdown_time": task.Parameters.offline_shutdown_time,
                "phout_path": _PHOUT_FILE,
            },
        }


class TankExecutor2:
    _config = {
        "tank": {
            "plugin_aggregator": "yandextank.plugins.Aggregator",
        },
        "aggregator": {
            "verbose_histogram": True,
        },
    }

    _virtualenv_dir = "tank-venv"

    def __init__(self, *plugins):
        self.__config = copy.deepcopy(self._config)
        for plugin in plugins:
            self.__merge_config(plugin.config)

    def __merge_config(self, config):
        for key, value in config.iteritems():
            self.__config.setdefault(key, value).update(value)

    def __dump_yaml(self, path, job_name):
        config = copy.deepcopy(self.__config)

        if "meta" in config and job_name is not None:
            # List to use '|' representation and avoid problems with special characters
            config["meta"]["job_name"] = [job_name.replace("\n", " ")]

        # convert plugin definition to new schema
        for key, value in config["tank"].iteritems():
            if key.startswith(_PLUGIN_KEY_PREFIX):
                plugin_section = key[len(_PLUGIN_KEY_PREFIX):]
                config.setdefault(plugin_section, {}).update({
                    "package": value,
                    "enabled": "true",
                })
        del config["tank"]

        prefix = "  "
        with open(path, "w") as yaml_file:
            for section_name, section_data in config.iteritems():
                yaml_file.write("{}:\n".format(section_name))
                for key, value in section_data.iteritems():
                    if isinstance(value, bool):
                        value = str(value).lower()
                    if isinstance(value, (list, tuple)):
                        value = "|\n" + "\n".join("{}{}".format(prefix*2, v) for v in value)
                    yaml_file.write("{}{}: {}\n".format(prefix, key, value))

    def fire(self, work_dir, job_name=None, cgroup=None, autostop_expected=False):
        config_path = os.path.join(work_dir, "load.yaml")
        self.__dump_yaml(config_path, job_name=job_name)

        artifact_path = os.path.join(work_dir, "artifacts")

        if cgroup is not None:
            if not os.path.exists("/sys/fs/cgroup/{}/{}/tasks".format(*cgroup.split(":"))):
                raise Exception("Failed to find required cgroup {} on host".format(cgroup))
            cmd = ["cgexec", "-g", cgroup]
        else:
            cmd = []

        cmd += [
            os.path.join(self._virtualenv_dir, "bin", "python"),
            os.path.join(self._virtualenv_dir, "bin", "yandex-tank"),
            "-n",
            "-c", config_path,
            "-k", work_dir,
            "-o", "tank.artifacts_base_dir={}".format(artifact_path)
        ]
        proc = process.run_process(cmd, log_prefix='yandex-tank', wait=False)
        retcode = proc.wait()

        if autostop_expected:
            if retcode not in (21, 22, 23, 24, 31, 32, 33):
                eh.check_failed("Autostop condition was not triggered (retcode={})".format(retcode))
        else:
            if retcode != 0:
                eh.check_failed("Tank execution failed (retcode={})".format(retcode))

        return artifact_path

    @staticmethod
    def get_report_path(artifacts_dir):
        report_path = glob.glob("{}/*/{}".format(artifacts_dir, "report*"))
        if not report_path:
            raise Exception("Failed to find html report inside {} directory".format(artifacts_dir))

        return report_path[0]

    @classmethod
    def init_virtualenv(cls, task, work_dir=".", tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19):
        """ Initialize virtualenv for tank """

        virtualenv_resource = sdk2.Resource.find(tank_resource_type, attrs=dict(released='stable')).first()
        virtualenv_archive = sdk2.ResourceData(virtualenv_resource).path
        virtualenv_path = os.path.join(work_dir, cls._virtualenv_dir)
        paths.make_folder(virtualenv_path, delete_content=True)
        tarfile.open(str(virtualenv_archive), "r:*").extractall(str(virtualenv_path))
