# coding: utf-8

import json
import logging
import logging.handlers
import time
import requests
# noinspection PyCompatibility

from sandbox.projects.tank.lunapark_client import exceptions
from sandbox.projects.tank.lunapark_client.settings import LUNAPARK_URL, REQUEST_TIMEOUT, REQUEST_RETRIES, API_VERSION

# TODO: add parameters
api_handles = {
    'summary': '/api/job/{test_id}/summary.json',
    'config': '/api/job/{test_id}/configinfo.txt',
    'sla': '/api/job/{test_id}/sla.json',
    'dist': {
        'times': '/api/job/{test_id}/dist/times.json',
        'percentiles': '/api/job/{test_id}/dist/percentiles.json',
        'http': '/api/job/{test_id}/dist/http.json',
        'net': '/api/job/{test_id}/dist/net.json'
    },
    'aggregates': '/api/job/{test_id}/aggregates.json',
    'task': {
        'summary': '/api/task/{task_id}/summary.json',
        'jobs': '/api/task/{task_id}/joblist.json'
    },
    'regress': {
        'components': '/api/regress/{project_id}/componentlist.json',
        'kpi': '/api/regress/{component_id}/kpilist.json'
    }
}


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'lunapi_client.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


class Connection(object):

    def __init__(
            self, base_url=LUNAPARK_URL,
            headers=None,
            api_version=API_VERSION
    ):

        self.session = requests.Session()

        if headers is not None:
            self.session.headers.update(headers)

        self.base_url = base_url
        self.timeout = REQUEST_TIMEOUT
        self.retries = REQUEST_RETRIES
        self.api_version = api_version

    def build_url(self, path):
        return '{}/{}'.format(self.base_url, path)

    @staticmethod
    def _log_error(level, response):
        # noinspection PyBroadException
        try:
            data = response.json()
            logger().log(level, 'Lunapark api errors: %s %s', data.get('statusCode'), data.get('errors'))
            messages = data.get('errorMessages', ())
            logger().log(level, '%d messages follow:', len(messages))
            for msg in messages:
                logger().log(level, ' - %s', msg)
        except Exception:
            logger().log(level, 'Strange lunapark error: %s', response.text)

    def request(self, method, path, data=None, headers=None):
        logger().info("Request %s %s", method, path)
        url = self.build_url(path)
        if headers is None:
            headers = {}
        if data is not None:
            data = json.dumps(data)

        return self._try_request(
            method=method,
            url=url,
            data=data,
            timeout=self.timeout,
            headers=headers
        )

    def _try_request(self, **kwargs):
        response = None
        exception = None
        iterations = max(self.retries + 1, 1)

        for retry in range(iterations):
            try:
                response = self.session.request(**kwargs)
            except Exception as excep:
                exception = excep
                logger().warning("Request failed with exception %s, retrying (%s)...", excep, retry)
            else:
                exception = None
                if 500 <= response.status_code < 600:
                    logger().warning("Request failed with status %d, retrying (%d)...", response.status_code, retry)
                    self._log_error(logging.WARNING, response)
                else:
                    break

        if exception is not None:
            raise exceptions.LunaparkError(exception)
        elif 500 <= response.status_code < 600:
            raise exceptions.OutOfRetries(response)
        elif 400 <= response.status_code < 500:
            self._log_error(logging.ERROR, response)
            exc_class = exceptions.STATUS_CODES.get(
                response.status_code,
                exceptions.LunaparkServerError
            )
            raise exc_class(response)

        return response


class LunaparkApiClient(object):
    """ Basic Lunapark API client,
    later should be replaced with
    https://github.yandex-team.ru/load/digger/blob/master/digger/digger.py
    """

    def __init__(self):
        self.connection = Connection()

    def get_summary(self, shoot_id):
        url = api_handles['summary'].format(test_id=shoot_id)
        return self.connection.request('GET', url).json()[0]

    def get_percentiles(self, shoot_id):
        url = api_handles['dist']['percentiles'].format(test_id=shoot_id)
        return self.connection.request('GET', url).json()

    def get_http(self, shoot_id):
        url = api_handles['dist']['http'].format(test_id=shoot_id)
        return self.connection.request('GET', url).json()

    def check_for_sla(self, shoot_id):
        url = api_handles['sla'].format(test_id=shoot_id)
        return self.connection.request('GET', url).json()[0]


if __name__ == '__main__':
    client = LunaparkApiClient()
    print(client.get_percentiles(1679134))
