# -*- coding: utf-8 -*-


class LunaparkError(Exception):
    """ Base class for all Lunapark API client errors"""
    pass


class LunaparkClientError(LunaparkError):
    """Base class for exceptions on client side"""
    pass


class LunaparkServerError(LunaparkError, IOError):
    """Base class for error conditions on server side."""

    response = None
    status_code = None
    reason = None
    errors = None
    error_messages = None

    def __init__(self, response):
        self.response = response
        self.status_code = response.status_code
        self.reason = response.reason
        # noinspection PyBroadException
        try:
            error = response.json()
        except Exception:
            pass
        else:
            self.errors = error.get('errors')
            self.error_messages = error.get('errorMessages')
        super(LunaparkServerError, self).__init__(response)

    def __str__(self):
        lines = []
        if self.__class__ is LunaparkServerError:
            # Include informational header for unrecognized exceptions
            # In presence a subclass name, it is superfluous.
            lines.append('({}) {}'.format(self.status_code, self.reason))
        if self.errors:
            lines.extend(
                '- {}: {}'.format(key, message)
                for key, message in sorted(self.errors.items())
            )
        if self.error_messages:
            lines.extend(self.error_messages)
        return '\n'.join(lines)


class OutOfRetries(LunaparkError):
    def __str__(self):
        return '\n'.join((
            'Out of retries, last error was:',
            super(OutOfRetries, self).__str__()
        ))


class BadRequest(LunaparkClientError):
    pass


class Forbidden(LunaparkClientError):
    pass


class NotFound(LunaparkClientError):
    pass


STATUS_CODES = {
    400: BadRequest,
    403: Forbidden,
    404: NotFound
}
