# -*- coding: utf-8 -*-
import os
import shutil
import logging
import time

from sandbox.projects.taxi import resources as taxi_resource_types
from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common import context_managers
from sandbox.projects.BuildSearch import BuildSearch
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.nanny import nanny


class PYmakeParameters(SandboxStringParameter):
    name, description, default_value = 'ymake_parameters', 'ymake parameters (all -D)', ''


class PYmakeFlags(SandboxStringParameter):
    name, description, default_value = 'ymake_flags', 'ymake -W, -f flags', ''


class PSaveMakePaths(SandboxBoolParameter):
    name, description, default_value = 'save_make_paths', 'Save make paths', False


class PWriteDepsTree(SandboxBoolParameter):
    name, description, default_value = 'write_deps_tree', 'Write dependencies tree', False


class PCheckoutFlag(SandboxBoolParameter):
    name, description, default_value = 'ymake_checkout_flag', 'clone+checkout(without copy all arcadia)', False


def GPBuildBin(bin_name):
    class PBuildBin(SandboxBoolParameter):
        name = 'build_taxi_' + bin_name
        description = name
    return PBuildBin


def get_build_bins_parameters():
    bin_names = [
        'dispatcher',
    ]
    return [GPBuildBin(bin_name) for bin_name in bin_names]


def get_build_main_params():
    return build_params.get_arcadia_params() + [
        build_params.ClearBuild,
        build_params.BuildSystemPlain,
        build_params.BuildType,
        build_params.Sanitize,
        build_params.TargetPlatform
    ]


class BuildTaxi(nanny.ReleaseToNannyTask, BuildSearch):
    type = 'BUILD_TAXI'
    client_tags = Tag.GENERIC & Tag.LINUX_PRECISE
    input_parameters = get_build_main_params() \
                       + get_build_bins_parameters() \
                       + [PYmakeParameters, PYmakeFlags,
                          PSaveMakePaths,
                          PWriteDepsTree, PCheckoutFlag]
    """
        TARGET_RESOURCE consists of three fields: (resource_type, target, target_path, filename)
        resource_type - type of the target resource
        target - title of the target
        target_path - final path of the built target resource. dirname of the target path is the path to checkout from arcadia
        filename is the name of target resource file.
    """
    TARGET_PATH_TO_NAME_MAP = {
        'taxi/logistic-dispatcher/dispatcher': 'taxi_dispatcher',
    }
    TARGET_RESOURCES = (
        (taxi_resource_types.TAXI_DISPATCHER, 'taxi/logistic-dispatcher/dispatcher', 'dispatcher'),
    )
    VIRTUAL_TARGETS = (
        (taxi_resource_types.TAXI_DISPATCHER, 'taxi_dispatcher', 'taxi/logistic-dispatcher/dispatcher'),
    )
    LOCAL_BIN_DIR = 'binaries'
    rtline_configs_res_dir = ''
    release_dir = ''
    required_ram = 80000
    execution_space = 80000

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        if self.ctx['build_bundle']:
            return
        self.do_on_enqueue()
        dprio = 0
        try:
            dprio = int(self.ctx.get('dprio', 0))
        except:
            pass
        self.score = dprio

    def mark_stage_start(self, stage):
        self.ctx['__STAGES'][stage + '_started'] = time.time()

    def mark_stage_finish(self, stage):
        self.ctx['__STAGES'][stage + '_finished'] = time.time()

    def do_on_enqueue(self):
        if self.ctx.get('save_make_paths') and self.ctx.get('build_system') in (consts.YMAKE_BUILD_SYSTEM, consts.DISTBUILD_BUILD_SYSTEM):
            return
        for resource_type, target_path, filename in self.TARGET_RESOURCES:
            target = self.get_target_name(target_path)
            if self._target_enabled(target):
                resource_path = os.path.join(self.LOCAL_BIN_DIR, filename)
                resource_name = '%s (%s)' % (self.descr, filename)
                resource = self._create_resource(resource_name,
                                                 resource_path,
                                                 resource_type,
                                                 arch=self.arch)
                self.ctx['%s_resource_id' % target] = resource.id
        self.ctx["don_t_release"] = []
        self.set_info('task enqueued for execution')

    def _run_ymake_build(self, enabled_targets, build_type, clear_build, sanitize):
        self.release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)
        if self.ctx.get('save_make_paths'):
            make_folder(self.release_dir)
            os.chdir(self.release_dir)
            ya_path = os.path.join(self.arcadia_src_dir, 'ya')
            ymake_cmd = [ya_path, 'make',
                         '-j0', '--dump-graph',
                         '--source-root=' + self.arcadia_src_dir,
                         '--results-root=' + self.release_dir,
                         ]
            ymake_cmd.append('--build-dir=' + self.abs_path('tmp'))
            if self.ctx.get('ymake_parameters', '').strip():
                ymake_cmd.extend(self.ctx.get('ymake_parameters').strip().split())
            if not self.ctx.get('write_deps_tree'):
                deps = sdk.dump_targets_deps(self.arcadia_src_dir, enabled_targets)
                self.save_dir_list_resource(deps, 'all')
                self.ctx['deps_count_all'] = len(deps)
            else:
                targets = [(os.path.basename(os.path.join(self.arcadia_src_dir, t)),
                            t)
                           for t in enabled_targets]
                for targ in targets:
                    deps = sdk.dump_targets_deps(self.arcadia_src_dir, [targ[1], ])
                    self.save_dir_list_resource(deps, targ[0])
                    self.ctx['deps_count_' + targ[0]] = len(deps)
            self.ctx['task_done'] = True
        else:
            user_flags = sdk.parse_flags(self.ctx.get('ymake_parameters', ''))
            no_d_flags = self.ctx.get('ymake_flags', '')
            if no_d_flags:
                try:
                    no_d_flags = 'CFLAGS=' + no_d_flags.strip().strip(';')
                    no_d_flags = no_d_flags.split(';')
                    no_d_flags = dict(
                        (f.split('=', 1)[0], '' + f.split('=', 1)[1] + '') for f in no_d_flags[:] if '=' in f
                    )
                except Exception as e:
                    logging.error('while parsing flags, error: %s' % e)
            flags = {}
            if user_flags:
                flags.update(user_flags)
            if no_d_flags:  # and 'flags' in self.descr:
                flags.update(no_d_flags)
                logging.info('all_flags: %s' % flags)
            platform = self.ctx.get(consts.TARGET_PLATFORM_KEY) or None
            build_system = self.ctx.get('build_system', consts.YMAKE_BUILD_SYSTEM)
            sdk.do_build(
                build_system, self.arcadia_src_dir, enabled_targets, clear_build=clear_build,
                build_type=build_type, def_flags=flags, results_dir=self.release_dir, target_platform=platform,
                sanitize=sanitize,
                timeout=7200, checkout=bool(self.ctx.get('ymake_checkout_flag', False))
            )

    def _build(self, enabled_targets):
        clear_build = self.ctx.get(consts.CLEAR_BUILD_KEY, True)
        build_type = self.ctx.get(consts.BUILD_TYPE_KEY)
        sanitize = self.ctx.get(consts.SANITIZE, False)
        self._run_ymake_build(enabled_targets, build_type, clear_build, sanitize)

    def do_execute(self):
        self.ctx['queue_time_min'] = round((self.timestamp_start - self.timestamp)/60, 1)
        self.ctx['__STAGES'] = self.ctx.get('__STAGES', {})
        os.chdir(self.abs_path())
        binaries_dir = os.path.join(self.abs_path(), self.LOCAL_BIN_DIR)
        make_folder(binaries_dir)
        self.mark_stage_start('svn')
        with self.prepare_repo() as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            arcadia_patch = self.ctx.get('arcadia_patch')
            if arcadia_patch.strip() and arcadia_patch.strip() not in ('None',):
                Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())
            self.mark_stage_finish('svn')
            # build dependencies tree
            if self.ctx.get('write_deps_tree'):
                # todo something here
                pass
            #
            self.fill_system_info()
            enabled_targets = []
            for _, target, target_path in self.VIRTUAL_TARGETS:
                if self._target_enabled(target):
                    enabled_targets.append(target_path)
            #
            self.mark_stage_start('build')
            self._build(enabled_targets)
            self.mark_stage_finish('build')
            if self.ctx.get('task_done', False):
                return
            for _, target_path, filename in self.TARGET_RESOURCES:
                target = self.get_target_name(target_path)
                logging.info('Check target %s, path %s, filename %s' % (target, target_path, filename))
                if self._target_enabled(target):
                    # build resource
                    build_dir = self.abs_path(os.path.join(self.release_dir, target_path))
                    os.chdir(build_dir)  # check abs path
                    logging.info('read_resource: %s, %s', target, self._target_resource_id(target))
                    resource_path = channel.sandbox.get_resource(self._target_resource_id(target)).path
                    target_file = os.path.join(target_path, filename)
                    ready_file = self.abs_path(os.path.join(self.release_dir, target_file))
                    try:
                        shutil.copy(ready_file, resource_path)
                        logging.info('Target file: %s copy to: %s' % (ready_file, resource_path))
                    except IOError as e:
                        raise SandboxTaskFailureError('Error with copy : %s, file %s' % (repr(e), ready_file))
            # cleanup
            self.clean_release_dir(self.release_dir)
            self.set_info('build completed')
            self.ctx['completed'] = True
            self.ctx['exec_time_min'] = round((int(time.time()) - self.timestamp_start)/60, 1)

    def save_dir_list_resource(self, lines, targ):
        res_path = self.abs_path("dir_list_" + targ)
        with open(res_path, 'w') as f:
            f.write('\n'.join(lines))
        res = self.create_resource('dir list for ' + targ, res_path, resource_types.DIR_LIST)
        return res

    def prepare_repo(self):
        if not self.ctx.get('arcadia_revision', 0):
            revision_info = None
            for i in range(5):
                try:
                    revision_info = Arcadia.info(self.ctx[consts.ARCADIA_URL_KEY])
                    break
                except Exception as e:
                    logging.warning('cannot get svn info, try %s, error %s' % (i, e))
                    if i == 4:
                        raise
            self.ctx['arcadia_revision'] = revision_info['entry_revision']
        if not self.ctx.get('ymake_checkout_flag', 0):
            return sdk.mount_arc_path(self.ctx[consts.ARCADIA_URL_KEY], use_arc_instead_of_aapi=True)
        else:
            return context_managers.nullcontext(sdk.do_clone(self.ctx[consts.ARCADIA_URL_KEY], self))

    def mark_all_ready(self):
        for resource in self.list_resources():
            if resource.type.name == 'TASK_LOGS':
                continue
            logging.info('found resource %s' % str(resource))
            if not resource.is_ready():
                try:
                    resource.mark_ready()
                except Exception as e:
                    raise SandboxTaskUnknownError('Problems with resource %s, error %s' % (str(resource.id), e))

    def on_finish(self):
        if not self.ctx.get('completed', False):
            return
        self.mark_stage_start('share')
        self.mark_all_ready()
        self.mark_stage_finish('share')


__Task__ = BuildTaxi
