import logging
import os
import re
import tarfile
import pathlib2
from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.environments import SandboxEnvironment
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.sdk2.helpers import subprocess

VAULT_PLACEHOLDER_RE = re.compile(r'{vault\.(.*?)}')
STRING_PARAMETERS_SEPARATOR = " "
STRING_PARAMETERS_DESCRIPTION = "Separated by '{}'".format(STRING_PARAMETERS_SEPARATOR)


def split_param(param):
    return [p for p in param.split(STRING_PARAMETERS_SEPARATOR) if p]


class CommonParameters(sdk2.Parameters):
    jdk_version = sdk2.parameters.String("JDK version", default="11.0.8")
    logging.info("jdk_version: %s" %jdk_version)

    build_sdk = sdk2.parameters.Resource("Android sdk", required=True, default=2525506206)  # id or what?
    logging.info("build_sdk: %s" %build_sdk)

    gradle_targets = sdk2.parameters.String(
        "Gradle targets", description="Gradle targets description", required=True
    )
    logging.info("gradle_targets: %s" %gradle_targets)

    gradle_arguments = sdk2.parameters.String(
        "Gradle arguments", default="", description="Gradle arguments description"
    )
    logging.info("gradle_arguments: %s" %gradle_arguments)

    emulator_launching_config = sdk2.parameters.Resource("Emulator launching config", required=True, default=2523425075)  # id or what?


class MobileTaxiClientScreentestsTask(sdk2.Task):
    """
    This is a task that will lead us to the screen tests!
    """

    logging.info("Hello, World!")

    class Parameters(CommonParameters):
        configuration = sdk2.parameters.String("Task configuration", default="", multiline=True)
        description = "This is a sample description for the task"

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024
        client_tags = ctc.Tag.SSD & ctc.Tag.Group.LINUX & ctc.Tag.Group.INTEL
        cores = 16
        dns = ctm.DnsType.DNS64
        ram = 31 * 1024
        environments = [
            PipEnvironment("get-deps==1.2.3"),
            PipEnvironment("yandex-avd-tools==1.4.0", custom_parameters=["--no-deps"]),
            PipEnvironment('killall>=1.1.0,<2'),
            PipEnvironment('pyjavaproperties>=0.7,<1'),
            PipEnvironment('retry>=0.9.2,<1'),
            PipEnvironment('six<2'),
        ]

    class Caches(sdk2.Task.Requirements.Caches):
        pass

    def on_prepare(self):
        logging.debug("on_prepare(), setting up jdk...")

        jdk_version = self.Parameters.jdk_version
        SandboxJavaJdkEnvironment(version=jdk_version, platform="Linux").prepare()
        super(MobileTaxiClientScreentestsTask, self).on_prepare()

    def artifacts_path(self, *args):
        return str(self.path("teamcity_artifacts", *args))

    def checkout_path(self, *args):
        return str(self.path("checkout-dir", *args))

    def build_sdk_path(self, *args):
        return str(self.path("build-sdk"), *args)

    def emulator_sdk_path(self, *args):
        return str(self.path("emulator-sdk"), *args)

    def device_log_path(self, *args):
        return self.checkout_path("logcat.txt")

    # checkout?

    def get_build_sdk(self):
        archive_part = str(sdk2.ResourceData(self.Parameters.build_sdk).path)
        with tarfile.open(archive_part, "r") as archive:
            archive.extractall(path=self.build_sdk_path())

    def get_emulators_sdk(self, emulator_sdk_url, emulator_sdk_hash):
        from get_deps import get_deps
        os.mkdir(self.emulator_sdk_path())
        cache_dir = os.path.join(SandboxEnvironment.build_cache_dir, "get-deps")
        getdeps =get_deps.GetDeps(cache_dir=cache_dir)
        getdeps.deploy(
            url=emulator_sdk_url,
            deploy_rules=[("copy", "*", self.emulator_sdk_path() + "/")],
            sha1=emulator_sdk_hash
        )

    def gradle(self):
        gradle_args = (split_param(self.Parameters.gradle_arguments) if self.Parameters.gradle_arguments else [])
        gradle_args = [
            re.sub(VAULT_PLACEHOLDER_RE, lambda match: sdk2.Vault.data(match.group(1), arg))
            for arg in gradle_args
        ]

        with TeamcityArtifactsContext(pathlib2.Path(self.checkout_path())) as artifacts_context:
            env = os.environ.copy()
            env["ANDROID_HOME"] = self.build_sdk_path()
            env["EMULATOR_SDK_PATH"] = self.emulator_sdk_path()
            env["GRADLE_USER_HOME"] = SandboxEnvironment.exclusive_build_cache_dir("gradle")
            return subprocess.call(
                ["./gradlew"] + gradle_args + split_param(self.Parameters.gradle_targets),
                cwd=self.checkout_path(),
                env=env,
                stdout=artifacts_context.output,
                stderr=artifacts_context.output
            )

    def start_logcat(self, log_file):
        from avd_tools import paths

        logcat_process = None

        try:
            logcat_process = subprocess.Popen(
                [paths.adb_path(), "logcat"], stdout=log_file
            )
            yield logcat_process
        finally:
            if logcat_process is not None:
                logcat_process.kill()

    def on_execute(self):
        import avd_tools

        logging.debug("on_execute()")
        logging.info("My id is %s" % self.id)

        launch_name = self.Parameters.launch_name or "default_config"
        logging.info("launch name: %s" %launch_name)

        avd_tools.utils.kill_stale_processes()
        avd_tools.utils.remove_android_home_dir()

        logging.info("Is that method will be called?")
        logging.debug("Is that method will be called?")
