# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import requests

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import environments

DEFAULT_VAULT_OWNER = 'TT-BACKEND'
DEFAULT_ARCADIA_SVN_URL = 'arcadia:/arc/trunk/arcadia'

DEFAULT_DOCKER_IMAGE_REPOSITORY = 'telephony'
DEFAULT_DOCKER_VAULT_NAME = 'docker-registry-token'
DEFAULT_DOCKER_USER = 'robot-telephony'

DEFAULT_QLOUD_SERVICE = 'telephony'
DEFAULT_QLOUD_PROJECT = 'backend-servants'
DEFAULT_QLOUD_ENV = 'test'
DEFAULT_QLOUD_VAULT_NAME = 'qloud-token'


class BuildAndDeployToQloud(sdk2.Task):
    '''
    - Build docker image from package.json
    - Push the image to the docker registry (registry.yandex.net)
    - Deploy the image to Qloud (https://platform.yandex-team.ru)
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('requests'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Arcadia') as versions_block:
            arcadia_svn_url = sdk2.parameters.String('Svn url for arcadia', default=DEFAULT_ARCADIA_SVN_URL, required=True)
            arcadia_package_config_path = sdk2.parameters.String('YA_PACKAGE config path (relative to arcadia root)', required=True)
            arcadia_svn_revision = sdk2.parameters.String('Arcadia revision', required=False)

        with sdk2.parameters.Group('Docker Registry Config') as docker_block:
            docker_image_repository = sdk2.parameters.String('Image repository', default=DEFAULT_DOCKER_IMAGE_REPOSITORY)
            docker_vault_name = sdk2.parameters.String('Docker token vault name', default=DEFAULT_DOCKER_VAULT_NAME, required=True)
            docker_user = sdk2.parameters.String('Docker login', default=DEFAULT_DOCKER_USER, required=True)

        with sdk2.parameters.Group('Qloud Config') as qloud_block:
            qloud_service_name = sdk2.parameters.String('Qloud Service', default=DEFAULT_QLOUD_SERVICE, required=True)
            qloud_project_name = sdk2.parameters.String('Qloud Project', default=DEFAULT_QLOUD_PROJECT, required=True)
            qloud_environment_name = sdk2.parameters.String('Qloud Environment', default=DEFAULT_QLOUD_ENV, required=True)
            qloud_vault_name = sdk2.parameters.String('Qloud token vault name', default=DEFAULT_QLOUD_VAULT_NAME, required=True)

    def on_execute(self):
        with self.memoize_stage.build_and_publish_docker_image:
            revision = self.Parameters.arcadia_svn_revision or Arcadia.info(self.Parameters.arcadia_svn_url)['commit_revision']
            build_docker_image_task = sdk2.Task['YA_PACKAGE'](
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Building and publishing Docker image from Arcadia source',

                # Arcadia
                checkout_arcadia_from_url='{}@{}'.format(self.Parameters.arcadia_svn_url, revision),

                # Packages
                packages=self.Parameters.arcadia_package_config_path,
                package_type='docker',

                # Docker Registry
                docker_image_repository=self.Parameters.docker_image_repository,
                docker_user=self.Parameters.docker_user,
                docker_token_vault_name=self.Parameters.docker_vault_name,
                docker_push_image=True,

                # Advanced, YT store params
                custom_version='{revision}',
                ya_yt_store=False,
            )

            build_docker_image_task.enqueue()
            self.Context.ya_package_task_id = build_docker_image_task.id
            raise sdk2.WaitTask(
                [build_docker_image_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        with self.memoize_stage.deploy_to_qloud:
            build_docker_image_task = self.find(id=self.Context.ya_package_task_id).first()
            if build_docker_image_task.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure('BuildDockerImage failed')

            components = []
            for resource in sdk2.Resource['YA_PACKAGE'].find(task=build_docker_image_task).limit(50):
                components.append({
                    "name": resource.resource_name,
                    "docker_repository": resource.resource_version,
                    "image_hash": self.__get_image_hash(resource.resource_version),
                })
            environment_id = ".".join([self.Parameters.qloud_service_name, self.Parameters.qloud_project_name, self.Parameters.qloud_environment_name])
            token = sdk2.Vault.data(self.owner, self.Parameters.qloud_vault_name)

            self._deploy_to_qloud(token, environment_id, components)

    def __get_image_hash(self, docker_repository):
        url, tag = docker_repository.split(':')
        r = requests.get('https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={url}&tag={tag}'.format(
            url=url,
            tag=tag,
        ))
        r.raise_for_status()
        return r.json()['hash']

    def _deploy_to_qloud(self, token, environment_id, components):
        QLOUD_API_URL = 'https://platform.yandex-team.ru/api/v1/environment'
        headers = {'authorization': 'OAuth ' + token}

        r = requests.get(
            '{}/{}/{}'.format(QLOUD_API_URL, 'current', environment_id),
            headers=headers,
        )
        r.raise_for_status()
        env_version = r.json()['version']

        data = []
        for component in components:
            data.append({
                "componentName": component["name"],
                "properties": {
                    "repository": component["docker_repository"],
                    "hash": component["image_hash"],
                }
            })
        postr = requests.post(
            '{}/{}/{}/{}'.format(QLOUD_API_URL, 'fast-deploy', environment_id, env_version),
            headers=headers,
            json=data,
        )
        postr.raise_for_status()
