# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common import environments
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.arcadia.sdk import do_build


DEFAULT_VAULT_OWNER = 'TT-BACKEND'
DB_MIGRATION_UTIL = 'yabs/telephony/backend/dbschema/liquibase-cli'
DEFAULT_DB_HOSTS = 'vla-uddrqno4ehbq1h2j.db.yandex.net:6432,man-2ffh3frzy3u94pg3.db.yandex.net:6432'
DEFAULT_DB_URL = 'jdbc:postgresql://' + DEFAULT_DB_HOSTS + '/telephony_test?prepareThreshold=0&targetServerType=master'
DEFAULT_DB_USER = 'robot_telephony'
DEFAULT_DB_PASSWORD_VAULT_NAME = 'robot-telephony-test-db-password'
SSL_PARAMS = 'ssl=true&sslrootcert=/etc/ssl/certs/YandexInternalRootCA.pem'
DEFAULT_CHANGELOG = '/changelog-pbx.yaml'


class TelephonyDbMigration(sdk2.Task):
    '''
        Apply liquibase migration to telephony database
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.SandboxJavaJdkEnvironment('11.0.2'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Arcadia') as versions_block:
            checkout_arcadia_from_url = sdk2.parameters.String('Svn url with standard name', default=Arcadia.trunk_url(), required=True)

        with sdk2.parameters.Group('Database') as database_block:
            db_url = sdk2.parameters.String('Database URL', default=DEFAULT_DB_URL, required=True)
            db_user = sdk2.parameters.String('Database user', default=DEFAULT_DB_USER, required=True)
            db_password_vault_name = sdk2.parameters.String('Database password vault name', default=DEFAULT_DB_PASSWORD_VAULT_NAME, required=True)
            db_ssl = sdk2.parameters.Bool('Use ssl connection', default=True, required=True)
            db_changelog = sdk2.parameters.String('Database changelog to use', default=DEFAULT_CHANGELOG, required=True)

        with sdk2.parameters.Group('Liquibase') as liquibase_block:
            liquibase_args = sdk2.parameters.String('Liquibase arguments', required=False)

    def on_execute(self):
        url = self.Parameters.checkout_arcadia_from_url
        with sdk.mount_arc_path(url, use_arc_instead_of_aapi=True) as source_root:
            db_password = sdk2.Vault.data(self.owner, self.Parameters.db_password_vault_name)
            db_url = self.Parameters.db_url
            if self.Parameters.db_ssl:
                if '?' in db_url:
                    db_url += '&' + SSL_PARAMS
                else:
                    db_url += '?' + SSL_PARAMS
            db_changelog = self.Parameters.db_changelog
            cur_dir = os.getcwd()
            do_build('ya', source_root, [DB_MIGRATION_UTIL], clear_build=False, results_dir=cur_dir)
            cmd = [
                'java',
                '-cp', os.path.join(cur_dir, 'yabs/telephony/backend/dbschema/liquibase-cli/dbschema.liquibase-cli/*'),
                'ru.yandex.telephony.backend.dbschema.liquibase_cli.Main',
                '--url', db_url,
                '--username', self.Parameters.db_user,
                '--changelog', db_changelog
            ]
            if self.Parameters.liquibase_args:
                cmd = cmd + self.Parameters.liquibase_args.split(',')
            env = os.environ.copy()
            env.update({
                'PGPASSWORD': db_password,
            })
            logging.info('Start subprocess {}'.format(' '.join(cmd)))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('liquibase')) as pl:
                ps = subprocess.Popen(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT, env=env)
                ps.wait()
                if ps.returncode != 0:
                    raise Exception('Subprocess failed with exit code={}'.format(ps.returncode))
            logging.info('Migration done')
