import json
import logging
import os
import itertools
import re

from sandbox import common, sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.errors import TaskFailure

from sandbox import sandboxsdk
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.telephony.lib.nodejs import NodeUtils
import sandbox.projects.common.constants as consts
import sandbox.projects.common.nanny.nanny as nanny
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.telephony import TelephonyNode
from sandbox.projects.common.build import ArcadiaTask
from sandbox.projects.common.build import parameters
from sandbox.projects.common import error_handlers as eh
from sandbox.projects import resource_types

PACKAGE_RESOURCES = '_package_resources'
PUBLISH_TO_LIST = '_publish_to_list'


class PackagesBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Packages'


class PackagesParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'packages'
    description = 'Package paths, related to arcadia \';\'-separated'
    required = False
    multiline = True


class DockerBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Docker'


class DockerImageRepositoryParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_image_repository'
    description = 'Image repository'
    required = False
    default_value = ""


class DockerSaveImageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'docker_save_image'
    description = 'Save docker image in resource'
    required = False
    default_value = False


class DockerPushImageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'docker_push_image'
    description = 'Push docker image'
    required = False
    default_value = False


class DockerRegistryParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_registry'
    description = 'Docker registry'
    required = False
    default_value = "registry.yandex.net"


class DockerUserParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_user'
    description = 'Docker user'
    required = False
    default_value = ""


class DockerTokenVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_token_vault_name'
    description = 'Docker token vault name'
    required = False
    default_value = ""


class DockerBuildNetwork(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_build_network'
    description = 'Docker build network'
    required = False
    default_value = "host"


class HostPlatformParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'host_platform'
    description = 'Host platform'
    required = False
    default_value = ''


class TargetPlatformParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'target_platform'
    description = 'Target platform'
    required = False
    default_value = ''


class CompressPackageArchiveParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'compress_package_archive'
    description = 'Compress package archive'
    required = False
    default_value = True


class SemiClearBuildParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'semi_clear_build'
    description = 'Clear build cache directory only'
    required = False
    default_value = False


class ClearBuildParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'clear_build'
    description = 'Clear build'
    required = False
    default_value = False
    sub_fields = {
        'true': [
            SemiClearBuildParameter.name,
        ]
    }


class ArchitectureAllParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'architecture_all'
    description = 'Architecture: all'
    required = False
    default_value = False


class ResourcesBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Resources'


class ResourceTypeParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'resource_type'
    description = 'Created package resource type, or \';\'-separated list'
    required = True
    default_value = resource_types.YA_PACKAGE.name


class ResourceIdParameter(sandboxsdk.parameters.SandboxStringParameter):
    """If specified, task will use existing resources instead of creating new ones
    """
    name = 'resource_id'
    description = 'Previously created package resource id, or \';\'-separated list.'
    required = False
    default_value = ''


class SaveBuildOutputParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'save_build_output'
    description = 'Save build output in a separate resource'
    required = False
    default_value = True


class PublishingBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Publishing'


class MultiplePublishMappingParameter(sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'publish_to_mapping'
    description = "Package file -> repos ';' separated (has higher priority if filled)"
    default_value = None


class PublishPackageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'publish_package'
    description = 'Publish package'
    required = True
    default_value = True


class AdvancedBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Advanced'


class UseYaDevParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'use_ya_dev'
    description = 'Use ya-dev to build'
    required = False
    default_value = False


class UseNewFormatParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'use_new_format'
    description = 'Use new ya package json format'
    required = False
    default_value = False


class CustomVersionParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'custom_version'
    description = 'Custom version'
    required = False
    default_value = None


class PackageResourceDescriptionParameter(
    sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter
):
    name = 'package_resource_description'
    description = "Package file -> created package description"
    required = False
    default_value = None


class PackageResourceAttrsParameter(sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'package_resource_attrs'
    description = "Package resource attributes"
    default_value = None


class ContainerParameter(sandboxsdk.parameters.Container):
    name = consts.SANDBOX_CONTAINER
    description = 'Container the task should execute in'
    default_value = None
    required = False


class NodeEnv(sandboxsdk.parameters.SandboxStringParameter):
    name = 'node_env'
    description = 'Environment for which image is built'
    required = True


class SentryTokenVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'sentry_token_vault_name'
    description = 'Vault key for Sentry authorization token'
    required = True


class TelephonyNodeResource(sandboxsdk.parameters.ResourceSelector):
    name = 'telephony_node_resource'
    description = 'Telephony Node.js resource'
    resource_type = TelephonyNode
    required = False
    default_value = 2326742019


def _parse_multi_parameter(p):
    return [s for s in itertools.chain.from_iterable(x.split(';') for x in p.split()) if s]


class TelephonyNodeBuild(ArcadiaTask.ArcadiaTask, nanny.ReleaseToNannyTask):
    """
    Docker image builder for Node.js services
    Reference: https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/build/YaPackage.py
    """

    type = 'TELEPHONY_NODE_BUILD'
    client_tags = ctc.Tag.LXC | ctc.Tag.ARCADIA_HG
    input_parameters = [
        parameters.ArcadiaUrl,
        parameters.DoNotRemoveResources,

        parameters.BuildType,
        HostPlatformParameter,
        TargetPlatformParameter,
        ClearBuildParameter,
        SemiClearBuildParameter,
        parameters.ForceBuildDepends,
        parameters.ForceVCSInfoUpdate,
        parameters.IgnoreRecurses,
        parameters.Sanitize,
        parameters.Musl,
        parameters.LTO,
        parameters.ThinLTO,

        PackagesBlock,
        # Packages
        PackagesParameter,
        CompressPackageArchiveParameter,
        ArchitectureAllParameter,
        DockerBlock,
        DockerImageRepositoryParameter,
        DockerSaveImageParameter,
        DockerPushImageParameter,
        DockerRegistryParameter,
        DockerUserParameter,
        DockerTokenVaultName,
        DockerBuildNetwork,
        NodeEnv,
        SentryTokenVaultName,
        TelephonyNodeResource,

        ResourcesBlock,
        # Resources
        ResourceTypeParameter,
        ResourceIdParameter,
        SaveBuildOutputParameter,

        PublishingBlock,
        # Publishing
        PublishPackageParameter,
        MultiplePublishMappingParameter,

        AdvancedBlock,
        # Advanced
        UseYaDevParameter,
        UseNewFormatParameter,
        parameters.CheckoutModeParameter,
        parameters.CheckoutParameter,
        CustomVersionParameter,
        PackageResourceDescriptionParameter,
        PackageResourceAttrsParameter,
        parameters.BuildSystem,
        parameters.YaTimeout,
        ContainerParameter,
    ] + parameters.get_yt_store_params()

    execution_space = 256 * 1024
    arcadia_src_dir = None

    def resource_types_iter(self):
        rt = _parse_multi_parameter(self.ctx.get(ResourceTypeParameter.name))
        logging.info('$$ Resource types {}'.format(rt))
        return itertools.chain(rt, itertools.repeat(rt[-1]))

    def resource_ids_list(self):
        return [
            int(resource_id_str)
            for resource_id_str in _parse_multi_parameter(self.ctx.get(ResourceIdParameter.name, ""))
            if resource_id_str.isdigit()
        ]

    def resource_ids_iter(self):
        resource_ids = self.resource_ids_list()
        logging.info('$$ Resource ids {}'.format(resource_ids))
        return itertools.chain(resource_ids, itertools.repeat(None))

    def package_type_map(self):
        packages = _parse_multi_parameter(self.ctx.get(PackagesParameter.name))
        logging.info('$$ Packages {}'.format(packages))
        types = self.resource_types_iter()
        ids = self.resource_ids_iter()
        return zip(packages, types, ids)

    def on_enqueue(self):
        channel.task = self

        if PACKAGE_RESOURCES in self.ctx:
            return

        self.dns = ctm.DnsType.DNS64
        if not self.ctx.get(ContainerParameter.name):
            self.ctx[ContainerParameter.name] = 1035154683

        packages = []
        resources = []

        for path, resource_type, resource_id in self.package_type_map():
            packages.append(path)
            if resource_id is not None:
                resource = sdk2.Resource[resource_id]
                self.update_foreign_package_resource(path, resource)
            else:
                resource = self.init_package_resource(path, resource_type)
            resources.append(resource.id)

        self.ctx[PACKAGE_RESOURCES] = {'packages': packages, 'resources': resources}
        if self.ctx.get(MultiplePublishMappingParameter.name):
            self.ctx[PUBLISH_TO_LIST] = self.ctx.get(MultiplePublishMappingParameter.name)
        else:
            self.ctx[PUBLISH_TO_LIST] = []

    def on_execute(self):

        arcadia_url = self.ctx.get(parameters.ArcadiaUrl.name)
        parsed_arcadia_url = Arcadia.parse_url(arcadia_url)
        revision = str(parsed_arcadia_url.revision) if parsed_arcadia_url.revision else 'trunk'
        if not arcadia_url.startswith('arcadia-arc'):
            svn_ref = 'r' + str(parsed_arcadia_url.revision) if parsed_arcadia_url.revision else 'trunk'
            arcadia_url = 'arcadia-arc://#' + svn_ref

        if not self.ctx.get(DockerPushImageParameter.name):
            self.set_info("<strong>Docker image will not be pushed due to 'Push docker image' is set to False</strong>")

        packages = self.ctx[PACKAGE_RESOURCES]['packages']
        eh.ensure(packages, "Packages are not specified")
        logging.info('Packages: %s', packages)

        self._docker_login()

        with sdk.mount_arc_path(arcadia_url) as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir

            logging.info('Arcadia path: {}, r{}'.format(self.arcadia_src_dir, revision))
            self.ctx['revision'] = revision
            common.rest.Client().task.current.context.value = {'key': 'revision', 'value': self.ctx['revision']}
            resources = iter(self.ctx[PACKAGE_RESOURCES]['resources'])

            # Main function
            def _do_package():
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('subprocess')) as pl:
                    # Get Nodejs
                    node_resource_path = self.sync_resource(self.ctx[TelephonyNodeResource.name])
                    logging.info('Node.js path: %s', node_resource_path)

                    for package in packages:
                        # Get service name
                        service_name = os.path.basename(os.path.dirname(str(package)))
                        logging.info('Service name: %s', service_name)

                        # Install dependencies and generate files
                        node_utils = NodeUtils(
                            archive_path=node_resource_path,
                            arc_root=arcadia_src_dir,
                            process_log=pl,
                        )
                        node_utils.install_dependencies()
                        node_utils.run_script('generate-proto')

                        # Build Docker image
                        sentry_token = sdk2.Vault.data(self.owner, self.ctx.get(SentryTokenVaultName.name, None))
                        release_env = self.ctx.get(NodeEnv.name, None)
                        custom_version = self.ctx.get(CustomVersionParameter.name, None)
                        subprocess_env = os.environ.copy()
                        subprocess_env['SENTRY_AUTH_TOKEN'] = sentry_token
                        subprocess_env['ENV'] = release_env
                        if custom_version:
                            subprocess_env['CUSTOM_VERSION'] = custom_version
                        build_log = node_utils.run_script(
                            'build:docker',
                            return_output=True,
                            service_name=service_name,
                            env=subprocess_env
                        )
                        docker_image = re.search('Successfully tagged (.+)', build_log).group(1)
                        logging.info('Docker image: %s', docker_image)

                        resource_id = next(resources)
                        log_path = service_name
                        with open(log_path, mode='w+') as log_file:
                            log_file.write(build_log)
                        self.populate_package_resource(
                            service_name,
                            resource_id,
                            log_path,
                            docker_image,
                            revision,
                        )

            try:
                _do_package()
            except Exception:
                output_html = os.path.join(self.log_path(), "output.html")
                if os.path.exists(output_html):
                    self.set_info(
                        "See human-readable build errors here: "
                        "<a href='{}/output.html'>output.html</a>".format(self._log_resource.proxy_url),
                        do_escape=False
                    )
                raise
            finally:
                if self.ctx.get(SaveBuildOutputParameter.name) and os.path.exists('build') and os.listdir('build'):
                    self.create_resource(
                        description='Build output',
                        resource_path='build',
                        resource_type=resource_types.BUILD_OUTPUT
                    )

    def _get_common_package_resource_attrs(self):
        arcadia_url = self.ctx.get(parameters.ArcadiaUrl.name)
        do_not_remove = self.ctx.get(parameters.DoNotRemoveResources.name)
        parsed_url = sandboxsdk.svn.Arcadia.parse_url(arcadia_url)
        revision, branch, tag = parsed_url[1], parsed_url[2], parsed_url[3]
        build_type = self.ctx.get(parameters.BuildType.name)

        return {
            'svn_path': arcadia_url,
            'svn_revision': self.ctx.get('revision') or revision or 'HEAD',
            'build_type': build_type,
            'branch': branch or tag or 'trunk',
            'platform': 'unknown',
            'ttl': 'inf' if do_not_remove else 30,
        }

    def update_foreign_package_resource(self, path, resource):
        attrs = self._get_common_package_resource_attrs()
        attrs.update(self.ctx.get(PackageResourceAttrsParameter.name) or {})
        for attr_key, attr_value in attrs.items():
            setattr(resource, attr_key, attr_value)

    def init_package_resource(self, path, resource_type):
        resource_filename = path.replace('/', '.')
        if self.ctx.get(PackageResourceDescriptionParameter.name):
            resource_description = self.ctx.get(PackageResourceDescriptionParameter.name).get(path, path)
        else:
            resource_description = path
        attrs = self.ctx.get(PackageResourceAttrsParameter.name) or {}
        attrs.update(self._get_common_package_resource_attrs())
        return self._create_resource(
            resource_description,
            resource_filename,
            resource_type,
            attrs=attrs,
        )

    def populate_package_resource(self, name, resource_id, path, version, revision):
        prepare_foreign_resource = resource_id in self.resource_ids_list()
        if prepare_foreign_resource:
            logging.warning("Skipping change resource basename because resource #%s is not mine", resource_id)
        else:
            self.change_resource_basename(resource_id, os.path.basename(path))

        self.set_info('{}={}'.format(name, version))
        self.ctx['output_resource_version'] = version

        if revision:
            channel.sandbox.set_resource_attribute(resource_id, 'svn_revision', revision)

        channel.sandbox.set_resource_attribute(
            resource_id, 'platform', self.ctx.get(TargetPlatformParameter.name) or self.platform
        )
        channel.sandbox.set_resource_attribute(resource_id, 'resource_name', name)
        channel.sandbox.set_resource_attribute(resource_id, 'resource_version', version)

        build_type = channel.sandbox.get_resource_attribute(resource_id, 'build_type')
        branch = channel.sandbox.get_resource_attribute(resource_id, 'branch')

        description = '.'.join([name, 'linux', build_type, branch, revision])
        channel.sandbox.set_resource_attribute(resource_id, 'description', description)

        if prepare_foreign_resource:
            logging.info("Prepairing data for parent resource #%s", resource_id)
            self.save_parent_task_resource(path, resource_id)
            self.mark_resource_ready(resource_id)

    def check_aapi_available(self):
        url = self.ctx.get(parameters.ArcadiaUrl.name)
        return sdk.wait_aapi_url(url)

    def on_release(self, additional_parameters):
        ArcadiaTask.ArcadiaTask.on_release(self, additional_parameters)
        image_name, _, image_tag = self.ctx.get('output_resource_version', '').partition(':')
        _, _, image_name = image_name.partition('/')
        release_payload = {
            'spec': {
                'type': "DOCKER_RELEASE",
                'docker_release': {
                    "image_name": image_name,
                    "image_tag": image_tag,
                    "release_type": additional_parameters['release_status'].upper(),
                }
            }
        }
        logging.info('Sending release of Docker task %s to Nanny', self.id)
        logging.info('Release payload: %s', json.dumps(release_payload, indent=4))
        result = self.nanny_client.create_release2(release_payload)
        logging.info('Release result is %s', result)

    def _docker_login(self):
        docker_token = sdk2.Vault.data(self.owner, self.ctx.get(DockerTokenVaultName.name, None))
        proc = self._subprocess('docker login {registry} -u {user} -p $DOCKER_TOKEN'.format(
            registry=self.ctx.get(DockerRegistryParameter.name),
            user=self.ctx.get(DockerUserParameter.name),
        ), wait=True, log_prefix='docker', shell=True, extra_env={'DOCKER_TOKEN': docker_token})
        if proc.returncode != 0:
            raise TaskFailure("Failed to login in Docker, code: {}".format(proc.returncode))
