import logging
import json
import calendar
import os
from sandbox import sdk2
from sandbox.projects.telephony.lib.nodejs import NodeUtils
from sandbox.sdk2.vcs.svn import Arcadia
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import environments
from sandbox.projects.telephony import TelephonyNode
from sandbox.projects.market.infra import TsumJsonResource
from sandbox.sdk2 import ResourceData
from sandbox.sdk2 import svn
from sandbox.projects.common.arcadia import sdk

HEAD_REVISION = 'HEAD'


class TelephonyNodeGenerateChangelogTask(sdk2.Task):
    """
    Changelog generator for Node.js services
    """

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            environments.GCCEnvironment(version='5.3.0')
        ]
        container_resource = sdk2.parameters.Container(
            platform='linux',
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 900
        node_resource = sdk2.parameters.Resource(
            'Node.js distributable resource',
            resource_type=TelephonyNode,
            default_value=1590016621,
            required=True,
        )
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Path to package',
            default_value='arcadia:/arc/trunk/arcadia/yabs/telephony/front',
            required=True,
        )
        module_path = sdk2.parameters.String(
            'Module path',
            description='Path to service folder',
            required=True
        )
        from_revision = sdk2.parameters.String(
            'From revision (exclusive)',
            required=True,
        )
        to_revision = sdk2.parameters.String(
            'To revision',
            default_value=HEAD_REVISION,
            required=True
        )
        current_branch = sdk2.parameters.ArcadiaUrl(
            'Branch path',
            description='Arcadia URL',
            required=True
        )
        with sdk2.parameters.Output:
            tsum_json_resource = sdk2.parameters.Resource('package.json', resource_type=TsumJsonResource)

    def on_execute(self):
        # parse Arcadia URL
        arcadia_url = self._parse_arcadia_url()
        logging.info('Arcadia URL: %s', arcadia_url)

        # get SVN log for specified commits
        svn_log = self._get_svn_log(arcadia_url)
        logging.info('Found %s commits', len(svn_log))

        # get service dependencies
        dependencies = self._get_dependency_list(arcadia_url)
        logging.info('Found %s dependencies', len(dependencies))
        logging.info('Dependencies: %s', ', '.join(dependencies))

        # form changelog
        changes = self._filter_changes(svn_log=svn_log, allowed_paths=dependencies)
        logging.info('Commits in changelog: %s', len(changes))

        # store changelog in resource
        self._make_tsum_json_resource(changes=changes)

    def _parse_arcadia_url(self):
        arcadia_url = self.Parameters.current_branch
        if len(arcadia_url.rsplit('/arcadia', 1)) == 1 and len(arcadia_url.rsplit('@', 1)) == 1:
            # Use only the repository Arcadia
            arcadia_url += '/arcadia'
        to_revision = str(self.Parameters.to_revision).strip() or HEAD_REVISION
        if to_revision != HEAD_REVISION:
            arcadia_url += '@{}'.format(to_revision)
        return arcadia_url

    def _get_svn_log(self, arcadia_url):
        from_revision = str(self.Parameters.from_revision).strip()
        to_revision = str(self.Parameters.to_revision).strip()

        logging.info('Getting commits from revision %s to %s', from_revision, to_revision)
        svn_log = svn.Arcadia.log(
            url=arcadia_url,
            revision_from=from_revision,
            revision_to=to_revision,
            fullpath=True,
            timeout=600
        )

        if svn_log and str(svn_log[0]['revision']) == from_revision:
            svn_log = svn_log[1:]
        return svn_log

    def _get_dependency_list(self, arcadia_url):
        with sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True) as arc_root:
            # form path to package
            parsed_arcadia_url = Arcadia.parse_url(self.Parameters.arcadia_url)
            relative_package_path = str(parsed_arcadia_url.subpath)
            package_path = arc_root + '/' + relative_package_path
            logging.info('Path to package: %s', package_path)

            # get service name
            service_name = self._get_service_name()
            logging.info('Service name: %s', service_name)

            # get Node.js
            node_resource_data = sdk2.ResourceData(self.Parameters.node_resource)
            node_resource_path = str(node_resource_data.path)
            logging.info('Node.js path: %s', node_resource_path)

            # build and test package
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('subprocess')) as pl:
                utils = NodeUtils(archive_path=node_resource_path, arc_root=arc_root, process_log=pl)
                utils.install_dependencies()
                utils.run_script('generate-proto')
                lines = utils.run_script('list:dependencies', service_name=service_name, return_output=True).splitlines()
                lines = map(lambda line: line.strip(), lines)
                lines = filter(lambda line: not line.startswith('>') and len(line) > 0, lines)
                return list(map(lambda line: self._strip_arcadia_path(line), lines))

    def _make_tsum_json_resource(self, changes):
        self.Parameters.tsum_json_resource = TsumJsonResource(self, description='packages.json', path='packages.json')
        app_res = ResourceData(self.Parameters.tsum_json_resource)
        app_res.path.write_bytes(json.dumps({
            'changelog': changes
        }))
        app_res.ready()

    def _filter_changes(self, svn_log, allowed_paths):
        changes = []
        for log_entry in svn_log:
            # Get paths from dicts and remove the branch substring
            changed_paths = [self._strip_arcadia_path(path_dict['text']) for path_dict in log_entry['paths']]
            for changed_path in changed_paths:
                is_found_match = False
                for allowed_path in allowed_paths:
                    # It should be faster then startswith but less accurate because substring `changed_path` could be
                    # in the middle or in the end of `allowed_path`
                    if allowed_path in changed_path:
                        # Ensure `allowed_path` is the prefix of `changed_path`
                        if changed_path.startswith(allowed_path):
                            logging.info('Commit r%s has been added into the list of changes', log_entry['revision'])
                            changes.append({
                                'author': log_entry['author'],
                                'revision': log_entry['revision'],
                                'change': log_entry['msg'],
                                'timestampSeconds': calendar.timegm(log_entry['date'].utctimetuple())
                            })
                            # Stop processing this `log_entry`
                            is_found_match = True
                            break
                if is_found_match:
                    break
        return changes

    def _strip_arcadia_path(self, path):
        return ''.join(path.split('arcadia/', 1)[1:])

    def _get_service_name(self):
        module_path = str(self.Parameters.module_path) + '/'
        return os.path.basename(os.path.dirname(module_path))
