import logging
from sandbox import sdk2
from sandbox.projects.telephony.lib.nodejs import NodeUtils
from sandbox.sdk2.vcs.svn import Arcadia
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import environments
from sandbox.projects.telephony import TelephonyNode
from sandbox.projects.common.arcadia import sdk as arcadia_sdk


class TelephonyNodeTestRunner(sdk2.Task):
    """
    Test runner for Node.js services
    """

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC
        environments = [
            environments.GCCEnvironment(version='5.3.0')
        ]
        container_resource = sdk2.parameters.Container(
            platform='linux',
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        node_resource = sdk2.parameters.Resource(
            'Node.js distributable resource',
            resource_type=TelephonyNode,
            default_value=2326742019,
            required=True,
        )
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Path to package',
            default_value='arcadia:/arc/trunk/arcadia/yabs/telephony/front',
            required=True,
        )
        arcadia_patch = sdk2.parameters.String(
            'Arcadia patch URL',
            required=False,
        )
        commit_hash = sdk2.parameters.String("Arc commit hash, without #", required=False)

    def on_execute(self):
        # parse parameters
        parsed_arcadia_url = Arcadia.parse_url(self.Parameters.arcadia_url)
        svn_ref = 'r' + str(parsed_arcadia_url.revision) if parsed_arcadia_url.revision else 'trunk'
        arc_hash = self.Parameters.commit_hash
        ref = arc_hash if arc_hash else svn_ref
        arc_url = 'arcadia-arc://#'+ ref
        relative_package_path = str(parsed_arcadia_url.subpath)

        # get Node.js
        node_resource_data = sdk2.ResourceData(self.Parameters.node_resource)
        node_resource_path = str(node_resource_data.path)
        logging.info('Node.js path: {}'.format(node_resource_path))

        # mount Arcadia
        with arcadia_sdk.mount_arc_path(arc_url) as arc_root:
            # apply patch
            if self.Parameters.arcadia_patch:
                logging.info('Applying patch: {}'.format(self.Parameters.arcadia_patch))
                Arcadia.apply_patch(
                    str(arc_root),
                    str(self.Parameters.arcadia_patch),
                    str(self.path())
                )

            # form path to package
            package_path = arc_root + '/' + relative_package_path
            logging.info('Path to package: {}'.format(package_path))

            # build and test package
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('subprocess')) as pl:
                utils = NodeUtils(archive_path=node_resource_path, arc_root=arc_root, process_log=pl)
                utils.install_dependencies()
                utils.run_script('generate-proto')
                utils.run_script('lint:check:all')
                utils.run_script('build:all')
                utils.run_script('test:unit')
