import os
import tarfile
from sandbox.sdk2.helpers import subprocess
from sandbox.common.errors import TaskFailure


class NodeUtils:
    def __init__(self, archive_path, arc_root, process_log, relative_package_path='yabs/telephony/front'):
        """
        :param archive_path: path to Node.js archive
        :param arc_root: path to Arc root
        :param process_log: process log object
        :param relative_package_path: path to folder with package.json relative to arc root
        """
        self._arc_root = arc_root
        self._process_log = process_log
        self._package_path = arc_root + '/' + relative_package_path
        with tarfile.open(archive_path) as tar:
            self._node_dir = os.getcwd() + '/' + tar.getnames()[0]
            tar.extractall(path=os.getcwd())

    def get_bin_dir(self):
        return self._node_dir + '/bin'

    def get_npm(self):
        return self.get_bin_dir() + '/npm'

    def run_script(self, script_name, return_output=False, service_name=None, env=None):
        return self._exec_command([self.get_npm(), 'run', script_name], return_output=return_output, service_name=service_name, env=env)

    def install_dependencies(self):
        self._exec_command([self.get_npm(), 'ci'], return_output=False, service_name=None, env=None)

    def _exec_command(self, cmd, return_output, service_name, env=None):
        node_path = self.get_bin_dir()
        subprocess_env = env if env is not None else os.environ.copy()
        subprocess_env['PATH'] = node_path + ':' + subprocess_env['PATH'] \
            if 'PATH' in subprocess_env else node_path
        subprocess_env['ARCROOT'] = self._arc_root
        if service_name is not None:
            subprocess_env['SERVICE_NAME'] = service_name

        proc = subprocess.Popen(
            cmd,
            cwd=self._package_path,
            env=subprocess_env,
            stdout=self._process_log.stdout if not return_output else subprocess.PIPE,
            stderr=self._process_log.stderr,
        )
        out = None
        if return_output:
            out = proc.communicate()[0]
            return_code = proc.returncode
        else:
            return_code = proc.wait()
        if return_code != 0:
            raise TaskFailure("Failed to execute command '{}', code: {}, output: {}".format(cmd, return_code, out))
        return out
