from __future__ import absolute_import

import pytest
import inspect

from sandbox import common

from sandbox import sdk2
from sandbox.sandboxsdk import task as sdk1_task


def _is_base_sdk1_task(cls):
    nearest_base_class = cls
    for base_cls in inspect.getmro(cls)[1:]:
        if issubclass(base_cls, (sdk2.Task, sdk1_task.SandboxTask)):
            nearest_base_class = base_cls
            break
    return nearest_base_class is sdk1_task.SandboxTask


def is_new_sdk1_task(cls):
    return _is_base_sdk1_task(cls) and cls.type not in _ALLOWED_SDK1_TASKS


@pytest.mark.usefixtures("project_types")
def test__sdk1_tasks():
    if common.system.inside_the_binary():
        # Run this test only inside binary: we do not want to stop tasks archive deploy if someone adds new sdk1-task.
        from sandbox import projects
        black_list = getattr(projects, "BLACK_LIST", [])
        assert not black_list, (
            "Task class{} {} {} based on `sandboxsdk.SandboxTask`. "
            "It's forbidden to create new SDK1-based tasks that inherit directly from `SandboxTask`. "
            "Inherit from existing SDK1-based tasks or use SDK2: "
            "https://wiki.yandex-team.ru/sandbox/tasks/sdk2/".format(
                ("es" if len(black_list) > 1 else ""),
                ", ".join(black_list),
                ("is" if len(black_list) == 1 else "are"),
            )
        )


@pytest.mark.usefixtures("project_types")
def test__dunder_tasks_are_sdk1():
    from sandbox import projects
    dunder_types = getattr(projects, "DUNDER_TYPES", {})
    dunder_sdk2_types = [cls.type for cls in dunder_types.itervalues() if not issubclass(cls, sdk1_task.SandboxTask)]
    assert not dunder_sdk2_types, (
        "SDK2 tasks cannot be declared with `__Task__`: {}"
        .format(dunder_sdk2_types)
    )


def _allowed_sdk1_tasks():
    sdk1_tasks = []
    for task_type, task_location in common.projects_handler.load_project_types(force_from_fs=True).iteritems():
        if _is_base_sdk1_task(task_location.cls):
            sdk1_tasks.append('    "{}",\n'.format(task_type))
    return "".join(sorted(sdk1_tasks))


# Please, do not add anything to this white list.
# Test is created to encourage usage of SDK2 and prevent from usage of deprecated SDK1.
_ALLOWED_SDK1_TASKS = {
    "ADDRS_BASE_PERFORMANCE_ACCEPTANCE",
    "ADDRS_DEPLOY",
    "ADDRS_SHARDED_BASESEARCH_PERFORMANCE",
    "ALLURE_REPORT_TASK",
    "ANALYZETHIS_IPREG",
    "ANNOTATE_CGI_PARAMS",
    "ANNOTATE_FML_2575_COMMENTS",
    "ANTIROBOT_COLLECT_FEATURES",
    "APP_PERFORMANCE_TEST",
    "AUTOCHECK_COMPARE_LOGS_BASE_TASK",
    "AUTOCHECK_DIFF",
    "AUTOCHECK_DIR_GRAPH_TASK",
    "AUTOCHECK_EMULATION_TASK",
    "AUTOCHECK_RESPONSIBILITY",
    "AUTOCHECK_STATISTICS_TASK",
    "AUTOCHECK_TESTS_STATISTICS",
    "BACKUP_MONGO",
    "BACKUP_MONGO_MANY_DATABASES",
    "BACKUP_NANNY_MONGODB",
    "BATCH_NOAPACHE_TEST_RESOURCES_UPDATER",
    "BATCH_WEB_RESULTS_CHECKER",
    "BB_VIEWER_UPDATE_BASES",
    "BEGEMOT_CREATE_RESPONSES_DIFF",
    "BEGEMOT_RUN_LIGHT_TESTS",
    "BEGEMOT_TEST_PERFORMANCE",
    "BENCHMARK_APP_HOST_ALL",
    "BENCHMARK_APP_HOST_CUSTOM",
    "BENCHMARK_APP_HOST_DIFF2",
    "BLENDER_CHECK_REARR_SET",
    "BLENDER_RD_TIMINGS_2_RAZLADKI",
    "BNO_APP_HOST_TO_APP_BUILD",
    "BNO_NEWS_HOSTS_BUILD",
    "BNO_RECIPES_BUILD",
    "BNO_UPLOAD_IMAGES",
    "BROADMATCH_BUILD_MR_CATALOGIA",
    "BUILD_ABGAME_BACKEND",
    "BUILD_ABGAME_FRONTEND",
    "BUILD_ADDRS_BUSINESS_URLS",
    "BUILD_ALEMATE",
    "BUILD_AND_DIFF_APP_HOST_BACKENDS_CONFIG",
    "BUILD_APACHE_BUNDLE",
    "BUILD_APP_HOST_BACKENDS_CONFIG",
    "BUILD_APP_HOST_CONFIG_BUNDLE_AUTO",
    "BUILD_APP_HOST_CONFIG_BUNDLE_DIFF",
    "BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE_DIFF",
    "BUILD_AURORA_BUNDLE",
    "BUILD_AWACS",
    "BUILD_BALANCER_NANNY_TRAFFIC_CONTROL_DASHBOARDS",
    "BUILD_BASESEARCH_STUB",
    "BUILD_BEGEMOT_CONFIG",
    "BUILD_BEGEMOT_LIGHT_TEST_CONFIG",
    "BUILD_BLOCKSTAT_DICT",
    "BUILD_BREAKPAD",
    "BUILD_BSCONFIG",
    "BUILD_CLICKHOUSE_YASM",
    "BUILD_CLUSTERMASTER_DEB",
    "BUILD_CLUSTERSTATE",
    "BUILD_CMAKE",
    "BUILD_CONVEYOR_DASHBOARD",
    "BUILD_CONVEYOR_DASHBOARD_BUSINESS_LOGIC",
    "BUILD_CONVEYOR_DASHBOARD_CACHER",
    "BUILD_CONVEYOR_DASHBOARD_FRONT",
    "BUILD_CONVEYOR_DASHBOARD_FRONT_PROXY",
    "BUILD_CONVEYOR_DASHBOARD_MK2",
    "BUILD_CONVEYOR_DASHBOARD_MK2_TEST",
    "BUILD_CORES_VIEW",
    "BUILD_CUSTOM_NGINX",
    "BUILD_CY_HASH",
    "BUILD_DELAYED_VIEW_ENTITY_BASE_TRIE",
    "BUILD_DELAYED_VIEW_SERIAL_BASE_TRIE",
    "BUILD_DOCKER_IMAGE",
    "BUILD_DOCKER_IMAGE_V6",
    "BUILD_DOCKER_IMAGE_V6_01",
    "BUILD_DOCKER_IMAGE_V6_TEST_RESOURCE",
    "BUILD_DYNAMIC_MODELS",
    "BUILD_ETCD",
    "BUILD_EXPERIMENTS_ADMINKA",
    "BUILD_EXPERIMENTS_ADMINKA_ENV",
    "BUILD_EXPERIMENTS_ADMINKA_GEO",
    "BUILD_EXP_MR_SERVER",
    "BUILD_FAKE_RESOURCE",
    "BUILD_FILTER_TRIE",
    "BUILD_FLAME_GRAPH_BUNDLE",
    "BUILD_FRESHNESS_SCRIPTS",
    "BUILD_GENISYS",
    "BUILD_GOSKY",
    "BUILD_GO_PACKAGE",
    "BUILD_GPERFTOOLS",
    "BUILD_GREENBOX_BUNDLE",
    "BUILD_HAPROXY",
    "BUILD_HASHICORP_VAULT",
    "BUILD_HTTP_GRIDFS",
    "BUILD_ICOOKIE_BLACKLIST",
    "BUILD_INFECTED_SERP_MASKS",
    "BUILD_INSTANCE_CTL",
    "BUILD_ISS_SHARDS",
    "BUILD_ITS",
    "BUILD_JAVA_JDK",
    "BUILD_JUPITER_SHARD",
    "BUILD_LOADLOGDUMP",
    "BUILD_MAPS_COMPANIES_POI_AND_URLS",
    "BUILD_MAPS_DATABASE_ADVERT",
    "BUILD_MAPS_DATABASE_POI",
    "BUILD_MAPS_FAST_FEATURES",
    "BUILD_MAPS_WEB_INDEXANN",
    "BUILD_MARKET_DOCKER_IMAGES",
    "BUILD_MARKET_GURU_ASSISTANT_AUTO_WHEELS",
    "BUILD_MARKET_GURU_ASSISTANT_DATA",
    "BUILD_MARKET_PACKAGE_REBUILD",
    "BUILD_MARKET_SERVICE_CONFIG",
    "BUILD_MEDIAPORTRAITS_FRONTEND",
    "BUILD_MEMCACHED",
    "BUILD_MONGO_TOOLS_PACKAGE",
    "BUILD_NANNY",
    "BUILD_NANNY_ALERT_SYNCER",
    "BUILD_NANNY_DEB",
    "BUILD_NANNY_ETCD",
    "BUILD_NANNY_FEDERATED",
    "BUILD_NANNY_HQ",
    "BUILD_NANNY_PLANNED_EVENT_PROCESSOR",
    "BUILD_NEWS_ARCHIVE_SHARD",
    "BUILD_NEWS_PACKAGE",
    "BUILD_NEWS_REPORT_CORE",
    "BUILD_NEWS_RKN_VIEWER",
    "BUILD_NEWS_SEARCH_SHARD",
    "BUILD_NEWS_UATRAITS_BROWSER_XML",
    "BUILD_NEWS_VIRTUALENV",
    "BUILD_NGINX_CPLB",
    "BUILD_NINJA",
    "BUILD_NISHOOTER_FRONT",
    "BUILD_NOAPACHEUPPER_CONFIG",
    "BUILD_NOAPACHEUPPER_REQUESTS",
    "BUILD_NOAPACHEUPPER_REQUESTS2",
    "BUILD_ONTODB_FIXES_STICKY",
    "BUILD_PERCONA_XTRABACKUP",
    "BUILD_PERCONA_XTRADB_CLUSTER",
    "BUILD_PLATFORM_BASE_IMAGES",
    "BUILD_PORTO",
    "BUILD_PORTO_LAYER",
    "BUILD_PROJECT_STUB_NODEJS_PACKAGE",
    "BUILD_PYSTICK",
    "BUILD_PYTHON_BUNDLE",
    "BUILD_PYTHON_BUNDLE2",
    "BUILD_PYTHON_DEV",
    "BUILD_RAZLADKI",
    "BUILD_READAHEAD",
    "BUILD_REPLICAMAP",
    "BUILD_REPLICAMAP2",
    "BUILD_REPLICAMAP3",
    "BUILD_REPLICAMAPC1R1",
    "BUILD_REPLICAMAPTEST",
    "BUILD_REVIEW_BOARD",
    "BUILD_ROTOR_WL",
    "BUILD_RTLINE_CONFIG",
    "BUILD_RTYSERVER_CONFIG",
    "BUILD_RTYSERVER_MAPS_USER_IF",
    "BUILD_RTYSERVER_MODELS",
    "BUILD_SALMON",
    "BUILD_SEMANTIC_MF_VERIFIER",
    "BUILD_SEMANTIC_PUBLIC_API",
    "BUILD_SEMANTIC_YT_MAPPER",
    "BUILD_SENTRY_VIRTUALENV",
    "BUILD_SEPELIB",
    "BUILD_SHARDMAP",
    "BUILD_SITESEARCH_MANAGER",
    "BUILD_SITESEARCH_SUGGEST",
    "BUILD_SKYCORE_EXAMPLE",
    "BUILD_SKYCORE_YASM_AGENT",
    "BUILD_SKYNET",
    "BUILD_SKYNET_DEPS",
    "BUILD_SKYNET_SERVICE",
    "BUILD_SNOWDEN_ARCHIVE",
    "BUILD_SOCIAL_PROFILE_2_GEOIDS",
    "BUILD_SPECSEARCH_PERL_BUNDLE",
    "BUILD_SPORT_PROXY_DATA",
    "BUILD_SPORT_PUSH_API",
    "BUILD_STATBOX_PUSHCLIENT_OLD",
    "BUILD_STATS_NANNY_SERVICES",
    "BUILD_SWITCH_ADMINKA",
    "BUILD_TANK_VIRTUALENV",
    "BUILD_TEMPLAR_BUNDLE",
    "BUILD_TEST_ENVIRONMENT_UI",
    "BUILD_UPPER_CONFIG",
    "BUILD_VIDEO_POPULAR_CONFIG",
    "BUILD_VIRTUALENV",
    "BUILD_WALLE_UI",
    "BUILD_YADI_TOOL",
    "BUILD_YASAP_HOT_FRONTEND",
    "BUILD_YASAP_ISEG",
    "BUILD_YASAP_PDB_FRONTEND",
    "BUILD_YASM_DOCS",
    "BUILD_YOBJECT",
    "BUILD_ZOOKEEPER",
    "BUILD_ZOOLOCK",
    "BUILD_ZOOREST",
    "CALCULATE_MATRIXNET_MODELS_AND_PREDICTIONS",
    "CALCULATE_MATRIXNET_MODEL_AND_PREDICTIONS",
    "CALCULATE_MATRIXNET_PREDICTIONS_MEAN",
    "CALC_BUSINESS_RATING",
    "CALC_COVERAGE",
    "CHECK_FRESH_DOCUMENTS",
    "CHECK_MATRIXNET_PREDICTIONS_DIFF",
    "CHECK_PORTS",
    "CHECK_RAZLADKA_PARSELIB",
    "CLEANUP_YT_DIRECTORY",
    "CLEAN_UP_BAD_PSI_CONTAINERS",
    "CLICKDAEMON_KEYGEN",
    "COCAINE_LAYERS_EXPORT_TO_REGISTRY",
    "COLLECT_ANTIROBOT_DATA",
    "COLLECT_TOP_QUERIES",
    "COMBINE_MIDDLESEARCH_QUERIES",
    "COMBINE_MR_TABLES",
    "COMBINE_NOAPACHEUPPER_REARRANGES",
    "COMPARE_BASESEARCH_MEMORY",
    "COMPARE_BASESEARCH_PERFORMANCE",
    "COMPARE_DEPENDENCIES",
    "COMPARE_DYNAMIC_MODELS",
    "COMPARE_ESTFEATURES_OUTPUTS",
    "COMPARE_EVENTLOG_STATS",
    "COMPARE_FLAME_GRAPHS",
    "COMPARE_GEOMETASEARCH_BINARIES",
    "COMPARE_GEOSEARCH_OUTPUTS",
    "COMPARE_GEOSEARCH_REQANS_LOGS",
    "COMPARE_GEOSEARCH_RESPONSES",
    "COMPARE_IDX_QUERY_OUTPUTS",
    "COMPARE_LOADLOGS",
    "COMPARE_MATRIXNETS",
    "COMPARE_METASEARCH_SUBSOURCE_REQUESTS",
    "COMPARE_MIDDLESEARCH_CACHEHIT",
    "COMPARE_MR_PROTO_POOLS",
    "COMPARE_NEWSD_RESPONSES",
    "COMPARE_NOAPACHEUPPER_REQUEST_TIMINGS",
    "COMPARE_PROFILE_STATS",
    "COMPARE_PROTO_POOLS",
    "COMPARE_QLOSS_OUTPUTS",
    "COMPARE_UNISTAT",
    "COPY_DATA_RUNTIME",
    "COUNT_UNKWNOWN_TASKS",
    "COVERAGE_YA_MAKE_TASK_SEMINANTI",
    "CREATE_CLUSTERSTATE_VENV",
    "CREATE_CROSS_MSVC_TOOLKIT",
    "CREATE_JAVA_JDK",
    "CREATE_PACKAGE",
    "CREATE_SESSIONS_TESTS",
    "CREATE_STATINFRA_EVENT_PROCESSOR_RESOURCE",
    "CREATE_TEXT_RESOURCE",
    "CREATE_UKROP_BRANCH",
    "CREATE_VENV",
    "DELETE_OLD_RESOURCES",
    "DELETE_OLD_RESOURCES_GEO",
    "DEPLOY_BAOBAB_PYTHON_API",
    "DEPLOY_MARKET_GURU_ASSISTANT_DATA",
    "DEPLOY_MARKET_SERVICE",
    "DEPLOY_NANNY_DASHBOARD",
    "DEPLOY_NANNY_DASHBOARD_DEV",
    "DEPLOY_NANNY_RELEASE_REQUESTS",
    "DEPLOY_NANNY_SHARDMAP",
    "DEPLOY_REFRESH_FROZEN",
    "DEPLOY_UKROP",
    "DEPLOY_VIDEO_BASE_NEWDB_SHARDMAP",
    "DETACH_RTYSERVER_INDEX",
    "DETACH_SERVICE_INDEX",
    "DIFF_ABT_METRICS",
    "DRAW_FRESHNESS_PLOTS",
    "DUMP_AND_CALC_EVENTLOG",
    "DUMP_EVENTLOG",
    "ENTITYSEARCH_CONFIG_BUILD",
    "ENTITYSEARCH_LOGS",
    "ENTITYSEARCH_NMETA_REQS_TO_APPHOST_REQS",
    "ENTITYSEARCH_REALTIME_GENERATION",
    "ENTITYSEARCH_REALTIME_RELEASE",
    "ERF_MONITOR",
    "EXPERIMENTS_CONFIG_CHECK",
    "EXPERIMENTS_CONFIG_UPDATE",
    "EXTRACT_PANTHER_GUTS",
    "FAST_CHECK_BASESEARCH_RESPONSES",
    "FETCHING_DATA_FOR_CREATE_SESSIONS_TESTS",
    "FILTER_EVENTLOG",
    "FORMING_WEEKLY_REPORT",
    "FUNCTIONAL_TEST_SEARCH",
    "GENERATE_BALANCER_ACTIVE_CHECKS",
    "GENERATE_GENCFG_CLICKHOUSE_TABLES",
    "GENERATE_LINEAR_MODEL_BINARY_DUMP",
    "GENERATE_LINEAR_MODEL_BINARY_DUMP_LAUNCHER",
    "GENERATE_LINEAR_MODEL_SHARDING_BASE",
    "GENERATE_OCSP_RESPONSE_FILES",
    "GENERATE_PLAN_FROM_QUERIES",
    "GENERATE_SLOW_PLAN",
    "GENERATE_SPORT_DEVICE_IDS",
    "GENERATE_THUMB_DAEMON_REQUESTS",
    "GENERATE_USERS_QUERIES",
    "GEODATA3BIN_STABLE",
    "GEODATA5BIN_PRESTABLE",
    "GEODATA5BORDERS_STABLE",
    "GEODATA5BORDERS_TESTING",
    "GEODATA5BORDERS_VIEWER",
    "GEODATATZDATA_STABLE",
    "GEODATA_VERSIONS",
    "GEO_SUGGEST_DOWNLOAD_YT_TABLE",
    "GEO_SUGGEST_MAKE_PREVIEW",
    "GEO_SUGGEST_TEST_PERFORMANCE_BEST",
    "GET_ADDRS_REQUESTS",
    "GET_ADVQUICK_DATABASE",
    "GET_BASESEARCH_DATABASES",
    "GET_BASESEARCH_QUERIES_PROD_SIMULATION",
    "GET_BASESEARCH_RESPONSES",
    "GET_FML_CONVEYOR_AUTOFORMULAS_SNAPSHOT",
    "GET_FORMULA_CONVEYORS_LIST",
    "GET_FUSION_MIDDLESEARCH_RESPONSES",
    "GET_FUSION_PROD_CONFIGS",
    "GET_IMAGES_MR_INDEX_CONFIG",
    "GET_IMAGES_RATED_URLS",
    "GET_LEMUR_VINTAGE_MAP_TEST_DATA",
    "GET_MAPS_SEARCH_BUSINESS_RESPONSES",
    "GET_MARKET_RESOURCES",
    "GET_MIDDLESEARCH_RESOURCES",
    "GET_MMETA_AND_INT_RESPONSES",
    "GET_MMETA_RESPONSES",
    "GET_NEWSD_APPHOST_RESPONSES",
    "GET_NEWSD_RESPONSES",
    "GET_POKAZOMETER_DATABASE",
    "GET_QUERIES_FROM_EVENTLOG",
    "GET_QUERIES_FROM_PLAN",
    "GET_RANDOM_REQUESTS",
    "GET_REFRESH_INDEX_FOR_TEST",
    "GET_REPORT_SIMULTANEOUS_RESPONSES",
    "GET_ROUTERD_RESPONSES",
    "GET_WIZARD_PRINTWZRD_RESPONSES",
    "GET_WIZARD_RESPONSES",
    "HASHED_SESSIONS_DIFF",
    "HTTP_UPLOAD",
    "IEX_BUILD_PACKAGES",
    "IEX_COMPARE_FUNC_TEST_RESULTS",
    "IEX_GET_LAST_RESULTS",
    "IEX_IMPORT_PATTERNS",
    "IEX_RELEASE",
    "IEX_TESTS_MANAGER",
    "IEX_UPDATE_CONTENTLINE_MODEL",
    "IMAGES_ANALYZE_CBIRDAEMON2_PERFORMANCE",
    "IMAGES_ANALYZE_NAILDAEMON_PERFORMANCE",
    "IMAGES_ANALYZE_RIMDAEMON_PERFORMANCE",
    "IMAGES_BUILD_DYNAMIC_MODELS",
    "IMAGES_BUILD_IMPROXY",
    "IMAGES_BUILD_PRISM_VIRTUALENV",
    "IMAGES_BUILD_TAGS_MMETA_MODELS_BUNDLE",
    "IMAGES_CHECK_BASESEARCH_DATABASE",
    "IMAGES_COMPARE_ANALYZE_PERFORMANCE",
    "IMAGES_COMPARE_CBIRDAEMON2_APPHOST_RESPONSES",
    "IMAGES_COMPARE_CBIRDAEMON2_RESPONSES",
    "IMAGES_COMPARE_COMMERCIAL_RESPONSES",
    "IMAGES_COMPARE_DYNAMIC_MODELS",
    "IMAGES_CREATE_THUMBS_PLAN_FROM_YT",
    "IMAGES_DOWNLOAD_SNIPPET_CONTEXTS",
    "IMAGES_GENERATE_BASE_USER_QUERIES",
    "IMAGES_GENERATE_CBIRDAEMON2_REQUESTS",
    "IMAGES_GENERATE_CBIRSIGN_USER_QUERIES",
    "IMAGES_GENERATE_CBIR_USER_QUERIES",
    "IMAGES_GENERATE_MIDDLESEARCH_REQUESTS",
    "IMAGES_GENERATE_NAILDAEMON_REQUESTS",
    "IMAGES_GENERATE_RIMDAEMON_REQUESTS",
    "IMAGES_GET_PAGE_RELEVANCES",
    "IMAGES_GET_SAAS_RESPONSES",
    "IMAGES_GRAY_QUERIES_FILTER",
    "IMAGES_LOAD_FML_TESTENV_DATA",
    "IMAGES_MANUAL_BOOSTING_REFRESH",
    "IMAGES_MR_INDEX_CREATE_BRANCH",
    "IMAGES_PRIEMKA_BASESEARCH_MODELS",
    "IMAGES_PRIEMKA_MIDDLESEARCH_MODELS",
    "IMAGES_PRIEMKA_THUMBS",
    "IMAGES_PROD_COMPARE_RIM_RESPONSES",
    "IMAGES_PROD_PARSE_RIM_RESPONSES",
    "IMAGES_RT_ULTRA_ACCEPTANCE",
    "IMAGES_SAMPLE_CBIR_PLAN",
    "IMAGES_STAGING_BASESEARCH",
    "IMAGES_STORE_INDEX_LOCATION",
    "IMAGES_TEST_COMMERCIAL_TRIE",
    "IMAGES_TEST_METASEARCH_UPDATE",
    "IMAGES_TEST_MIDDLESEARCH_BAN",
    "IMAGES_TEST_MRINDEX",
    "IMAGES_TEST_MRINDEX_COMPARE_TABLES",
    "IMAGES_TEST_MRINDEX_METADOC",
    "IMAGES_TEST_PUMPKIN_SERP",
    "IMAGES_ULTRA_ACCEPTANCE",
    "IMAGES_UPLOAD_SAAS",
    "IMPORT_CACHER_HOOKS_AND_CONFIGS",
    "IMPROXY_GENERATE_REQUESTS",
    "IMPROXY_GET_PROD_RESPONSES",
    "IMPROXY_SET_BASE_VERSION",
    "IMPROXY_TEST_FUNC",
    "IMPROXY_TEST_PERFORMANCE",
    "INSTALL_VIDEO_QUOTES_TASK",
    "ISS_CRUTCH",
    "ITS_SET_VALUE",
    "IZNANKA_COMPANY_URLS_BUILDER",
    "KIMKIM_TEST",
    "LAUNCH_RUNTIME_CLOUD_BALANCER_TASKS",
    "LB_CHECK_SAAS_CONSISTENCY",
    "LB_GET_SAAS_RESPONSES",
    "LOAD_MARKET_REPORT_INDEX",
    "MAKE_CONVEYOR_FML_PROCESS_TEMPLATES_LIST_REPORT",
    "MAKE_FML_CONVEYOR_FORMULA_COMMIT_REPORT",
    "MAKE_GEODB_DATA_2",
    "MAKE_GEODB_DATA_FROM_REVISION",
    "MAKE_MAPS_PACKAGES_LIST",
    "MAKE_NEW_PERSONAL_RESOURCE",
    "MAKE_NEW_PERSONAL_RESOURCE_SUBTASK",
    "MAKE_NIRVANA_WORKFLOW_SNAPSHOT",
    "MAKE_PERSONAL_BUNDLE",
    "MAKE_SHARD",
    "MAKE_TRAIN_MATRIXNET_ON_CPU_NIRVANA_OPS",
    "MARKET_CATALOGER_SHOOTER",
    "MARKET_CATALOGER_SHOOTER2",
    "MARKET_DEBUILDER",
    "MARKET_DO_SOME_TEST",
    "MARKET_HELLO_WORLD",
    "MARKET_NGINX_BINARY",
    "MARKET_SERVER_COUNT_TO_STAT",
    "MARKET_WIZARD_TIRES",
    "MEDIA_COPY_SANDBOX_RESOURCE",
    "MEDIA_TASK_MONITORING",
    "MIDDLESEARCH_LOAD",
    "ML_ENGINE_DUMP_MONITOR",
    "MMETA_FREQUENT_STABILITY",
    "MODADVERT_BANNERS_KARMA",
    "MODADVERT_CATALOGIA_FLAGS_DIFF",
    "MODADVERT_CHECK_NEW_CIRCUIT",
    "MODADVERT_CLIENT_FACTORS",
    "MODADVERT_EVIL_MISPRINTS_STATISTICS",
    "MODADVERT_GET_BANNER_MANUAL_RESULTS",
    "MODADVERT_MONITORING",
    "MODADVERT_TOLOKA_AUTOBUDGET",
    "MODADVERT_UPDATE_BANNERS_COUNT",
    "MODEL_DATE_MONITOR",
    "MODIFY_RESOURCE",
    "MULTIMEDIA_RTMR_MONITOR",
    "NANNY_DELETE_SNAPSHOT_RESOURCE",
    "NEWS_AUTORELEASABLE_TASK",
    "ONLINE_LEARNING_DUMP_TXT_PREPARE",
    "ORG1_SERP_DOWNLOADER",
    "ORG_EVENTS_UPDATER",
    "PANTERA_FACTORS",
    "PARSE_SANDBOX_API_LOGS",
    "PARSE_SANDBOX_XMLRPC_LOGS",
    "PARSIBILITY_TESTER_FOR_SESSIONS_CREATE",
    "PATCH_NEWS_APP_HOST_GRAPHS",
    "PATCH_PLAN",
    "PATCH_QUERIES",
    "PREPARE_ABT_REGRESSION_SAMPLE",
    "PREPARE_ADDRS_BASE_DATA_FOR_TESTENV",
    "PREPARE_QUERIES_AND_BINARIES_FOR_REARRANGE_DYNAMIC_TESTS",
    "PRIEMKA_BASESEARCH_DATABASE_JUPITER",
    "PRIEMKA_CREATE_SESSIONS",
    "PROCESS_SIMILARS_PREPARAT",
    "PROGRAM_CRASHER",
    "PROXY_WIZARD_WEATHER_TRANSLATIONS",
    "PUBLISH_NEWS_ARCHIVE",
    "PUBLISH_NEWS_ARCHIVE_WRAPPER",
    "PUBLISH_SIMILAR_ORGS",
    "PUSH_BUILD",
    "RELEASE_ADDRS_SHARD_AND_ADDITIONAL_RESOURCES",
    "RELEASE_ANTIROBOT_FORMULAS",
    "RELEASE_ANY",
    "RELEASE_ARCNEWS_SHARDMAP",
    "RELEASE_CYCOUNTER_FILES",
    "RELEASE_DELAYED_VIEW_TRIES",
    "RELEASE_DOMAIN_HAS_METRIKA_TRIE",
    "RELEASE_G2LD_LIST",
    "RELEASE_ISS_PRESTABLE",
    "RELEASE_MEDIA_SHARDMAPS",
    "RELEASE_NANNY_SHARDMAP",
    "RELEASE_PARENT_TASK",
    "RELEASE_PERSONAL_BUNDLE",
    "RELEASE_POETRYLOVER_SHARDMAP",
    "RELEASE_REPORT_DATA_RUNTIME_BUNDLE",
    "RELEASE_REPORT_DATA_RUNTIME_BUNDLE2",
    "RELEASE_SHARD_INSTALL_SCRIPTS",
    "RELEASE_SMISEARCH_SHARD",
    "RELEASE_SPORT_PROXY_DATA",
    "RELEASE_VIDEO_VEGAS_CONFIG",
    "RELEASE_YT_LOCAL",
    "RELEV_FML_UNUSED",
    "REMOTE_COPY_RESOURCE",
    "REMOVE_REARRANGE_DYNAMIC_FORMULAS",
    "REPORT_CHECK_AUTOTEST_STATUS",
    "REPORT_DATA_RUNTIME_ITEM",
    "REPORT_DATA_RUNTIME_TAGS",
    "REPORT_RULE_TEST_FULL",
    "RESHARE_SHARDS",
    "RESHARE_SHARDS_ON_HOST",
    "RESOURCES_TO_YT",
    "RESOURCE_LOADER_FROM_NANNY",
    "RESTORE_INDEXFRQ",
    "RESTORE_MONGO",
    "RUN_ABT_METRICS",
    "RUN_CUBE_STATUS",
    "RUN_IDX_OPS_URLNORM_TEST",
    "RUN_JUGGLER_MARKET_PLAYBOOK",
    "RUN_JUGGLER_SEARCH_PLAYBOOK",
    "RUN_JUGGLER_VIDEO_QUALITY_PLAYBOOK",
    "RUN_MARKET_FW_MOVER",
    "RUN_NANNY_DASHBOARD_RECIPE",
    "RUN_NANNY_RECIPE",
    "RUN_NEWS_STORAGE_TRIMMER",
    "RUN_NIRVANA_BASED_INTEGRATION_TEST",
    "RUN_NIRVANA_ONLINE_GRAPH_LAUNCHER",
    "RUN_NIRVANA_ONLINE_LEARNING",
    "RUN_PRINTWZRD",
    "RUN_REFRESH_ACCEPTANCE",
    "RUN_REM_JOBPACKET",
    "RUN_SCRIPT",
    "RUN_SOK",
    "RUN_SPYNET_YTRANK",
    "RUN_VIDEO_QUOTES_TASK",
    "RUN_WALLE_SCRIPTS",
    "RUN_YABS_PYTHON_SCRIPT",
    "RUN_YQL",
    "SAAS_ALERTS_MANAGE",
    "SAAS_GARDEN_GRAPH_CONVERT",
    "SAAS_GARDEN_HELPER",
    "SAAS_KV_UPLOADER",
    "SAAS_ROADS_GRAPH_MATCHER",
    "SAAS_ROADS_GRAPH_TRAFFIC_HISTORY_BUILDER",
    "SAMOGON_TEST_PACKAGE",
    "SANDBOX_ACCEPTANCE_SDK1",
    "SANDBOX_MONGO_DEFRAG",
    "SANDBOX_STATISTICS",
    "SANITIZER_COMPARE_FUNC_TEST_RESULTS",
    "SCRIPT_BUILD_TASK",
    "SEARCH_CREATE_SHARDMAP",
    "SERP_COLLECTOR",
    "SERVICE_SANDBOX_OUTDATED_TASKS",
    "SHOOT_LOG_PREPARE",
    "SIMILAR_ORGS_DO_ALL",
    "STABILIZE_NEWS_APACHE_BUNDLE",
    "STABILIZE_NEWS_APP_HOST_BUNDLE",
    "STABILIZE_NEWS_APP_HOST_PRODUCTION_INSTANCECTL_CONF",
    "STABILIZE_NEWS_APP_HOST_SRC_SETUP_BUNDLE",
    "STABILIZE_NEWS_APP_HOST_SRC_SETUP_PRODUCTION_LOOP_CONF",
    "STATBOX_DEBUILDER",
    "STATINFRA_TEST_TASK",
    "SUB_TASKS_RUNNER",
    "SUP_BACKUP",
    "SUP_BACKUP_COMBINE",
    "SUP_REGISTRATION_LOAD",
    "SUP_REMOVE_OUTDATED",
    "SUP_UPDATE_APP_TAGS",
    "SUP_UPDATE_GEO_TAGS",
    "SUSPICIOUS_COMMITS",
    "SWITCH_VIDEO_DATABASE",
    "SYNC_ICEBERG_TASK",
    "TEST_ALEMATE",
    "TEST_BASESEARCH_CGI_PARAMS",
    "TEST_BASESEARCH_EXPERIMENT",
    "TEST_BASESEARCH_PAGE_FAULTS",
    "TEST_BASESEARCH_PERFORMANCE",
    "TEST_CVDUP_ON_SYNTHETIC",
    "TEST_DYNAMIC_MODELS_ARCHIVE",
    "TEST_DYNAMIC_MODELS_FEATURES",
    "TEST_ENVIRONMENT_BACKUP_TEST",
    "TEST_ENVIRONMENT_DUMMY",
    "TEST_EXECUTE_WITH_PGO_TASK",
    "TEST_FROM_EXTERNAL_SCRIPT",
    "TEST_INSTANCE_CTL",
    "TEST_LEMUR_VINTAGE",
    "TEST_MAPS_SEARCH_BUSINESS_PERFORMANCE",
    "TEST_MATRIXNET",
    "TEST_NEH",
    "TEST_NEWSD_PERFORMANCE",
    "TEST_NEWS_BUILDER",
    "TEST_NOAPACHEUPPER",
    "TEST_OCR_RUNNER",
    "TEST_PEOPLESEARCH_TOOL",
    "TEST_PERFORMANCE_BEST",
    "TEST_RALIB_PERFOMANCE",
    "TEST_REPORT_CONTEXT",
    "TEST_REPORT_LOGS",
    "TEST_REPORT_UNIT_UPDATE_RES",
    "TEST_ROUTERD_PERFORMANCE",
    "TEST_RTYSERVER_MANAGE",
    "TEST_RTYSERVER_PROXY",
    "TEST_RTYSERVER_UNIT",
    "TEST_RTYSERVER_UT",
    "TEST_SAAS_FML_UT",
    "TEST_SYNC_RESOURCE",
    "TEST_TASK",
    "TEST_TASK_21",
    "TEST_TEST_ENVIRONMENT",
    "TEST_THUMB_DAEMON",
    "TEST_THUMB_DAEMON_FUNC",
    "TEST_TRAIN_MATRIXNET_ON_CPU",
    "TEST_TRANSPARENT_HUGEPAGES",
    "TEST_UT_MEMCHECK",
    "THUMBS_BAN_BUILD_QUERYSEARCH",
    "TMP_ABT_METRICS_DEV",
    "TOOLS_CAAS_CONFIGURE",
    "TOOLS_CAAS_GENCONFIG",
    "TOP_MUSICIANS_FROM_PPB_POSTLOG",
    "TOUCH_TEST_PUMPKIN_SERP",
    "TV_SNIP_DATA_BUILD",
    "TV_SNIP_DATA_BUILD_GMAKE",
    "TV_SNIP_DATA_RELEASE",
    "UNIT_TEST",
    "UNPACK_RESOURCES_ARCHIVE",
    "UPDATE_ABUSE_REMODERATION_TABLE",
    "UPDATE_BUSINESS_SNIPPET_PHOTOS",
    "UPDATE_CONVEYOR_DASHBOARD_CONVEYORS_CACHE",
    "UPDATE_CONVEYOR_DASHBOARD_FORMULA_COMMIT_TREES_CACHE",
    "UPDATE_CONVEYOR_DASHBOARD_NIRVANA_API_CACHE",
    "UPDATE_CONVEYOR_DASHBOARD_OUTPUT_FORMULA_CACHE",
    "UPDATE_EXTERNAL_DATA",
    "UPDATE_HOSTS_DATA",
    "UPDATE_MAPS_GEO_HOST_FACTORS",
    "UPDATE_MAPS_GEO_SUGGEST_SAAS_INDEX",
    "UPDATE_NOAPACHEUPPER_YASM_PANELS",
    "UPLOAD_GEOSEARCH_ENTITIES",
    "UPLOAD_IMAGES_DATABASE",
    "UPLOAD_VIDEO_DATABASE",
    "USER_SESSIONS_PROCESSES_RECALC",
    "USER_SESSIONS_PROCESSES_TEST",
    "VALIDATE_MATRIXNET_MODELS",
    "VERTICALS_TEST_WRAPPER_2_SIMPLIFIED",
    "VIDEO_BUILD_AND_TEST_DYNAMIC_MODELS",
    "VIDEO_BUILD_CONNECTORS_RAW",
    "VIDEO_BUILD_INDEX_INFO",
    "VIDEO_BUILD_RKN_BAN",
    "VIDEO_BUILD_UNBAN_RAW",
    "VIDEO_BUILD_VEGAS_CONFIG",
    "VIDEO_COMPARE_DYNAMIC_MODELS",
    "VIDEO_INDEX_VALIDATION",
    "VIDEO_LOAD_BASESEARCH_TESTENV_DATA",
    "VIDEO_LOAD_MIDDLESEARCH_RESOURCES",
    "VIDEO_METRICS_INDEX_STABILITY",
    "VIDEO_METRICS_SHARD_SIZE",
    "VIDEO_RECOMMENDER_BUILD_SESSIONS_TOOL_DATA",
    "VIDEO_ROBOT_SELRANK_METRICS",
    "VIDEO_SET_MEDIA_ZK_FLAGS",
    "VIDEO_STORE_THUMBS_SHARDMAP",
    "VIDEO_TEST_BASESEARCH_PERFORMANCE",
    "VIDEO_TEST_DYNAMIC_MODELS_ARCHIVE",
    "VIDEO_TEST_PUMPKIN_SERP",
    "VPS_BUILD",
    "VPS_CUSTOM_CONFIG",
    "VPS_GENERATE_AMMO",
    "WAIT_TRUNK_DB",
    "WALLE_STATS",
    "WATCH_SERP_ITS_FLAGS",
    "WEB_QUERIES_FILTER",
    "WEB_RESULT_SCHEMA_CHECKER_BASE",
    "WIZARD_BETA",
    "WIZARD_LOGS",
    "WIZARD_PROXY_COMPARE_REQRES",
    "WIZARD_PROXY_GENERATE_REQRES",
    "WIZARD_QUERIES_PROD",
    "WIZARD_REDUCER",
    "WIZARD_RESPONSES",
    "WIZARD_RUNTIME_BUILD",
    "WIZARD_TEST_PERFORMANCE",
    "YABS_BSCOUNT_PERFORMANCE",
    "YABS_COLLECTORS_MON",
    "YABS_DEBUILDER",
    "YABS_DEBUILDER_DEV",
    "YABS_DELAY_SBYT_STAT",
    "YABS_DOWNLOAD_AB_INFO",
    "YABS_DOWNLOAD_HOST_OPTIONS",
    "YABS_DOWNLOAD_SHARDED_TIMESTATS4",
    "YABS_DOWNLOAD_TIMESTATS4",
    "YABS_LBYT_JOBS_MONITORING",
    "YABS_REPORT_FOR_SSP",
    "YABS_SBYT_LINKER",
    "YABS_SBYT_RUNNER",
    "YABS_SERVER_ARCHIVE_CS_INPUT",
    "YABS_SERVER_DB_SIZE_AGGREGATE",
    "YABS_SERVER_GET_SQL_ARCHIVE",
    "YABS_SERVER_ONE_SHOT",
    "YABS_SERVER_RUN_CAPACITY_ESTIMATE",
    "YABS_SERVER_STAT_PERFORMANCE_AVG",
    "YABS_SERVER_STAT_PERFORMANCE_TOTAL_AVG_WEIGHTED",
    "YABS_WHITE_USERS_TRANSFER",
    "YABS_YQL_CRON_TASKS",
    "YABS_YQL_CRON_TNS_TASK",
    "YABS_YTSTAT_TABLE_CLONER",
    "YABS_YT_BACKUP_RESTORE_19",
    "YABS_YT_STAT_COLLECTOR",
    "YABS_YT_STAT_COORDINATOR",
    "YADI_FEED_UPDATE",
    "YANE_PARSE_YANSWER_FACT_LOG",
    "YANE_PARSE_YANSWER_LOG",
    "YA_PACKAGE_ACCEPTANCE",
    "YT_BACKUP",
    "YT_BUILD_OTSDB_COLLECTOR_BUNDLE",
    "YT_BUILD_SOLOMON_BRIDGE_BUNDLE",
    "YT_BUILD_SPINACH_BUNDLE",
    "YT_BUILD_SPINACH_SENSORS_BUNDLE",
    "YT_CHEF_RESOURCES",
    "YT_UPLOAD_RESOURCES",
}
