import datetime
from sandbox import sdk2
from sandbox.sdk2.parameters import Group, List, String, YavSecret


class FetchGithubStats(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        with Group('GitHub parameters'):
            gh_secret = YavSecret(
                'GitHub token. Must have `public_repo` permission',
                default='sec-01f8dq26hj10am6h37tykrvmvj#ver-01f8dq26j08xqgdsm3pw6rwq6f#github_token',
                required=True
            )
            repos = List(
                'A list of GitHub repositories to proces',
                default=['toloka/toloka-kit', 'toloka/crowd-kit'],
                required=True
            )

        with Group('YT parameters'):
            yt_cluster = String('YT cluster', default='hahn', required=True)
            yt_secret = YavSecret(
                'YT token',
                default='sec-01f8gc0kr52hy1tgj974x1ddeb#ver-01f8gc0krd1bygvdh1atd6zebv#hahn-yt-token',
                required=True
            )

            followers_table = String(
                'Dynamic table with GitHub followers information to update',
                default='//home/toloka-analytics/prod/toloka_kit/followers',
                required=True,
            )
            referrers_table = String(
                'Dynamic table with GitHub referrers information to update',
                default='//home/toloka-analytics/prod/toloka_kit/referrers',
                required=True,
            )
            traffic_table = String(
                'Dynamic table with GitHub traffic information to update',
                default='//home/toloka-analytics/prod/toloka_kit/traffic',
            )
            top_paths_table = String(
                'Dynamic table with GitHub top-10 popular paths information to update',
                default='//home/toloka-analytics/prod/toloka_kit/top_paths',
                required=True,
            )

    def get_views_update_rows(self, repo):
        for view in repo.get_views_traffic()['views']:
            yield {
                'repo': repo.full_name.lower(),
                'date': view.timestamp.date().isoformat(),
                'total_views': view.count,
                'unique_views': view.uniques,
            }

    def get_clones_update_rows(self, repo):
        for clone in repo.get_clones_traffic()['clones']:
            yield {
                'repo': repo.full_name.lower(),
                'date': clone.timestamp.date().isoformat(),
                'total_clones': clone.count,
                'unique_clones': clone.uniques,
            }

    def get_referrers_rows(self, repo):
        for referrer in repo.get_top_referrers():
            yield {
                'repo': repo.full_name.lower(),
                'datetime': datetime.datetime.utcnow().isoformat(),
                'referrer': referrer.referrer,
                'total_views': referrer.count,
                'unique_views': referrer.uniques,
            }

    def get_followers_row(self, repo):
        return {
            'repo': repo.full_name.lower(),
            'datetime': datetime.datetime.utcnow().isoformat(),
            'stars_count': repo.stargazers_count,
            'subscribers_count': repo.subscribers_count,
        }

    def get_top_paths_row(self, repo):
        for path in repo.get_top_paths():
            yield {
                'repo': repo.full_name.lower(),
                'date': datetime.datetime.utcnow().isoformat(),
                'path': path.path,
                'count': path.count,
                'uniques': path.uniques,
            }

    def on_execute(self):

        from github import Github
        from yt.wrapper import YtClient

        gh_token = self.Parameters.gh_secret.data()[self.Parameters.gh_secret.default_key]
        yt_token = self.Parameters.yt_secret.data()[self.Parameters.yt_secret.default_key]

        gh_client = Github(gh_token)
        yt_client = YtClient(proxy=self.Parameters.yt_cluster, token=yt_token)

        for repo_path in self.Parameters.repos:
            repo = gh_client.get_repo(repo_path)

            yt_client.insert_rows(self.Parameters.traffic_table, self.get_views_update_rows(repo), update=True)
            yt_client.insert_rows(self.Parameters.traffic_table, self.get_clones_update_rows(repo), update=True)
            yt_client.insert_rows(self.Parameters.referrers_table, self.get_referrers_rows(repo))
            yt_client.insert_rows(self.Parameters.followers_table, [self.get_followers_row(repo)])
            yt_client.insert_rows(self.Parameters.top_paths_table, self.get_top_paths_row(repo))
